const express = require('express');
const axios = require('axios');
const https = require('https');
const AES256 = require('aes-everywhere');
const mysql = require('mysql2/promise');

const router = express.Router();

// Database pool - get from main server or create if needed
let dbPool = null;
function setDatabasePool(pool) {
  dbPool = pool;
}

// Expect these in environment variables; do NOT hardcode
const BRIDGECARD_BASE = process.env.BRIDGECARD_BASE_URL || 'https://issuecards.api.bridgecard.co/v1/issuing/sandbox';
const BRIDGECARD_SECRET = process.env.BRIDGECARD_SECRET_KEY || '';
const BRIDGECARD_ACCESS_TOKEN = process.env.BRIDGECARD_ACCESS_TOKEN || '';
const BRIDGECARD_DEFAULT_PRODUCT_ID = process.env.BRIDGECARD_PRODUCT_ID || '';

// Fix SSL/TLS handshake issues globally for axios
// Temporarily disable SSL verification for debugging (REMOVE IN PRODUCTION)
process.env.NODE_TLS_REJECT_UNAUTHORIZED = '0';

const agent = new https.Agent({ 
  secureProtocol: 'TLSv1_2_method',
  rejectUnauthorized: false, // Temporarily disabled for debugging
});
axios.defaults.httpsAgent = agent;

// App-level authentication middleware
router.use((req, res, next) => {
  const key = req.headers['x-app-key'] || req.headers['authorization']?.replace('Bearer ', '');
  const appKey = process.env.APP_API_KEY || 'test-key'; // Default key for testing
  if (!key || (appKey && key !== appKey)) {
    console.log('Auth check:', { key, appKey, headers: req.headers });
    return res.status(401).json({ error: 'Unauthorized', key, appKey });
  }
  next();
});

// Create axios instance with defaults
const bridgecardAPI = axios.create({
  baseURL: BRIDGECARD_BASE,
  timeout: 45000,
  headers: {
    'token': `Bearer ${BRIDGECARD_ACCESS_TOKEN || BRIDGECARD_SECRET}`,
    'Content-Type': 'application/json',
  },
  // Use the global TLS 1.2 agent
  httpsAgent: agent,
});

// Input validation helper
function validateRequiredFields(req, fields) {
  const missing = fields.filter(field => !req.body[field]);
  if (missing.length > 0) {
    return { error: `Missing required fields: ${missing.join(', ')}` };
  }
  return null;
}

// Create cardholder - REAL BRIDGECARD API CALL
router.post('/cardholder', async (req, res) => {
  // Declare variables outside try block so they're accessible in catch
  let endpoint = '/cardholder/register_cardholder'; // Default endpoint
  let payload = null;
  
  try {
    // Validate required fields
    const validation = validateRequiredFields(req, ['first_name', 'last_name', 'email']);
    if (validation) {
      return res.status(400).json(validation);
    }

    // Build payload according to Bridgecard API requirements
    // Bridgecard uses 'email' (not 'email_address') and requires top-level 'country' field
    const bridgecardPayload = {
      first_name: (req.body.first_name || '').trim(),
      last_name: (req.body.last_name || '').trim(),
      email: (req.body.email || req.body.email_address || '').trim(), // Bridgecard uses 'email' not 'email_address'
      phone: (req.body.phone || '').trim(),
      country: (req.body.country || req.body.address?.country || 'US').trim(), // Top-level country field is REQUIRED
      ...(req.body.date_of_birth ? { date_of_birth: req.body.date_of_birth } : {}),
    };
    
    // Validate required fields
    if (!bridgecardPayload.first_name || !bridgecardPayload.last_name || !bridgecardPayload.email || !bridgecardPayload.phone || !bridgecardPayload.country) {
      return res.status(400).json({
        success: false,
        error: 'Missing required fields',
        message: 'first_name, last_name, email, phone, and country are required',
        missing: {
          first_name: !bridgecardPayload.first_name,
          last_name: !bridgecardPayload.last_name,
          email: !bridgecardPayload.email,
          phone: !bridgecardPayload.phone,
          country: !bridgecardPayload.country
        }
      });
    }
    
    // Handle address - Bridgecard expects specific structure
    // Address is required for cardholder registration
    // Bridgecard uses 'street' not 'address' inside the address object
    if (req.body.address) {
      const addressData = {
        street: (req.body.address.street || req.body.address.address || '').trim(), // Bridgecard uses 'street' not 'address'
        city: (req.body.address.city || '').trim(),
        state: (req.body.address.state || '').trim(), // Remove trailing spaces
        country: (req.body.address.country || bridgecardPayload.country || 'US').trim(),
        postal_code: (req.body.address.postalCode || req.body.address.postal_code || '').trim(),
      };
      
      // Add house_no if provided
      if (req.body.address.house_no) {
        addressData.house_no = String(req.body.address.house_no).trim();
      }
      
      // Validate required address fields
      const requiredAddressFields = ['street', 'city', 'state', 'country', 'postal_code'];
      const missingFields = requiredAddressFields.filter(field => !addressData[field] || addressData[field] === '');
      
      if (missingFields.length > 0) {
        return res.status(400).json({
          success: false,
          error: 'Incomplete address information',
          message: 'Address must include: street, city, state, country, and postal_code',
          missingFields: missingFields
        });
      }
      
      bridgecardPayload.address = addressData;
    } else {
      // Address is required - return error if not provided
      return res.status(400).json({
        success: false,
        error: 'Address is required for cardholder registration',
        message: 'Please provide a complete address including street, city, state, country, and postal_code'
      });
    }
    
    // Handle identity verification if provided
    if (req.body.identity) {
      bridgecardPayload.identity = {
        id_type: req.body.identity.id_type || 'NIGERIAN_BVN_VERIFICATION',
        ...(req.body.identity.bvn ? { bvn: req.body.identity.bvn } : {}),
        ...(req.body.identity.selfie_image ? { selfie_image: req.body.identity.selfie_image } : {}),
      };
    }
    
    // Add meta_data if provided
    if (req.body.meta_data) {
      bridgecardPayload.meta_data = req.body.meta_data;
    }
    
    payload = bridgecardPayload;

    // Use asynchronous endpoint (recommended) or synchronous if specified
    const useSynchronous = req.body.synchronous === true;
    endpoint = useSynchronous 
      ? '/cardholder/register_cardholder_synchronously'  // Synchronous (45 sec timeout)
      : '/cardholder/register_cardholder';  // Asynchronous (recommended, webhook-based)

    console.log('🔄 Creating cardholder with Bridgecard API:');
    console.log('   Payload:', JSON.stringify(payload, null, 2));
    console.log('   Endpoint:', endpoint);
    console.log('   Method:', useSynchronous ? 'SYNCHRONOUS (45 sec timeout)' : 'ASYNCHRONOUS (webhook-based)');
    console.log('   Timeout: 45000ms (45 seconds)');
    
    // Call REAL Bridgecard API to create cardholder
    // This can take up to 45 seconds for synchronous verification
    // CORRECT ENDPOINT: /cardholder/register_cardholder (async) or /cardholder/register_cardholder_synchronously (sync)
    let response;
    let lastError = null;
    
    // Try different header formats
    const headerFormats = [
      { 'token': `Bearer ${BRIDGECARD_ACCESS_TOKEN || BRIDGECARD_SECRET}`, 'Content-Type': 'application/json' },
      { 'token': BRIDGECARD_ACCESS_TOKEN || BRIDGECARD_SECRET, 'Content-Type': 'application/json' },
      { 'Authorization': `Bearer ${BRIDGECARD_ACCESS_TOKEN || BRIDGECARD_SECRET}`, 'Content-Type': 'application/json' },
    ];
    
    for (const headers of headerFormats) {
      try {
        response = await bridgecardAPI.post(endpoint, payload, {
          headers: headers,
          timeout: 45000
        });
        
        // If successful, break
        if (response && (response.status === 200 || response.status === 201)) {
          console.log(`✅ Success with headers: ${Object.keys(headers).join(', ')}`);
          break;
        }
      } catch (error) {
        lastError = error;
        // Try next header format
        continue;
      }
    }
    
    // If all attempts failed, throw error
    if (!response || (response.status !== 200 && response.status !== 201)) {
      throw lastError || new Error('All header format attempts failed');
    }
    
    console.log('✅ Bridgecard API response:', JSON.stringify(response.data, null, 2));
    
    // Return the response from Bridgecard
    return res.json({ 
      success: true, 
      data: response.data 
    });
  } catch (err) {
    const status = err.response?.status;
    const data = err.response?.data || { error: err.message };
    const attemptedUrl = err.config?.url || `${BRIDGECARD_BASE}${endpoint || '/cardholder/register_cardholder'}`;
    
    console.error('❌ Bridgecard create cardholder error:', { 
      status, 
      data, 
      detail: data.detail, // Log validation errors
      url: attemptedUrl,
      endpoint: endpoint,
      payload: payload, // Log the transformed payload
      originalPayload: req.body,
      message: err.message
    });
    
    // Handle 422 validation errors with detailed messages
    if (status === 422) {
      const validationErrors = data.detail || [];
      const errorMessages = Array.isArray(validationErrors) 
        ? validationErrors.map(err => {
            if (typeof err === 'object') {
              return `${err.loc?.join('.') || 'field'}: ${err.msg || err.message || 'validation error'}`;
            }
            return String(err);
          })
        : [String(validationErrors)];
      
      return res.status(422).json({
        success: false,
        error: 'Validation failed',
        message: 'Bridgecard API validation errors',
        validationErrors: errorMessages,
        details: data,
        payload: payload // Include the payload that was sent for debugging
      });
    }
    
    return res.status(status || 500).json({ 
      success: false, 
      error: 'Failed to create cardholder', 
      details: data, 
      status 
    });
  }
});

// Check if cardholder exists by email
router.get('/cardholder/:email', async (req, res) => {
  try {
    const { email } = req.params;
    
    if (!email) {
      return res.status(400).json({ 
        success: false,
        error: 'Email parameter is required' 
      });
    }

    console.log('🔍 Checking cardholder by email:', email);
    
    // Call Bridgecard API to list cardholders by email
    const response = await bridgecardAPI.get(`/cardholder/list?email=${encodeURIComponent(email)}`);
    
    console.log('📋 Bridgecard list response:', JSON.stringify(response.data, null, 2));
    
    if (response.data?.data && Array.isArray(response.data.data) && response.data.data.length > 0) {
      return res.json({ 
        success: true,
        exists: true, 
        cardholder: response.data.data[0] 
      });
    }
    
    return res.json({ 
      success: true,
      exists: false 
    });
  } catch (err) {
    const data = err.response?.data || { error: err.message };
    console.error('❌ Bridgecard check cardholder error:', data);
    return res.status(err.response?.status || 500).json({ 
      success: false,
      error: 'Failed to check cardholder', 
      details: data 
    });
  }
});

// Issue card
router.post('/issue-card', async (req, res) => {
  try {
    const validation = validateRequiredFields(req, ['cardholder_id']);
    if (validation) {
      return res.status(400).json(validation);
    }

    const payload = { ...req.body };
    
    // Set default values for USD card creation
    payload.card_type = payload.card_type || 'virtual';
    payload.card_brand = payload.card_brand || 'Mastercard';
    payload.card_currency = payload.card_currency || 'USD';
    payload.card_limit = payload.card_limit || '500000'; // $5,000 limit
    payload.funding_amount = payload.funding_amount || '300'; // Minimum $3 for $5k limit
    
    // Encrypt PIN if provided
    if (payload.pin && payload.pin.length === 4) {
      payload.pin = AES256.encrypt(payload.pin, BRIDGECARD_SECRET);
    }
    
    const response = await bridgecardAPI.post('/cards/create_card', payload);
    return res.json({ success: true, data: response.data });
  } catch (err) {
    const data = err.response?.data || { error: err.message };
    console.error('Bridgecard issue card error:', data);
    return res.status(500).json({ success: false, error: 'Failed to issue card', details: data });
  }
});

// Get card details (using relay URL for decryption)
router.post('/get-card-details', async (req, res) => {
  try {
    const validation = validateRequiredFields(req, ['card_id']);
    if (validation) {
      return res.status(400).json(validation);
    }

    const { card_id } = req.body;
    
    // Use relay URL for decryption as per Bridgecard docs
    // Relay URLs decrypt card details automatically
    const isProduction = process.env.NODE_ENV === 'production';
    const relayBaseURL = isProduction 
      ? 'https://issuecards-api-bridgecard-co.relay.evervault.com/v1/issuing'
      : 'https://issuecards-api-bridgecard-co.relay.evervault.com/v1/issuing/sandbox';
    
    const relayUrl = `${relayBaseURL}/cards/get_card_details?card_id=${card_id}`;
    
    console.log('🔐 Getting card details via relay URL (decrypted):', relayUrl);
    
    // Call relay endpoint which automatically decrypts card details
    const response = await axios.get(relayUrl, {
      headers: {
        'token': `Bearer ${BRIDGECARD_ACCESS_TOKEN || BRIDGECARD_SECRET}`,
        'Content-Type': 'application/json'
      },
      timeout: 30000,
      httpsAgent: agent
    });
    
    console.log('✅ Card details response (decrypted):', JSON.stringify(response.data, null, 2));
    
    // Return decrypted card details (be careful - contains sensitive info)
    // In production, you might want to mask some fields before returning
    return res.json({ 
      success: true, 
      data: response.data 
    });
  } catch (err) {
    const data = err.response?.data || { error: err.message };
    console.error('❌ Bridgecard get card details error:', data);
    return res.status(err.response?.status || 500).json({ 
      success: false, 
      error: 'Failed to get card details', 
      details: data 
    });
  }
});

// List cards for a cardholder
router.get('/cardholders/:cardholderId/cards', async (req, res) => {
  try {
    const { cardholderId } = req.params;
    const response = await bridgecardAPI.get(`/cards/get_all_cardholder_cards?cardholder_id=${cardholderId}`);
    return res.json({ success: true, data: response.data });
  } catch (err) {
    const data = err.response?.data || { error: err.message };
    console.error('Bridgecard list cards error:', data);
    return res.status(500).json({ success: false, error: 'Failed to list cards', details: data });
  }
});

// Fund a card (loads from your Bridgecard issuing wallet balance)
router.post('/cards/fund', async (req, res) => {
  try {
    const validation = validateRequiredFields(req, ['card_id', 'amount']);
    if (validation) {
      return res.status(400).json(validation);
    }

    const { card_id, amount, currency = 'USD' } = req.body;
    const payload = {
      card_id,
      amount: amount.toString(),
      currency,
      transaction_reference: `fund_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
    };
    
    console.log('💰 Funding card:', { card_id, amount, currency });
    const response = await bridgecardAPI.patch('/cards/fund_card_asynchronously', payload);
    
    console.log('✅ Fund card response:', JSON.stringify(response.data, null, 2));
    return res.json({ success: true, data: response.data });
  } catch (err) {
    const data = err.response?.data || { error: err.message };
    console.error('❌ Bridgecard fund card error:', data);
    return res.status(err.response?.status || 500).json({ 
      success: false, 
      error: 'Failed to fund card', 
      details: data 
    });
  }
});

// Get card transactions
router.get('/cards/:card_id/transactions', async (req, res) => {
  try {
    const { card_id } = req.params;
    const { page = 1, limit = 50 } = req.query;
    
    if (!card_id) {
      return res.status(400).json({ 
        success: false,
        error: 'card_id parameter is required' 
      });
    }

    console.log('📋 Getting transactions for card:', card_id);
    
    const response = await bridgecardAPI.get(
      `/cards/get_card_transactions?card_id=${card_id}&page=${page}&limit=${limit}`
    );
    
    console.log('✅ Card transactions response:', JSON.stringify(response.data, null, 2));
    return res.json({ 
      success: true, 
      data: response.data 
    });
  } catch (err) {
    const data = err.response?.data || { error: err.message };
    console.error('❌ Bridgecard get transactions error:', data);
    return res.status(err.response?.status || 500).json({ 
      success: false, 
      error: 'Failed to get card transactions', 
      details: data 
    });
  }
});

// Freeze card
router.patch('/cards/freeze', async (req, res) => {
  try {
    const validation = validateRequiredFields(req, ['card_id']);
    if (validation) {
      return res.status(400).json(validation);
    }

    const { card_id } = req.body;
    
    console.log('🔒 Freezing card:', card_id);
    const response = await bridgecardAPI.patch('/cards/freeze_card', { card_id });
    
    console.log('✅ Freeze card response:', JSON.stringify(response.data, null, 2));
    return res.json({ 
      success: true, 
      data: response.data 
    });
  } catch (err) {
    const data = err.response?.data || { error: err.message };
    console.error('❌ Bridgecard freeze card error:', data);
    return res.status(err.response?.status || 500).json({ 
      success: false, 
      error: 'Failed to freeze card', 
      details: data 
    });
  }
});

// Unfreeze card
router.patch('/cards/unfreeze', async (req, res) => {
  try {
    const validation = validateRequiredFields(req, ['card_id']);
    if (validation) {
      return res.status(400).json(validation);
    }

    const { card_id } = req.body;
    
    console.log('🔓 Unfreezing card:', card_id);
    const response = await bridgecardAPI.patch('/cards/unfreeze_card', { card_id });
    
    console.log('✅ Unfreeze card response:', JSON.stringify(response.data, null, 2));
    return res.json({ 
      success: true, 
      data: response.data 
    });
  } catch (err) {
    const data = err.response?.data || { error: err.message };
    console.error('❌ Bridgecard unfreeze card error:', data);
    return res.status(err.response?.status || 500).json({ 
      success: false, 
      error: 'Failed to unfreeze card', 
      details: data 
    });
  }
});

// Bridgecard webhook handler
router.post('/webhook', async (req, res) => {
  try {
    console.log('═══════════════════════════════════════════════════════════');
    console.log('📨 Bridgecard Webhook Event Received');
    console.log('═══════════════════════════════════════════════════════════');
    console.log('Event Type:', req.body.event_type || req.body.type || 'unknown');
    console.log('Event Data:', JSON.stringify(req.body, null, 2));
    console.log('═══════════════════════════════════════════════════════════');
    
    // TODO: Verify webhook signature if Bridgecard provides one
    // const signature = req.headers['x-bridgecard-signature'];
    // if (signature) {
    //   const isValid = verifyWebhookSignature(req.body, signature);
    //   if (!isValid) {
    //     return res.status(401).json({ error: 'Invalid webhook signature' });
    //   }
    // }
    
    const eventType = req.body.event_type || req.body.type || 'unknown';
    const eventData = req.body.data || req.body;
    
    // Handle different webhook event types
    switch (eventType) {
      case 'cardholder.verified':
      case 'cardholder_verified':
        console.log('✅ Cardholder verified event:', eventData);
        // Update cardholder verification status in database
        if (dbPool && eventData.cardholder_id) {
          try {
            await dbPool.execute(
              'UPDATE cardholders SET is_bridgecard_verified = TRUE WHERE cardholder_id = ?',
              [eventData.cardholder_id]
            );
            console.log(`✅ Updated cardholder ${eventData.cardholder_id} verification status to TRUE`);
          } catch (dbError) {
            console.error('❌ Failed to update cardholder verification in database:', dbError);
          }
        }
        break;
        
      case 'card.created':
      case 'card_created':
        console.log('✅ Card created event:', eventData);
        // TODO: Store card in database or notify user
        break;
        
      case 'card.funded':
      case 'card_funded':
        console.log('✅ Card funded event:', eventData);
        // TODO: Update card balance in database
        break;
        
      case 'transaction.completed':
      case 'transaction_completed':
        console.log('✅ Transaction completed event:', eventData);
        // TODO: Update transaction status in database
        break;
        
      case 'transaction.failed':
      case 'transaction_failed':
        console.log('❌ Transaction failed event:', eventData);
        // TODO: Update transaction status in database
        break;
        
      case 'card.frozen':
      case 'card_frozen':
        console.log('🔒 Card frozen event:', eventData);
        // TODO: Update card status in database
        break;
        
      case 'card.unfrozen':
      case 'card_unfrozen':
        console.log('🔓 Card unfrozen event:', eventData);
        // TODO: Update card status in database
        break;
        
      default:
        console.log('⚠️ Unknown webhook event type:', eventType);
    }
    
    // Always return 200 to acknowledge receipt
    res.sendStatus(200);
  } catch (err) {
    console.error('❌ Webhook processing error:', err);
    // Still return 200 to prevent Bridgecard from retrying
    res.sendStatus(200);
  }
});

// Export router and setDatabasePool function
module.exports = router;
module.exports.setDatabasePool = setDatabasePool;