// Live Crypto Price Service
// Fetches real-time crypto prices from external APIs

const axios = require('axios');
const mysql = require('mysql2/promise');

class CryptoPriceService {
  constructor() {
    this.isRunning = false;
    this.updateInterval = null;
    this.dbConfig = {
      host: '127.0.0.1',
      user: 'root',
      password: '',
      database: 'Globana',
      waitForConnections: true,
      connectionLimit: 10,
      queueLimit: 0
    };
  }

  // Create database connection
  async getConnection() {
    return await mysql.createConnection(this.dbConfig);
  }

  // Fetch prices from CoinGecko API
  async fetchCoinGeckoPrices() {
    try {
      console.log('📈 Fetching prices from CoinGecko...');
      
      const response = await axios.get('https://api.coingecko.com/api/v3/coins/markets', {
        params: {
          vs_currency: 'usd',
          ids: 'bitcoin,ethereum,usd-coin,tether,binancecoin,cardano,solana,polygon,avalanche-2,polkadot',
          order: 'market_cap_desc',
          per_page: 10,
          page: 1,
          sparkline: false
        },
        timeout: 10000
      });

      const prices = response.data.map(coin => ({
        symbol: this.mapCoinGeckoSymbol(coin.id),
        name: coin.name,
        price_usd: coin.current_price,
        change_24h: coin.price_change_percentage_24h || 0,
        market_cap: coin.market_cap || 0,
        volume_24h: coin.total_volume || 0,
        last_updated: new Date()
      }));

      console.log('✅ CoinGecko prices fetched:', prices.length, 'coins');
      return prices;
    } catch (error) {
      console.error('❌ CoinGecko API error:', error.message);
      return null;
    }
  }

  // Fetch prices from CoinCap API (backup)
  async fetchCoinCapPrices() {
    try {
      console.log('📈 Fetching prices from CoinCap...');
      
      const response = await axios.get('https://api.coincap.io/v2/assets', {
        params: {
          limit: 10
        },
        timeout: 10000
      });

      const prices = response.data.data.map(coin => ({
        symbol: coin.symbol,
        name: coin.name,
        price_usd: parseFloat(coin.priceUsd),
        change_24h: parseFloat(coin.changePercent24Hr) || 0,
        market_cap: parseFloat(coin.marketCapUsd) || 0,
        volume_24h: parseFloat(coin.volumeUsd24Hr) || 0,
        last_updated: new Date()
      }));

      console.log('✅ CoinCap prices fetched:', prices.length, 'coins');
      return prices;
    } catch (error) {
      console.error('❌ CoinCap API error:', error.message);
      return null;
    }
  }

  // Map CoinGecko IDs to symbols
  mapCoinGeckoSymbol(id) {
    const mapping = {
      'bitcoin': 'BTC',
      'ethereum': 'ETH',
      'usd-coin': 'USDC',
      'tether': 'USDT',
      'binancecoin': 'BNB',
      'cardano': 'ADA',
      'solana': 'SOL',
      'polygon': 'MATIC',
      'avalanche-2': 'AVAX',
      'polkadot': 'DOT'
    };
    return mapping[id] || id.toUpperCase();
  }

  // Update prices in database
  async updatePricesInDatabase(prices) {
    if (!prices || prices.length === 0) {
      console.log('⚠️ No prices to update');
      return;
    }

    const connection = await this.getConnection();
    
    try {
      for (const price of prices) {
        await connection.execute(`
          INSERT INTO crypto_prices (symbol, name, price_usd, change_24h, market_cap, volume_24h, last_updated)
          VALUES (?, ?, ?, ?, ?, ?, ?)
          ON DUPLICATE KEY UPDATE
          price_usd = VALUES(price_usd),
          change_24h = VALUES(change_24h),
          market_cap = VALUES(market_cap),
          volume_24h = VALUES(volume_24h),
          last_updated = VALUES(last_updated)
        `, [
          price.symbol,
          price.name,
          price.price_usd,
          price.change_24h,
          price.market_cap,
          price.volume_24h,
          price.last_updated
        ]);
      }
      
      console.log('✅ Prices updated in database:', prices.length, 'coins');
    } catch (error) {
      console.error('❌ Database update error:', error);
    } finally {
      await connection.end();
    }
  }

  // Get current prices from database
  async getCurrentPrices() {
    const connection = await this.getConnection();
    
    try {
      const [prices] = await connection.execute(
        'SELECT * FROM crypto_prices ORDER BY market_cap DESC'
      );
      
      return prices;
    } catch (error) {
      console.error('❌ Error getting current prices:', error);
      return [];
    } finally {
      await connection.end();
    }
  }

  // Get exchange rate between two currencies
  async getExchangeRate(fromCurrency, toCurrency) {
    const connection = await this.getConnection();
    
    try {
      const [prices] = await connection.execute(
        'SELECT price_usd FROM crypto_prices WHERE symbol IN (?, ?)',
        [fromCurrency, toCurrency]
      );
      
      if (prices.length < 2) {
        return { exchangeRate: 1, fromPrice: 1, toPrice: 1, lastUpdated: new Date() };
      }
      
      const fromPrice = prices.find(p => p.symbol === fromCurrency)?.price_usd || 1;
      const toPrice = prices.find(p => p.symbol === toCurrency)?.price_usd || 1;
      const exchangeRate = fromPrice / toPrice;
      
      return {
        exchangeRate,
        fromPrice,
        toPrice,
        lastUpdated: new Date()
      };
    } catch (error) {
      console.error('❌ Error getting exchange rate:', error);
      return { exchangeRate: 1, fromPrice: 1, toPrice: 1, lastUpdated: new Date() };
    } finally {
      await connection.end();
    }
  }

  // Update prices from multiple sources
  async updatePrices() {
    console.log('🔄 Updating crypto prices...');
    
    // Try CoinGecko first
    let prices = await this.fetchCoinGeckoPrices();
    
    // If CoinGecko fails, try CoinCap
    if (!prices || prices.length === 0) {
      console.log('⚠️ CoinGecko failed, trying CoinCap...');
      prices = await this.fetchCoinCapPrices();
    }
    
    // If both fail, use fallback prices
    if (!prices || prices.length === 0) {
      console.log('⚠️ All APIs failed, using fallback prices...');
      prices = this.getFallbackPrices();
    }
    
    // Update database
    await this.updatePricesInDatabase(prices);
    
    console.log('✅ Price update completed');
  }

  // Fallback prices when APIs fail
  getFallbackPrices() {
    return [
      {
        symbol: 'BTC',
        name: 'Bitcoin',
        price_usd: 45000,
        change_24h: 2.5,
        market_cap: 850000000000,
        volume_24h: 25000000000,
        last_updated: new Date()
      },
      {
        symbol: 'ETH',
        name: 'Ethereum',
        price_usd: 3000,
        change_24h: -1.2,
        market_cap: 360000000000,
        volume_24h: 15000000000,
        last_updated: new Date()
      },
      {
        symbol: 'USDC',
        name: 'USD Coin',
        price_usd: 1.00,
        change_24h: 0.0,
        market_cap: 30000000000,
        volume_24h: 5000000000,
        last_updated: new Date()
      },
      {
        symbol: 'USDT',
        name: 'Tether',
        price_usd: 1.00,
        change_24h: 0.0,
        market_cap: 80000000000,
        volume_24h: 30000000000,
        last_updated: new Date()
      },
      {
        symbol: 'BNB',
        name: 'Binance Coin',
        price_usd: 300,
        change_24h: 1.5,
        market_cap: 45000000000,
        volume_24h: 2000000000,
        last_updated: new Date()
      }
    ];
  }

  // Start automatic price updates
  startPriceUpdates() {
    if (this.isRunning) {
      console.log('⚠️ Price updates already running');
      return;
    }

    console.log('🚀 Starting live crypto price updates...');
    this.isRunning = true;

    // Update immediately
    this.updatePrices();

    // Update every 5 minutes
    this.updateInterval = setInterval(() => {
      this.updatePrices();
    }, 5 * 60 * 1000); // 5 minutes

    console.log('✅ Live price updates started (every 5 minutes)');
  }

  // Stop automatic price updates
  stopPriceUpdates() {
    if (this.updateInterval) {
      clearInterval(this.updateInterval);
      this.updateInterval = null;
    }
    this.isRunning = false;
    console.log('⏹️ Live price updates stopped');
  }
}

module.exports = new CryptoPriceService();