const MetaMapService = require('./services/metaMapService');
const multer = require('multer');
const path = require('path');

// Configure multer for file uploads
const storage = multer.memoryStorage();
const upload = multer({ 
  storage: storage,
  limits: {
    fileSize: 10 * 1024 * 1024, // 10MB limit
  },
  fileFilter: (req, file, cb) => {
    // Only allow image files
    if (file.mimetype.startsWith('image/')) {
      cb(null, true);
    } else {
      cb(new Error('Only image files are allowed'), false);
    }
  }
});

class MetaMapController {
  // Start KYC verification process
  static async startVerification(req, res) {
    try {
      const { userId, metamapId = 'default-kyc' } = req.body;
      
      if (!userId) {
        return res.status(400).json({
          success: false,
          error: 'User ID is required'
        });
      }

      // Create verification instance via MetaMap
      const verificationResult = await MetaMapService.createVerification(metamapId, {
        user_id: userId,
        timestamp: new Date().toISOString()
      });

      if (!verificationResult.success) {
        return res.status(400).json({
          success: false,
          error: verificationResult.error
        });
      }

      // Store verification info in database
      const verificationData = {
        userId,
        identityId: verificationResult.data.id,
        metamapId,
        status: 'initiated',
        createdAt: new Date(),
        requiredInputs: verificationResult.data.requiredInputs || []
      };

      // TODO: Save to database
      // await db.verifications.create(verificationData);

      res.json({
        success: true,
        data: {
          identityId: verificationResult.data.id,
          requiredInputs: verificationResult.data.requiredInputs || [],
          metamapInfo: verificationResult.data.metamapInfo
        }
      });

    } catch (error) {
      console.error('Start verification error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to start verification process'
      });
    }
  }

  // Upload document/image to MetaMap
  static async uploadInput(req, res) {
    try {
      const { identityId, inputType } = req.body;
      const file = req.file;

      if (!identityId || !inputType || !file) {
        return res.status(400).json({
          success: false,
          error: 'Missing required parameters'
        });
      }

      // Validate file
      if (!file.mimetype.startsWith('image/')) {
        return res.status(400).json({
          success: false,
          error: 'Only image files are allowed'
        });
      }

      // Convert buffer to base64 for MetaMap API
      const base64Data = file.buffer.toString('base64');
      const dataUri = `data:${file.mimetype};base64,${base64Data}`;

      // Send input to MetaMap
      const inputData = {
        inputs: [{
          type: inputType,
          data: dataUri,
          description: `${inputType} upload for verification`
        }]
      };

      const result = await MetaMapService.sendUserInput(identityId, inputData);

      if (!result.success) {
        return res.status(400).json({
          success: false,
          error: result.error
        });
      }

      res.json({
        success: true,
        data: {
          status: 'uploaded',
          inputType,
          identityId
        }
      });

    } catch (error) {
      console.error('Upload input error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to upload input'
      });
    }
  }

  // Get verification status
  static async getVerificationStatus(req, res) {
    try {
      const { identityId } = req.params;
      const { userId } = req.query;

      if (!identityId) {
        return res.status(400).json({
          success: false,
          error: 'Identity ID is required'
        });
      }

      // Get status from MetaMap
      const statusResult = await MetaMapService.getVerificationStatus(identityId);

      if (!statusResult.success) {
        return res.status(400).json({
          success: false,
          error: statusResult.error
        });
      }

      // TODO: Update database with status
      // await db.verifications.updateStatus(identityId, statusResult.data);

      res.json({
        success: true,
        data: statusResult.data
      });

    } catch (error) {
      console.error('Get status error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to get verification status'
      });
    }
  }

  // Handle MetaMap webhooks
  static async handleWebhook(req, res) {
    try {
      const webhookData = req.body;
      const signature = req.headers['x-signature'];
      
      console.log('MetaMap webhook received:', webhookData);
      console.log('Webhook signature:', signature);

      // Verify webhook signature
      if (!this.verifyWebhookSignature(webhookData, signature)) {
        console.error('Invalid webhook signature');
        return res.status(401).json({ error: 'Invalid signature' });
      }

      // Process different webhook events
      const { event, data } = webhookData;
      
      switch (event) {
        case 'verification.completed':
          await this.handleVerificationCompleted(data);
          break;
        case 'verification.failed':
          await this.handleVerificationFailed(data);
          break;
        case 'verification.processing':
          await this.handleVerificationProcessing(data);
          break;
        case 'document.reading':
          await this.handleDocumentReading(data);
          break;
        case 'facematch.completed':
          await this.handleFacematchCompleted(data);
          break;
        case 'liveness.completed':
          await this.handleLivenessCompleted(data);
          break;
        default:
          console.log('Unhandled webhook event:', event);
      }

      res.status(200).json({ received: true });

    } catch (error) {
      console.error('Webhook error:', error);
      res.status(500).json({ error: 'Webhook processing failed' });
    }
  }

  // Verify webhook signature using HMAC SHA256
  static verifyWebhookSignature(payload, signature) {
    try {
      const crypto = require('crypto');
      const webhookSecret = process.env.METAMAP_WEBHOOK_SECRET;
      
      if (!webhookSecret) {
        console.warn('METAMAP_WEBHOOK_SECRET not set, skipping signature verification');
        return true; // Allow in development
      }

      const hmac = crypto.createHmac('sha256', webhookSecret);
      hmac.update(JSON.stringify(payload), 'utf8');
      const digest = hmac.digest('hex');
      
      return crypto.timingSafeEqual(
        Buffer.from(digest, 'hex'),
        Buffer.from(signature, 'hex')
      );
    } catch (error) {
      console.error('Signature verification error:', error);
      return false;
    }
  }

  // Handle verification completed
  static async handleVerificationCompleted(data) {
    try {
      const { identityId, verificationId, result } = data;
      
      console.log(`Verification completed for identity ${identityId}:`, result);
      
      // Extract verification results
      const verificationResult = {
        status: 'completed',
        identityId,
        verificationId,
        result: {
          documentVerification: result.documentVerification || null,
          biometricVerification: result.biometricVerification || null,
          livenessCheck: result.livenessCheck || null,
          overallStatus: result.status || 'completed'
        },
        completedAt: new Date()
      };

      // TODO: Update user KYC status in database
      // await db.users.updateKYCStatus(userId, verificationResult);
      
      console.log('KYC verification completed successfully');
      
    } catch (error) {
      console.error('Error handling verification completed:', error);
    }
  }

  // Handle verification failed
  static async handleVerificationFailed(data) {
    try {
      const { identityId, verificationId, error } = data;
      
      console.log(`Verification failed for identity ${identityId}:`, error);
      
      const verificationResult = {
        status: 'failed',
        identityId,
        verificationId,
        error: error || 'Verification failed',
        failedAt: new Date()
      };

      // TODO: Update user KYC status in database
      // await db.users.updateKYCStatus(userId, verificationResult);
      
      console.log('KYC verification failed');
      
    } catch (error) {
      console.error('Error handling verification failed:', error);
    }
  }

  // Handle verification processing
  static async handleVerificationProcessing(data) {
    try {
      const { identityId, verificationId, step } = data;
      
      console.log(`Verification processing for identity ${identityId}, step: ${step}`);
      
      // TODO: Update processing status
      // await db.users.updateKYCStatus(userId, { status: 'processing', currentStep: step });
      
    } catch (error) {
      console.error('Error handling verification processing:', error);
    }
  }

  // Handle document reading
  static async handleDocumentReading(data) {
    try {
      const { identityId, verificationId, documentData } = data;
      
      console.log(`Document reading completed for identity ${identityId}:`, documentData);
      
      // Extract document information
      const extractedData = {
        name: documentData.name,
        dateOfBirth: documentData.dateOfBirth,
        documentNumber: documentData.documentNumber,
        expiryDate: documentData.expiryDate,
        documentType: documentData.documentType,
        country: documentData.country
      };

      // TODO: Store extracted document data
      // await db.users.updateDocumentData(userId, extractedData);
      
      console.log('Document data extracted successfully');
      
    } catch (error) {
      console.error('Error handling document reading:', error);
    }
  }

  // Handle facematch completed
  static async handleFacematchCompleted(data) {
    try {
      const { identityId, verificationId, facematchResult } = data;
      
      console.log(`Facematch completed for identity ${identityId}:`, facematchResult);
      
      // TODO: Store facematch results
      // await db.users.updateFacematchResult(userId, facematchResult);
      
      console.log('Facematch completed successfully');
      
    } catch (error) {
      console.error('Error handling facematch completed:', error);
    }
  }

  // Handle liveness completed
  static async handleLivenessCompleted(data) {
    try {
      const { identityId, verificationId, livenessResult } = data;
      
      console.log(`Liveness check completed for identity ${identityId}:`, livenessResult);
      
      // TODO: Store liveness results
      // await db.users.updateLivenessResult(userId, livenessResult);
      
      console.log('Liveness check completed successfully');
      
    } catch (error) {
      console.error('Error handling liveness completed:', error);
    }
  }

  // Cancel verification
  static async cancelVerification(req, res) {
    try {
      const { identityId } = req.params;
      const { userId } = req.body;

      if (!identityId) {
        return res.status(400).json({
          success: false,
          error: 'Identity ID is required'
        });
      }

      // Cancel upload if active
      const cancelResult = MetaMapService.cancelUpload(identityId);

      // TODO: Update database status
      // await db.verifications.updateStatus(identityId, 'cancelled');

      res.json({
        success: cancelResult.success,
        message: cancelResult.message
      });

    } catch (error) {
      console.error('Cancel verification error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to cancel verification'
      });
    }
  }
}

module.exports = {
  MetaMapController,
  upload
};
