const express = require('express');
const cors = require('cors');
const mysql = require('mysql2/promise');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const multer = require('multer');
const path = require('path');
const axios = require('axios');
const stripe = require('stripe')(`sk_test_51RiDcgQN8X8zlq0XfIQNk2c8phbTQfcwiyt5VPoZj7UAGyKvpxvmgjBCV0j78sJyTyxpee0ksSUeOGqirhrWW8UB00PZlI75uS`);
const VTPassService = require('./services/vtpassService');
const { MetaMapController, upload: metamapUpload } = require('./metamapController');
require('dotenv').config();

// Initialize VTPass service
const vtpass = new VTPassService();

const app = express();
const PORT = process.env.PORT || 5000;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// Database configuration
const dbConfig = {
  host: '127.0.0.1',
  user: 'root',
  password: '',
  database: 'Globana',
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0
};

// Create database pool
const pool = mysql.createPool(dbConfig);
console.log('✅ Database pool created successfully');

// JWT Secret
const JWT_SECRET = process.env.JWT_SECRET || 'your-secret-key';

// Stripe configuration
const STRIPE_SECRET_KEY = process.env.STRIPE_SECRET_KEY || 'sk_test_51RiDcgQN8X8zlq0XfIQNk2c8phbTQfcwiyt5VPoZj7UAGyKvpxvmgjBCV0j78sJyTyxpee0ksSUeOGqirhrWW8UB00PZlI75uS';

// File upload configuration
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    cb(null, 'uploads/');
  },
  filename: function (req, file, cb) {
    cb(null, Date.now() + path.extname(file.originalname));
  }
});

const upload = multer({ storage: storage });

// Bridgecard routes (virtual card issuing)
try {
  const bridgecardRoutes = require('./bridgecardRoutes');
  // Pass database pool to bridgecardRoutes for webhook database updates
  if (bridgecardRoutes.setDatabasePool) {
    bridgecardRoutes.setDatabasePool(pool);
  }
  app.use('/api/bridgecard', bridgecardRoutes);
  console.log('Bridgecard routes mounted at /api/bridgecard');
} catch (e) {
  console.warn('Bridgecard routes not mounted:', e.message);
}

// Bridgecard webhook endpoint (also handled by bridgecardRoutes, but keeping this for backward compatibility)
app.post('/webhook/bridgecard', async (req, res) => {
  try {
    console.log('═══════════════════════════════════════════════════════════');
    console.log('📨 Bridgecard Webhook Event Received (main endpoint)');
    console.log('═══════════════════════════════════════════════════════════');
    console.log('Event Type:', req.body.event_type || req.body.type || 'unknown');
    console.log('Event Data:', JSON.stringify(req.body, null, 2));
    
    // TODO: Verify webhook signature if Bridgecard provides one
    // const signature = req.headers['x-bridgecard-signature'];
    // if (signature && process.env.BRIDGECARD_WEBHOOK_SECRET) {
    //   const isValid = verifyWebhookSignature(req.body, signature, process.env.BRIDGECARD_WEBHOOK_SECRET);
    //   if (!isValid) {
    //     return res.status(401).json({ error: 'Invalid webhook signature' });
    //   }
    // }
    
    const eventType = req.body.event_type || req.body.type || 'unknown';
    const eventData = req.body.data || req.body;
    
    // Handle cardholder verification event
    if (eventType === 'cardholder.verified' || eventType === 'cardholder_verified') {
      const cardholderId = eventData.cardholder_id || eventData.id;
      if (cardholderId) {
        try {
          const connection = await pool.getConnection();
          await connection.execute(
            'UPDATE cardholders SET is_bridgecard_verified = TRUE WHERE cardholder_id = ?',
            [cardholderId]
          );
          connection.release();
          console.log(`✅ Updated cardholder ${cardholderId} verification status to TRUE in database`);
        } catch (dbError) {
          console.error('❌ Failed to update cardholder verification in database:', dbError);
        }
      }
    }
    
    res.json({ received: true });
  } catch (err) {
    console.error('Bridgecard webhook error:', err);
    // Still return 200 to prevent Bridgecard from retrying
    res.status(200).json({ received: true, error: err.message });
  }
});

// Initialize database tables
async function initializeDatabase() {
  try {
    const connection = await pool.getConnection();
    
    // Update transactions table to add currency_exchange type if it doesn't exist
    try {
      await connection.execute(`
        ALTER TABLE transactions 
        MODIFY COLUMN transaction_type ENUM('transfer', 'deposit', 'withdrawal', 'bill_payment', 'airtime', 'data', 'currency_exchange') NOT NULL
      `);
      console.log('✅ Updated transactions table to include currency_exchange type');
    } catch (alterError) {
      // Column might already be updated or table doesn't exist yet
      if (alterError.code !== 'ER_DUP_FIELDNAME' && !alterError.message.includes('doesn\'t exist')) {
        console.log('⚠️ Could not update transaction_type ENUM (may already be updated):', alterError.message);
      }
    }
    
    // Users table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        email VARCHAR(255) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        first_name VARCHAR(100) NOT NULL,
        last_name VARCHAR(100) NOT NULL,
        phone VARCHAR(20),
        date_of_birth DATE,
        address TEXT,
        profile_image VARCHAR(255),
        is_verified BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    // Wallets table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS wallets (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        wallet_name VARCHAR(100) NOT NULL,
        wallet_type ENUM('fiat', 'crypto') DEFAULT 'fiat',
        currency VARCHAR(10) DEFAULT 'USD',
        balance DECIMAL(15,2) DEFAULT 0.00,
        is_active BOOLEAN DEFAULT TRUE,
        is_default BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
      )
    `);

    // Bank accounts table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS bank_accounts (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        account_name VARCHAR(100) NOT NULL,
        account_number VARCHAR(50) NOT NULL,
        bank_name VARCHAR(100) NOT NULL,
        routing_number VARCHAR(20),
        account_type ENUM('savings', 'checking', 'business') DEFAULT 'checking',
        country VARCHAR(10),
        swift_code VARCHAR(20),
        is_default BOOLEAN DEFAULT FALSE,
        is_verified BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
      )
    `);

    // Transactions table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS transactions (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        transaction_type ENUM('transfer', 'deposit', 'withdrawal', 'bill_payment', 'airtime', 'data', 'currency_exchange') NOT NULL,
        amount DECIMAL(15,2) NOT NULL,
        currency VARCHAR(10) DEFAULT 'USD',
        status ENUM('pending', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
        reference_id VARCHAR(100) UNIQUE,
        description TEXT,
        recipient_id INT,
        sender_id INT,
        wallet_id INT,
        sender_wallet_id INT,
        receiver_wallet_id INT,
        transaction_fee DECIMAL(15,2) DEFAULT 0.00,
        recipient_phone VARCHAR(20),
        recipient_name VARCHAR(100),
        recipient_bank VARCHAR(100),
        recipient_account_number VARCHAR(50),
        sender_phone VARCHAR(20),
        sender_name VARCHAR(100),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (recipient_id) REFERENCES users(id) ON DELETE SET NULL,
        FOREIGN KEY (sender_id) REFERENCES users(id) ON DELETE SET NULL
      )
    `);

    // Recipients table - Create or migrate
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS recipients (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        name VARCHAR(100) NOT NULL,
        type ENUM('beneficiary', 'p2p') NOT NULL DEFAULT 'p2p',
        email VARCHAR(255),
        phone VARCHAR(20),
        bank_name VARCHAR(100),
        bank_code VARCHAR(20),
        account_number VARCHAR(50),
        account_name VARCHAR(100),
        routing_number VARCHAR(20),
        recipient_user_id INT,
        nickname VARCHAR(100),
        is_active BOOLEAN DEFAULT TRUE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (recipient_user_id) REFERENCES users(id) ON DELETE SET NULL,
        INDEX idx_user_id (user_id),
        INDEX idx_type (type)
      )
    `);

    // Migrate existing recipients table if needed (add new columns)
    try {
      // Check if bank_code column exists
      const [bankCodeCheck] = await connection.execute(`
        SELECT COUNT(*) as count FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'recipients' 
        AND COLUMN_NAME = 'bank_code'
      `);
      
      if (bankCodeCheck[0].count === 0) {
        await connection.execute(`ALTER TABLE recipients ADD COLUMN bank_code VARCHAR(20) NULL AFTER bank_name`);
        console.log('Added bank_code column to recipients table');
      }

      const [accountNameCheck] = await connection.execute(`
        SELECT COUNT(*) as count FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'recipients' 
        AND COLUMN_NAME = 'account_name'
      `);
      
      if (accountNameCheck[0].count === 0) {
        await connection.execute(`ALTER TABLE recipients ADD COLUMN account_name VARCHAR(100) NULL AFTER account_number`);
        console.log('Added account_name column to recipients table');
      }

      const [recipientUserIdCheck] = await connection.execute(`
        SELECT COUNT(*) as count FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'recipients' 
        AND COLUMN_NAME = 'recipient_user_id'
      `);
      
      if (recipientUserIdCheck[0].count === 0) {
        await connection.execute(`ALTER TABLE recipients ADD COLUMN recipient_user_id INT NULL AFTER routing_number`);
        console.log('Added recipient_user_id column to recipients table');
        
        // Add foreign key
        try {
          await connection.execute(`
            ALTER TABLE recipients 
            ADD CONSTRAINT fk_recipient_user 
            FOREIGN KEY (recipient_user_id) REFERENCES users(id) ON DELETE SET NULL
          `);
          console.log('Added foreign key for recipient_user_id');
        } catch (fkError) {
          if (fkError.code !== 'ER_DUP_FK_NAME') {
            console.log('Note: Could not add foreign key:', fkError.message);
          }
        }

        // Add index
        try {
          await connection.execute(`ALTER TABLE recipients ADD INDEX idx_recipient_user_id (recipient_user_id)`);
          console.log('Added index for recipient_user_id');
        } catch (idxError) {
          if (idxError.code !== 'ER_DUP_KEYNAME') {
            console.log('Note: Could not add index:', idxError.message);
          }
        }
      }

      // Migrate old type values to new type system
      try {
        await connection.execute(`UPDATE recipients SET type = 'p2p' WHERE type IN ('email', 'phone')`);
        await connection.execute(`UPDATE recipients SET type = 'beneficiary' WHERE type = 'bank'`);
      } catch (updateError) {
        // Type migration might fail if ENUM doesn't include old values - that's ok
        console.log('Note: Type migration may have skipped (table may already be migrated)');
      }

      // Try to modify ENUM type (only works if table doesn't have old ENUM values with data)
      try {
        await connection.execute(`ALTER TABLE recipients MODIFY COLUMN type ENUM('beneficiary', 'p2p') NOT NULL DEFAULT 'p2p'`);
        console.log('Updated recipients type ENUM to new values');
      } catch (enumError) {
        console.log('Note: Could not update ENUM (may need manual migration if old data exists)');
      }

    } catch (migrateError) {
      console.log('Note: Recipients table migration check completed with notes:', migrateError.message);
    }

    // Transaction limits table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS transaction_limits (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        wallet_id INT,
        daily_limit DECIMAL(15,2) DEFAULT 1000.00,
        weekly_limit DECIMAL(15,2) DEFAULT 5000.00,
        monthly_limit DECIMAL(15,2) DEFAULT 20000.00,
        single_transaction_limit DECIMAL(15,2) DEFAULT 500.00,
        is_enabled BOOLEAN DEFAULT TRUE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (wallet_id) REFERENCES wallets(id) ON DELETE CASCADE,
        UNIQUE KEY unique_user_wallet_limits (user_id, wallet_id)
      )
    `);

    // Stripe Connect accounts table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS stripe_connect_accounts (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        stripe_account_id VARCHAR(255) UNIQUE NOT NULL,
        account_type ENUM('express', 'standard', 'custom') DEFAULT 'express',
        country VARCHAR(10) NOT NULL,
        currency VARCHAR(10) NOT NULL,
        charges_enabled BOOLEAN DEFAULT FALSE,
        payouts_enabled BOOLEAN DEFAULT FALSE,
        details_submitted BOOLEAN DEFAULT FALSE,
        requirements_currently_due TEXT,
        requirements_eventually_due TEXT,
        requirements_past_due TEXT,
        account_link_url TEXT,
        account_link_expires_at TIMESTAMP NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        UNIQUE KEY unique_user_stripe (user_id)
      )
    `);

    // Withdrawals table (enhanced with Stripe payout info)
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS withdrawals (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        wallet_id INT NOT NULL,
        bank_account_id INT,
        stripe_payout_id VARCHAR(255),
        amount DECIMAL(15,2) NOT NULL,
        currency VARCHAR(10) NOT NULL,
        withdrawal_fee DECIMAL(15,2) DEFAULT 0.00,
        total_amount DECIMAL(15,2) NOT NULL,
        status ENUM('pending', 'processing', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
        failure_reason TEXT,
        reference_id VARCHAR(100) UNIQUE NOT NULL,
        description TEXT,
        estimated_arrival TIMESTAMP NULL,
        completed_at TIMESTAMP NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (wallet_id) REFERENCES wallets(id) ON DELETE CASCADE,
        FOREIGN KEY (bank_account_id) REFERENCES bank_accounts(id) ON DELETE SET NULL
      )
    `);

    // Add missing columns to transactions table if they don't exist
    const transactionColumnsToAdd = [
      { name: 'sender_wallet_id', type: 'INT' },
      { name: 'receiver_wallet_id', type: 'INT' },
      { name: 'transaction_fee', type: 'DECIMAL(15,2) DEFAULT 0.00' },
      { name: 'recipient_phone', type: 'VARCHAR(20)' },
      { name: 'recipient_name', type: 'VARCHAR(100)' },
      { name: 'recipient_bank', type: 'VARCHAR(100)' },
      { name: 'recipient_account_number', type: 'VARCHAR(50)' },
      { name: 'sender_phone', type: 'VARCHAR(20)' },
      { name: 'sender_name', type: 'VARCHAR(100)' },
      { name: 'stripe_payment_intent_id', type: 'VARCHAR(255)' },
      { name: 'stripe_payout_id', type: 'VARCHAR(255)' },
      { name: 'exchange_rate', type: 'DECIMAL(10,6) DEFAULT 1.000000' },
      { name: 'original_amount', type: 'DECIMAL(15,2)' },
      { name: 'original_currency', type: 'VARCHAR(10)' },
      { name: 'converted_amount', type: 'DECIMAL(15,2)' },
      { name: 'converted_currency', type: 'VARCHAR(10)' }
    ];

    // Add missing columns to bank_accounts table if they don't exist
    const bankAccountColumnsToAdd = [
      { name: 'country', type: 'VARCHAR(10)' },
      { name: 'swift_code', type: 'VARCHAR(20)' },
      { name: 'is_default', type: 'BOOLEAN DEFAULT FALSE' },
      { name: 'provider', type: 'VARCHAR(20) DEFAULT NULL' },
      { name: 'flutterwave_subaccount_id', type: 'VARCHAR(100) DEFAULT NULL' }
    ];

    // Add account_number to users table if it doesn't exist
    try {
      await connection.execute(`ALTER TABLE users ADD COLUMN account_number VARCHAR(50) UNIQUE`);
      console.log('Added account_number column to users table');
    } catch (error) {
      if (error.code === 'ER_DUP_FIELDNAME') {
        console.log('Column account_number already exists in users table');
      } else {
        console.log('Note: Could not add account_number to users table:', error.message);
      }
    }

    // Add columns to transactions table
    for (const column of transactionColumnsToAdd) {
      try {
        await connection.execute(`ALTER TABLE transactions ADD COLUMN ${column.name} ${column.type}`);
        console.log(`Added column ${column.name} to transactions table`);
      } catch (error) {
        if (error.code === 'ER_DUP_FIELDNAME') {
          console.log(`Column ${column.name} already exists in transactions table`);
        } else {
          console.log(`Error adding column ${column.name}:`, error.message);
        }
      }
    }

    // Add columns to bank_accounts table
    for (const column of bankAccountColumnsToAdd) {
      try {
        await connection.execute(`ALTER TABLE bank_accounts ADD COLUMN ${column.name} ${column.type}`);
        console.log(`Added column ${column.name} to bank_accounts table`);
      } catch (error) {
        if (error.code === 'ER_DUP_FIELDNAME') {
          console.log(`Column ${column.name} already exists in bank_accounts table`);
        } else {
          console.log(`Error adding column ${column.name}:`, error.message);
        }
      }
    }

    // Bill categories table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS bill_categories (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        description TEXT,
        icon VARCHAR(50),
        color VARCHAR(20),
        is_active BOOLEAN DEFAULT TRUE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    // Bill providers table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS bill_providers (
        id INT AUTO_INCREMENT PRIMARY KEY,
        category_id INT NOT NULL,
        name VARCHAR(100) NOT NULL,
        code VARCHAR(50) UNIQUE NOT NULL,
        description TEXT,
        logo_url VARCHAR(255),
        is_active BOOLEAN DEFAULT TRUE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (category_id) REFERENCES bill_categories(id) ON DELETE CASCADE
      )
    `);

    // Bill payments table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS bill_payments (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        provider_id INT NOT NULL,
        customer_number VARCHAR(100) NOT NULL,
        customer_name VARCHAR(100),
        customer_address TEXT,
        amount DECIMAL(15,2) NOT NULL,
        currency VARCHAR(10) DEFAULT 'USD',
        processing_fee DECIMAL(15,2) DEFAULT 0.00,
        total_amount DECIMAL(15,2) NOT NULL,
        status ENUM('pending', 'processing', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
        stripe_payment_intent_id VARCHAR(255),
        transaction_id VARCHAR(100) UNIQUE,
        receipt_url VARCHAR(255),
        payment_method VARCHAR(50),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (provider_id) REFERENCES bill_providers(id) ON DELETE CASCADE
      )
    `);

    // Withdrawals table
    // First, try to alter existing table to allow NULL for bank_account_id (for card withdrawals)
    try {
      await connection.execute(`
        ALTER TABLE withdrawals 
        MODIFY COLUMN bank_account_id INT NULL
      `);
      console.log('✅ Updated withdrawals table to allow NULL bank_account_id');
    } catch (error) {
      // Column might already be nullable or table doesn't exist yet
      if (error.code !== 'ER_BAD_FIELD_ERROR' && error.code !== 'ER_NO_SUCH_TABLE') {
        console.log('⚠️ Could not alter bank_account_id column (may already be nullable):', error.message);
      }
    }

    await connection.execute(`
      CREATE TABLE IF NOT EXISTS withdrawals (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        wallet_id INT NOT NULL,
        bank_account_id INT NULL,
        amount DECIMAL(15,2) NOT NULL,
        currency VARCHAR(10) DEFAULT 'NGN',
        withdrawal_fee DECIMAL(15,2) DEFAULT 0.00,
        total_amount DECIMAL(15,2) NOT NULL,
        status ENUM('pending', 'processing', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
        reference_id VARCHAR(100) UNIQUE,
        description TEXT,
        bank_transfer_id VARCHAR(255),
        processed_at TIMESTAMP NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (wallet_id) REFERENCES wallets(id) ON DELETE CASCADE,
        FOREIGN KEY (bank_account_id) REFERENCES bank_accounts(id) ON DELETE CASCADE
      )
    `);

    // Payment methods table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS payment_methods (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        type ENUM('card', 'bank_account', 'wallet') NOT NULL,
        name VARCHAR(100) NOT NULL,
        details JSON,
        is_default BOOLEAN DEFAULT FALSE,
        is_active BOOLEAN DEFAULT TRUE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
      )
    `);

    // Crypto transactions table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS crypto_transactions (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        crypto_type VARCHAR(20) NOT NULL,
        transaction_type ENUM('buy', 'sell', 'transfer') NOT NULL,
        amount DECIMAL(15,8) NOT NULL,
        price_per_unit DECIMAL(15,2) NOT NULL,
        total_value DECIMAL(15,2) NOT NULL,
        wallet_id INT,
        status ENUM('pending', 'completed', 'failed') DEFAULT 'pending',
        transaction_hash VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (wallet_id) REFERENCES wallets(id)
      )
    `);

    // Password reset tokens table
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS password_reset_tokens (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        token VARCHAR(255) NOT NULL,
        expires_at TIMESTAMP NOT NULL,
        used BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
      )
    `);

    // Insert default bill categories
    await connection.execute(`
      INSERT IGNORE INTO bill_categories (id, name, description, icon, color) VALUES
      (1, 'Electricity', 'Pay electricity bills', 'flash', '#FFD700'),
      (2, 'Cable TV', 'Pay cable TV subscriptions', 'tv', '#FF6B6B'),
      (3, 'Internet', 'Pay internet service bills', 'globe', '#4ECDC4'),
      (4, 'Water', 'Pay water utility bills', 'water', '#45B7D1'),
      (5, 'Airtime', 'Buy airtime credit', 'phone-portrait', '#FF9F43'),
      (6, 'Data', 'Buy data bundles', 'cellular', '#6C5CE7')
    `);

    // Insert default bill providers with currency support
    await connection.execute(`
      INSERT IGNORE INTO bill_providers (category_id, name, code, description, currency) VALUES
      (1, 'Eko Electricity (EKEDC)', 'ekedc', 'Eko Electricity Distribution Company', 'NGN'),
      (1, 'Ikeja Electric (IKEDC)', 'ikedc', 'Ikeja Electricity Distribution Company', 'NGN'),
      (1, 'Abuja Electricity (AEDC)', 'aedc', 'Abuja Electricity Distribution Company', 'NGN'),
      (1, 'Port Harcourt Electric (PHED)', 'phed', 'Port Harcourt Electricity Distribution Company', 'NGN'),
      (2, 'DSTV', 'dstv', 'DSTV Cable TV', 'NGN'),
      (2, 'GOtv', 'gotv', 'GOtv Cable TV', 'NGN'),
      (2, 'StarTimes', 'startimes', 'StarTimes Cable TV', 'NGN'),
      (2, 'Showmax', 'showmax', 'Showmax Streaming', 'NGN'),
      (3, 'MTN Broadband', 'mtn_broadband', 'MTN Internet Service', 'NGN'),
      (3, 'Airtel Broadband', 'airtel_broadband', 'Airtel Internet Service', 'NGN'),
      (3, 'Spectranet', 'spectranet', 'Spectranet Internet Service', 'NGN'),
      (3, 'Swift Networks', 'swift', 'Swift Networks Internet Service', 'NGN'),
      (4, 'Lagos Water Corporation', 'lagos_water', 'Lagos State Water Corporation', 'NGN'),
      (4, 'FCT Water Board', 'abuja_water', 'Federal Capital Territory Water Board', 'NGN'),
      (4, 'Kano State Water Board', 'kano_water', 'Kano State Water Board', 'NGN'),
      (4, 'Rivers State Water Board', 'rivers_water', 'Rivers State Water Board', 'NGN')
    `);

    // Cardholders table (for Bridgecard integration)
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS cardholders (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL UNIQUE,
        cardholder_id VARCHAR(255) UNIQUE NOT NULL,
        first_name VARCHAR(100) NOT NULL,
        last_name VARCHAR(100) NOT NULL,
        email VARCHAR(255) NOT NULL,
        phone VARCHAR(20),
        date_of_birth DATE,
        country VARCHAR(10) DEFAULT 'US',
        address TEXT,
        status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
        is_bridgecard_verified BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        INDEX idx_user_id (user_id),
        INDEX idx_cardholder_id (cardholder_id)
      )
    `);
    console.log('✅ Cardholders table created/verified');
    
    // Add is_bridgecard_verified column if it doesn't exist (for existing tables)
    try {
      const [columnCheck] = await connection.execute(`
        SELECT COUNT(*) as count FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'cardholders' 
        AND COLUMN_NAME = 'is_bridgecard_verified'
      `);
      
      if (columnCheck[0].count === 0) {
        await connection.execute(`ALTER TABLE cardholders ADD COLUMN is_bridgecard_verified BOOLEAN DEFAULT FALSE AFTER status`);
        console.log('✅ Added is_bridgecard_verified column to cardholders table');
      }
    } catch (alterError) {
      console.log('Note: Could not add is_bridgecard_verified column (may already exist):', alterError.message);
    }

    // Insert default payment methods
    await connection.execute(`
      INSERT IGNORE INTO payment_methods (user_id, type, name, details, is_default) VALUES
      (1, 'wallet', 'Default Wallet', '{"walletId": 1}', TRUE),
      (1, 'card', 'Visa Card', '{"last4": "4242", "brand": "visa", "expMonth": 12, "expYear": 2025}', FALSE),
      (1, 'bank_account', 'Bank Account', '{"accountNumber": "1234567890", "bankName": "Globana Bank"}', FALSE)
    `);

    connection.release();
    console.log('Database tables initialized successfully');
  } catch (error) {
    console.error('Database initialization error:', error);
  }
}

// Initialize database on startup
initializeDatabase();

// Authentication middleware
const authenticateToken = async (req, res, next) => {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];

  if (!token) {
    return res.status(401).json({ error: 'Access token required' });
  }

  try {
    const decoded = jwt.verify(token, JWT_SECRET);
    
    const connection = await pool.getConnection();
    const [users] = await connection.execute('SELECT * FROM users WHERE id = ?', [decoded.userId]);
    connection.release();

    if (users.length === 0) {
      return res.status(401).json({ error: 'Invalid token' });
    }

    req.user = users[0];
    req.user.kycStatus = users[0].kyc_status || 'pending';
    next();
  } catch (error) {
    return res.status(403).json({ error: 'Invalid token' });
  }
};

// KYC validation middleware - blocks transactions if KYC not approved
const requireKYCApproval = (req, res, next) => {
  const kycStatus = req.user.kyc_status || req.user.kycStatus || 'pending';
  
  if (kycStatus !== 'approved') {
    return res.status(403).json({ 
      error: 'KYC verification required',
      message: 'Please complete your identity verification to perform transactions',
      kycStatus 
    });
  }
  
  next();
};

// Generate unique reference ID
const generateReferenceId = () => {
  const timestamp = Date.now();
  const random = Math.random().toString(36).substr(2, 9).toUpperCase();
  const processId = process.pid || Math.floor(Math.random() * 10000);
  const counter = Math.floor(Math.random() * 1000000);
  return `GT${timestamp}${processId}${counter}${random}`;
};

// ==================== BRIDGECARD ENDPOINTS ====================

// In-memory store for cardholders (in production, use database)
const cardholderStore = new Map();

// Bridgecard authentication middleware
const bridgecardAuth = (req, res, next) => {
  const key = req.headers['x-app-key'] || req.headers['authorization']?.replace('Bearer ', '');
  const appKey = process.env.APP_API_KEY || 'test-key'; // Default key for testing
  if (!key || (appKey && key !== appKey)) {
    console.log('Bridgecard auth check:', { key, appKey, headers: req.headers });
    return res.status(401).json({ error: 'Unauthorized', key, appKey });
  }
  next();
};

// Check if user has a cardholder profile
app.get('/api/bridgecard/cardholder/check', authenticateToken, async (req, res) => {
  try {
    const connection = await pool.getConnection();
    const [cardholders] = await connection.execute(
      'SELECT * FROM cardholders WHERE user_id = ? AND status = ?',
      [req.user.id, 'active']
    );
    connection.release();

    if (cardholders.length === 0) {
      return res.json({
        success: true,
        hasCardholder: false,
        isVerified: false,
        canIssueCard: false,
        message: 'User does not have a cardholder profile'
      });
    }

    const cardholder = cardholders[0];
    // Check if is_bridgecard_verified is explicitly true (handle NULL as false)
    const dbVerified = cardholder.is_bridgecard_verified === 1 || cardholder.is_bridgecard_verified === true;
    
    // If marked as verified in DB, optionally verify with Bridgecard API
    let isVerified = dbVerified;
    let bridgecardCheck = null;
    
    const token = process.env.BRIDGECARD_ACCESS_TOKEN;
    if (token && dbVerified && cardholder.cardholder_id) {
      try {
        // Verify cardholder exists and is verified in Bridgecard
        // Use the get_cardholder endpoint as per Bridgecard docs
        const baseURL = process.env.BRIDGECARD_BASE_URL || 'https://issuecards.api.bridgecard.co/v1/issuing/sandbox';
        const bridgecardUrl = `${baseURL}/cardholder/get_cardholder?cardholder_id=${cardholder.cardholder_id}`;
        
        const response = await axios.get(bridgecardUrl, {
          headers: {
            'token': `Bearer ${token}`, // Bridgecard uses 'token: Bearer <token>' format
            'Content-Type': 'application/json'
          },
          timeout: 30000
        });
        
        console.log('📋 Bridgecard cardholder check response:', JSON.stringify(response.data, null, 2));
        
        if (response.data && response.data.status === 'success' && response.data.data) {
          const cardholderData = response.data.data;
          const exists = !!cardholderData.cardholder_id;
          const verified = cardholderData.verified === true || cardholderData.verified === 'true';
          const status = cardholderData.status; // active, deactivated, etc.
          
          isVerified = exists && verified && status === 'active';
          bridgecardCheck = { 
            exists: exists, 
            verified: verified,
            status: status,
            canIssueCard: isVerified
          };
          
          if (isVerified) {
            console.log(`✅ Cardholder ${cardholder.cardholder_id} verified and active in Bridgecard`);
          } else {
            console.warn(`⚠️ Cardholder ${cardholder.cardholder_id} exists but: verified=${verified}, status=${status}`);
          }
        } else {
          isVerified = false;
          bridgecardCheck = { exists: false, verified: false };
          console.warn(`⚠️ Cardholder ${cardholder.cardholder_id} not found in Bridgecard`);
        }
      } catch (apiError) {
        // If API check fails, trust the database flag but log warning
        console.warn(`⚠️ Could not verify cardholder with Bridgecard API:`, apiError.message);
        console.warn(`   Error details:`, apiError.response?.data || apiError.message);
        bridgecardCheck = { exists: null, verified: null, error: apiError.message };
        // Still use DB flag if API check fails
        isVerified = dbVerified;
      }
    }
    
    return res.json({
      success: true,
      hasCardholder: true,
      isVerified: isVerified,
      canIssueCard: isVerified, // Only can issue if verified
      bridgecardCheck: bridgecardCheck,
      cardholder: {
        cardholder_id: cardholder.cardholder_id,
        first_name: cardholder.first_name,
        last_name: cardholder.last_name,
        email: cardholder.email,
        status: cardholder.status,
        is_bridgecard_verified: dbVerified
      }
    });
  } catch (error) {
    console.error('Check cardholder error:', error);
    return res.status(500).json({
      success: false,
      error: 'Failed to check cardholder status',
      details: error.message
    });
  }
});

// Create cardholder
app.post('/api/bridgecard/cardholder', authenticateToken, async (req, res) => {
  try {
    // Check if user already has a cardholder
    const connection = await pool.getConnection();
    const [existingCardholders] = await connection.execute(
      'SELECT * FROM cardholders WHERE user_id = ?',
      [req.user.id]
    );

    if (existingCardholders.length > 0) {
      connection.release();
      return res.status(400).json({
        success: false,
        error: 'User already has a cardholder profile',
        cardholder_id: existingCardholders[0].cardholder_id
      });
    }

    // Validate required fields
    const { first_name, last_name, email } = req.body;
    if (!first_name || !last_name || !email) {
      connection.release();
      return res.status(400).json({ 
        success: false, 
        error: 'Missing required fields: first_name, last_name, email' 
      });
    }

    // Keep connection for later use

    // Build payload according to Bridgecard API requirements
    // Bridgecard uses 'email' (not 'email_address') and requires top-level 'country' field
    const bridgecardPayload = {
      first_name: (req.body.first_name || '').trim(),
      last_name: (req.body.last_name || '').trim(),
      email: (req.body.email || req.body.email_address || '').trim(), // Bridgecard uses 'email' not 'email_address'
      phone: (req.body.phone || '').trim(),
      country: (req.body.country || req.body.address?.country || 'US').trim(), // Top-level country field is REQUIRED
      ...(req.body.date_of_birth ? { date_of_birth: req.body.date_of_birth } : {}),
    };
    
    // Validate required fields
    if (!bridgecardPayload.first_name || !bridgecardPayload.last_name || !bridgecardPayload.email || !bridgecardPayload.phone || !bridgecardPayload.country) {
      connection.release();
      return res.status(400).json({
        success: false,
        error: 'Missing required fields',
        message: 'first_name, last_name, email, phone, and country are required',
        missing: {
          first_name: !bridgecardPayload.first_name,
          last_name: !bridgecardPayload.last_name,
          email: !bridgecardPayload.email,
          phone: !bridgecardPayload.phone,
          country: !bridgecardPayload.country
        }
      });
    }
    
    // Handle address - Bridgecard expects specific structure
    // Address is required for cardholder registration
    // Bridgecard uses 'street' not 'address' inside the address object
    if (req.body.address) {
      const addressData = {
        street: (req.body.address.street || req.body.address.address || '').trim(), // Bridgecard uses 'street' not 'address'
        city: (req.body.address.city || '').trim(),
        state: (req.body.address.state || '').trim(), // Remove trailing spaces
        country: (req.body.address.country || bridgecardPayload.country || 'US').trim(),
        postal_code: (req.body.address.postalCode || req.body.address.postal_code || '').trim(),
      };
      
      // Add house_no if provided
      if (req.body.address.house_no) {
        addressData.house_no = String(req.body.address.house_no).trim();
      }
      
      // Validate required address fields
      const requiredAddressFields = ['street', 'city', 'state', 'country', 'postal_code'];
      const missingFields = requiredAddressFields.filter(field => !addressData[field] || addressData[field] === '');
      
      if (missingFields.length > 0) {
        connection.release();
        return res.status(400).json({
          success: false,
          error: 'Incomplete address information',
          message: 'Address must include: street, city, state, country, and postal_code',
          missingFields: missingFields
        });
      }
      
      bridgecardPayload.address = addressData;
    } else {
      // Address is required - return error if not provided
      connection.release();
      return res.status(400).json({
        success: false,
        error: 'Address is required for cardholder registration',
        message: 'Please provide a complete address including street, city, state, country, and postal_code'
      });
    }
    
    // Handle identity verification if provided
    if (req.body.identity) {
      bridgecardPayload.identity = {
        id_type: req.body.identity.id_type || 'NIGERIAN_BVN_VERIFICATION',
        ...(req.body.identity.bvn ? { bvn: req.body.identity.bvn } : {}),
        ...(req.body.identity.selfie_image ? { selfie_image: req.body.identity.selfie_image } : {}),
      };
    }
    
    // Add meta_data if provided
    if (req.body.meta_data) {
      bridgecardPayload.meta_data = req.body.meta_data;
    } else {
      // Add default meta_data with user_id
      bridgecardPayload.meta_data = {
        user_id: req.user.id.toString()
      };
    }
    
    const payload = bridgecardPayload;

    console.log('Creating cardholder with payload:', payload);
    
    // ============================================================
    // STEP 1: CALL BRIDGECARD API FIRST - BEFORE DATABASE
    // ============================================================
    // IMPORTANT: We MUST create the cardholder in Bridgecard API first,
    // then use the cardholder_id returned by Bridgecard to store in our database.
    // We do NOT create local/mock IDs - only use IDs from Bridgecard.
    
    const token = process.env.BRIDGECARD_ACCESS_TOKEN;
    let cardholderId = null;
    let isBridgecardVerified = false;
    let cardholderData = null;

    if (!token) {
      connection.release();
      return res.status(400).json({
        success: false,
        error: 'Bridgecard API token not configured. Cardholder creation requires Bridgecard API access.',
        requiresBridgecardToken: true
      });
    }

    console.log('═══════════════════════════════════════════════════════════');
    console.log('STEP 1: CALLING BRIDGECARD API TO CREATE CARDHOLDER');
    console.log('═══════════════════════════════════════════════════════════');
    
    try {
      // Use real Bridgecard API to create cardholder - REQUIRED
      // According to Bridgecard docs: synchronous API can take up to 45 seconds
      const baseURL = process.env.BRIDGECARD_BASE_URL || 'https://issuecards.api.bridgecard.co/v1/issuing/sandbox';
      
      // Bridgecard endpoint for creating cardholder - CORRECT ENDPOINT
      // Use asynchronous endpoint (recommended) or synchronous if specified
      const useSynchronous = req.body.synchronous === true;
      const endpoint = useSynchronous 
        ? '/cardholder/register_cardholder_synchronously'  // Synchronous (45 sec timeout)
        : '/cardholder/register_cardholder';  // Asynchronous (recommended, webhook-based)
      
      const url = `${baseURL}${endpoint}`;
      
      console.log('🔄 STEP 1: Creating cardholder with Bridgecard API (BEFORE database):');
      console.log('   URL:', url);
      console.log('   Method:', useSynchronous ? 'SYNCHRONOUS (45 sec timeout)' : 'ASYNCHRONOUS (webhook-based)');
      console.log('   Payload:', JSON.stringify(payload, null, 2));
      console.log('   Token:', token ? `${token.substring(0, 10)}...` : 'NOT SET');
      console.log('   Timeout: 45000ms (45 seconds as per Bridgecard docs)');
      console.log('   ⚠️  This API call happens BEFORE any database insert!');
      
      // Bridgecard uses 'token' header with Bearer format
      // Set timeout to 45 seconds as per Bridgecard documentation
      let response;
      let lastError = null;
      
      // Try different header formats
      const headerFormats = [
        { 'token': `Bearer ${token}`, 'Content-Type': 'application/json' },
        { 'token': token, 'Content-Type': 'application/json' },
        { 'Authorization': `Bearer ${token}`, 'Content-Type': 'application/json' },
      ];
      
      for (const headers of headerFormats) {
        try {
          response = await axios.post(url, payload, {
            headers: headers,
            timeout: 45000 // 45 seconds as per Bridgecard docs
          });
          
          // If we get a successful response, break
          if (response && (response.status === 200 || response.status === 201)) {
            console.log(`✅ Success with headers: ${Object.keys(headers).join(', ')}`);
            break;
          }
        } catch (error) {
          lastError = error;
          // Continue to next header format
          continue;
        }
      }
      
      // If all attempts failed, throw the last error
      if (!response || (response.status !== 200 && response.status !== 201)) {
        throw lastError || new Error('All header format attempts failed');
      }

      console.log('✅ Bridgecard API response status:', response.status);
      console.log('✅ Bridgecard API response data:', JSON.stringify(response.data, null, 2));
      
      // Check for success response - Bridgecard typically returns:
      // { status: 'success', message: '...', data: { cardholder_id: '...', verified: true/false, ... } }
      if (response.data) {
        // Bridgecard standard format: { status: 'success', data: { cardholder_id: '...', verified: ... } }
        if (response.data.status === 'success' && response.data.data) {
          cardholderId = response.data.data.cardholder_id || response.data.data.id;
          
          // Check if cardholder is verified (synchronous verification can happen immediately)
          // If verified is true, mark as verified; otherwise it will be verified via webhook later
          const verified = response.data.data.verified === true || response.data.data.verified === 'true';
          isBridgecardVerified = verified;
          
          if (cardholderId) {
            console.log('✅ Created cardholder with Bridgecard API:', cardholderId);
            console.log('   Verified status:', verified ? 'VERIFIED' : 'PENDING VERIFICATION (webhook will notify)');
            
            // If not verified yet, we still store it but mark as pending
            // Webhook will update is_bridgecard_verified when verification completes
            if (!verified) {
              console.log('⚠️ Cardholder created but verification pending. Webhook will notify when verified.');
            }
          } else {
            throw new Error('Bridgecard API returned success but no cardholder_id in response.data.data');
          }
        }
        // Alternative format: { success: true, cardholder_id: '...' }
        else if (response.data.success && response.data.cardholder_id) {
          cardholderId = response.data.cardholder_id;
          isBridgecardVerified = response.data.verified === true || response.data.verified === 'true';
          console.log('✅ Created cardholder with Bridgecard API (alt format):', cardholderId);
        }
        // Direct cardholder_id in response
        else if (response.data.cardholder_id || response.data.id) {
          cardholderId = response.data.cardholder_id || response.data.id;
          isBridgecardVerified = response.data.verified === true || response.data.verified === 'true';
          console.log('✅ Created cardholder with Bridgecard API (direct format):', cardholderId);
        }
        // If HTTP status is success but we can't find cardholder_id, try to extract it
        else if (response.status === 200 || response.status === 201) {
          console.warn('⚠️ Bridgecard API returned HTTP success but unexpected response format:');
          console.warn('   Full response:', JSON.stringify(response.data, null, 2));
          
          // Try to extract any ID from the response
          const responseStr = JSON.stringify(response.data);
          const idMatch = responseStr.match(/"cardholder_id"\s*:\s*"([^"]+)"/i) || 
                         responseStr.match(/"id"\s*:\s*"([^"]+)"/i);
          if (idMatch) {
            cardholderId = idMatch[1];
            isBridgecardVerified = false; // Assume pending if we had to extract
            console.log('✅ Extracted cardholder ID from response:', cardholderId);
            console.log('⚠️ Verification status unknown - will be updated via webhook');
          } else {
            throw new Error('Bridgecard API returned success but no cardholder_id found in response');
          }
        }
      }
      
      // If we didn't get a cardholder ID from Bridgecard API, return error
      // DO NOT proceed to database insert if API call failed
      if (!cardholderId) {
        const errorMessage = response?.data?.message || 
                             response?.data?.error || 
                             'Failed to create cardholder - no cardholder_id in response';
        connection.release();
        console.error('❌ STEP 1 FAILED: Bridgecard API did not return cardholder_id');
        console.error('   NOT proceeding to database insert - cardholder was NOT created in Bridgecard');
        return res.status(400).json({
          success: false,
          error: `Bridgecard API error: ${errorMessage}`,
          details: response?.data || { message: 'Unexpected response format' },
          responseStatus: response?.status
        });
      }
      
      console.log('═══════════════════════════════════════════════════════════');
      console.log('✅ STEP 1 SUCCESS: Cardholder created in Bridgecard API');
      console.log('   Cardholder ID from Bridgecard:', cardholderId);
      console.log('   Verified status:', isBridgecardVerified ? 'VERIFIED' : 'PENDING');
      console.log('═══════════════════════════════════════════════════════════');
      
      // Note: isBridgecardVerified might be false if verification is pending (async)
      // We still store the cardholder, but mark verification status
      // Webhook will update is_bridgecard_verified when verification completes
    } catch (apiError) {
      console.error('═══════════════════════════════════════════════════════════');
      console.error('❌ STEP 1 FAILED: Bridgecard API call failed');
      console.error('═══════════════════════════════════════════════════════════');
      console.error('❌ Bridgecard API error:', apiError.response?.data || apiError.message);
      console.error('   Status:', apiError.response?.status);
      console.error('   URL attempted:', apiError.config?.url || url);
      console.error('   Payload sent:', JSON.stringify(payload, null, 2));
      console.error('   NOT proceeding to database insert - cardholder was NOT created in Bridgecard');
      connection.release();
      
      // Handle 422 validation errors with detailed messages
      if (apiError.response?.status === 422) {
        const validationErrors = apiError.response?.data?.detail || [];
        const errorMessages = Array.isArray(validationErrors) 
          ? validationErrors.map(err => {
              if (typeof err === 'object') {
                return `${err.loc?.join('.') || 'field'}: ${err.msg || err.message || 'validation error'}`;
              }
              return String(err);
            })
          : [String(validationErrors)];
        
        return res.status(422).json({
          success: false,
          error: 'Validation failed',
          message: 'Bridgecard API validation errors',
          validationErrors: errorMessages,
          details: apiError.response?.data
        });
      }
      
      const errorMessage = apiError.response?.data?.message || 
                           apiError.response?.data?.error || 
                           apiError.message || 
                           'Failed to create cardholder with Bridgecard API';
      
      return res.status(apiError.response?.status || 400).json({
        success: false,
        error: `Failed to create cardholder: ${errorMessage}`,
        details: apiError.response?.data || { message: apiError.message },
        status: apiError.response?.status
      });
    }

    // Only proceed to database insert if we have a valid cardholder ID from Bridgecard
    if (!cardholderId) {
      connection.release();
      console.error('❌ CRITICAL: No cardholder_id from Bridgecard API - cannot proceed to database');
      return res.status(500).json({
        success: false,
        error: 'Failed to get cardholder ID from Bridgecard API'
      });
    }
    
    // ============================================================
    // STEP 2: STORE IN DATABASE - ONLY AFTER BRIDGECARD API SUCCESS
    // ============================================================
    console.log('═══════════════════════════════════════════════════════════');
    console.log('STEP 2: STORING CARDHOLDER IN DATABASE');
    console.log('═══════════════════════════════════════════════════════════');
    console.log('✅ Using cardholder_id from Bridgecard API:', cardholderId);
    console.log('   This happens AFTER successful Bridgecard API call');
    
    // Store cardholder in database (even if verification is pending - webhook will update later)
    // According to Bridgecard docs: verification can be synchronous (immediate) or async (via webhook ~2 min)
    await connection.execute(
      `INSERT INTO cardholders 
       (user_id, cardholder_id, first_name, last_name, email, phone, date_of_birth, country, address, status, is_bridgecard_verified)
       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'active', ?)`,
      [
        req.user.id,
        cardholderId,
        payload.first_name,
        payload.last_name,
        payload.email,
        payload.phone || null,
        payload.date_of_birth || null,
        payload.country,
        payload.address ? JSON.stringify(payload.address) : null,
        isBridgecardVerified ? 1 : 0 // Store verification status (true if verified, false if pending)
      ]
    );
    
    // Also store in memory for backward compatibility
    cardholderData = {
      cardholder_id: cardholderId,
      first_name: payload.first_name,
      last_name: payload.last_name,
      email: payload.email,
      phone: payload.phone,
      date_of_birth: payload.date_of_birth,
      country: payload.country,
      address: payload.address,
      created_at: new Date().toISOString(),
      status: 'active'
    };
    cardholderStore.set(cardholderId, cardholderData);
    
    connection.release();
    console.log('═══════════════════════════════════════════════════════════');
    console.log('✅ STEP 2 SUCCESS: Cardholder stored in database');
    console.log(`   Cardholder ID: ${cardholderId}`);
    console.log(`   User ID: ${req.user.id}`);
    console.log(`   Verified: ${isBridgecardVerified ? 'YES' : 'PENDING (webhook will update)'}`);
    console.log('═══════════════════════════════════════════════════════════');
    
    return res.json({ 
      success: true, 
      data: {
        cardholder_id: cardholderId,
        status: 'created',
        message: 'Cardholder created successfully in Bridgecard and stored in database',
        verified: isBridgecardVerified,
        cardholder_data: cardholderData
      }
    });
  } catch (err) {
    console.error('Bridgecard create cardholder error:', err);
    return res.status(500).json({ 
      success: false, 
      error: 'Failed to create cardholder', 
      details: err.message
    });
  }
});

// Issue card
app.post('/api/bridgecard/issue-card', authenticateToken, async (req, res) => {
  try {
    let { cardholder_id } = req.body;
    
    // Get user's cardholder profile - check both with and without cardholder_id
    const connection = await pool.getConnection();
    let cardholder = null;
    
    if (!cardholder_id) {
      // If cardholder_id not provided, get from user's profile
      const [cardholders] = await connection.execute(
        'SELECT * FROM cardholders WHERE user_id = ? AND status = ?',
        [req.user.id, 'active']
      );
      
      if (cardholders.length === 0) {
        connection.release();
      return res.status(400).json({ 
        success: false, 
          error: 'No active cardholder profile found. Please create a cardholder profile first.',
          requiresCardholder: true
        });
      }
      
      cardholder = cardholders[0];
      cardholder_id = cardholder.cardholder_id;
    } else {
      // Validate that the cardholder belongs to the user
      const [cardholders] = await connection.execute(
        'SELECT * FROM cardholders WHERE cardholder_id = ? AND user_id = ? AND status = ?',
        [cardholder_id, req.user.id, 'active']
      );
      
      if (cardholders.length === 0) {
        connection.release();
        return res.status(403).json({
          success: false,
          error: 'Cardholder ID does not belong to this user or is not active. Please create a cardholder profile first.',
          requiresCardholder: true
        });
      }
      
      cardholder = cardholders[0];
    }
    
    // Check if cardholder is active
    if (cardholder.status !== 'active') {
      connection.release();
      return res.status(400).json({
        success: false,
        error: `Cardholder profile is ${cardholder.status}. Cannot issue cards.`,
        requiresCardholder: true,
        cardholder_id: cardholder.cardholder_id
      });
    }
    
    // Check if cardholder was verified in Bridgecard (handle NULL as false)
    const isVerified = cardholder.is_bridgecard_verified === 1 || cardholder.is_bridgecard_verified === true;
    if (!isVerified) {
      connection.release();
      return res.status(400).json({
        success: false,
        error: 'Cardholder profile exists but was not verified in Bridgecard. Please recreate your cardholder profile.',
        requiresCardholder: true,
        cardholder_id: cardholder.cardholder_id
      });
    }
    
    console.log(`✅ Using verified cardholder ${cardholder_id} from user's profile`);
    console.log(`✅ Cardholder details: ${cardholder.first_name} ${cardholder.last_name} (${cardholder.email})`);

    const payload = { ...req.body };
    payload.cardholder_id = cardholder_id; // Ensure cardholder_id is in payload
    
    // Set default values for USD card creation
    payload.card_type = payload.card_type || 'virtual';
    payload.card_brand = payload.card_brand || 'Mastercard';
    payload.card_currency = payload.card_currency || 'USD';
    payload.card_limit = payload.card_limit || '500000'; // $5,000 limit
    payload.funding_amount = payload.funding_amount || '300'; // Minimum $3 for $5k limit
    
    console.log('Issuing card with payload:', payload);
    
    // Use real Bridgecard API
    const bridgecardUrl = 'https://issuecards.api.bridgecard.co/v1/issuing/sandbox/cards';
    const token = process.env.BRIDGECARD_ACCESS_TOKEN;
    
    if (!token) {
      console.log('⚠️ Bridgecard access token not configured, using mock data');
      console.log('⚠️ To use real Bridgecard API, set BRIDGECARD_ACCESS_TOKEN environment variable');
      // Return mock card data for testing
      const mockCardId = `card_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      const last4Digits = Math.floor(Math.random() * 9000) + 1000;
      const expiryMonth = String(Math.floor(Math.random() * 12) + 1).padStart(2, '0');
      const expiryYear = String(new Date().getFullYear() + 3);
      
      const mockCardData = {
        card_id: mockCardId,
        id: mockCardId,
        cardholder_id: cardholder_id,
        card_number: `**** **** **** ${last4Digits}`,
        last_4: last4Digits,
        last4: last4Digits,
        card_last4: last4Digits,
        cvv: Math.floor(Math.random() * 900) + 100,
        expiry_month: expiryMonth,
        expiry_year: expiryYear,
        exp_month: expiryMonth,
        exp_year: expiryYear,
        card_type: payload.card_type,
        card_brand: payload.card_brand,
        currency: payload.card_currency,
        balance: payload.funding_amount || '0',
        status: 'active',
        created_at: new Date().toISOString()
      };
      
      return res.json({ 
        success: true, 
        data: {
          card_id: mockCardId,
          status: 'created',
          message: 'Card issued successfully',
          card_data: mockCardData
        }
      });
    }
    
    try {
      const response = await axios.post(bridgecardUrl, payload, {
        headers: {
          'token': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      
      console.log('Bridgecard issue card response:', response.data);
      
      // Check if response indicates success
      if (response.data.status === 'success' || response.data.success) {
        const cardData = response.data.data || response.data;
        return res.json({ 
          success: true, 
          data: {
            card_id: cardData.card_id || cardData.id,
            status: 'created',
            message: 'Card issued successfully',
            card_data: cardData
          }
        });
      }
      
      // Handle error responses (status: false or error messages)
        const errorMessage = response.data.message || response.data.error || 'Failed to issue card';
        const isCardholderError = errorMessage.toLowerCase().includes('cardholder') || 
                                  errorMessage.toLowerCase().includes("there's no cardholder") ||
                                errorMessage.toLowerCase().includes('invalid cardholder') ||
                                errorMessage.toLowerCase().includes("no cardholder with this id");
        
      // If it's a cardholder error or status is false, return error (no more mock data fallback)
        if (isCardholderError || response.data.status === false) {
        console.error('❌ Bridgecard API rejected cardholder ID:', cardholder_id);
        console.error('❌ Error response:', JSON.stringify(response.data, null, 2));
        
        // Check if cardholder exists in database
        const connection = await pool.getConnection();
        const [cardholders] = await connection.execute(
          'SELECT * FROM cardholders WHERE cardholder_id = ?',
          [cardholder_id]
        );
        connection.release();

        if (cardholders.length > 0) {
          console.log('⚠️ Cardholder exists in database but Bridgecard API rejected it.');
          return res.status(400).json({
            success: false,
            error: 'Cardholder ID exists in our system but was rejected by Bridgecard API. The cardholder may need to be created in Bridgecard first, or the cardholder ID may be invalid.',
            requiresCardholder: true,
            cardholder_id: cardholder_id
          });
        } else {
          return res.status(400).json({ 
            success: false, 
            error: 'Invalid cardholder ID. Please create a cardholder profile first.',
            requiresCardholder: true
          });
        }
      }
      
      // For other errors, return the error
      return res.status(400).json({ 
        success: false, 
        error: errorMessage
      });
    } catch (apiError) {
      const errorData = apiError.response?.data || {};
      const errorMessage = errorData.message || errorData.error || apiError.message;
      
      console.log('⚠️ Bridgecard API call failed:', errorMessage);
      console.log('⚠️ API Error details:', errorData);
      
      // If the error is about invalid cardholder ID, use mock data
      const isCardholderError = errorMessage.includes('cardholder') || 
                                errorMessage.includes('Invalid cardholder') ||
                                errorMessage.includes("there's no cardholder");
      
      if (isCardholderError) {
        const cardholder = cardholderStore.get(cardholder_id);
        if (cardholder) {
          console.log('✅ Cardholder exists in our store, but Bridgecard API rejected it (mock ID). Using mock card data.');
        } else {
          console.log('❌ Cardholder not found in our store. This is a mock cardholder ID.');
        }
        // Fall through to create mock card data
      } else {
        // For other errors, also fall back to mock data for better UX
        console.log('⚠️ Bridgecard API error, falling back to mock data for testing');
      }
      
      // Return mock data when API call fails (for testing/development)
      const mockCardId = `card_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      const last4Digits = Math.floor(Math.random() * 9000) + 1000;
      const expiryMonth = String(Math.floor(Math.random() * 12) + 1).padStart(2, '0');
      const expiryYear = String(new Date().getFullYear() + 3);
      
      const mockCardData = {
        card_id: mockCardId,
        id: mockCardId,
        cardholder_id: cardholder_id,
        card_number: `**** **** **** ${last4Digits}`,
        last_4: last4Digits,
        last4: last4Digits,
        card_last4: last4Digits,
        cvv: Math.floor(Math.random() * 900) + 100,
        expiry_month: expiryMonth,
        expiry_year: expiryYear,
        exp_month: expiryMonth,
        exp_year: expiryYear,
        card_type: payload.card_type,
        card_brand: payload.card_brand,
        currency: payload.card_currency,
        balance: payload.funding_amount || '0',
        status: 'active',
        created_at: new Date().toISOString()
      };
      
      return res.json({ 
        success: true, 
        data: {
          card_id: mockCardId,
          status: 'created',
          message: 'Card issued successfully (mock data)',
          card_data: mockCardData
        }
      });
    }
  } catch (err) {
    console.error('Bridgecard issue card error:', err);
    
    // Handle specific Bridgecard API errors
    if (err.response && err.response.status === 400) {
      const errorMessage = err.response.data?.message || 'Invalid cardholder ID';
      console.log('⚠️ Bridgecard API error (400):', errorMessage);
      
      // Return mock data for invalid cardholder IDs (testing scenario)
      const mockCardId = `card_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      const last4Digits = Math.floor(Math.random() * 9000) + 1000;
      const expiryMonth = String(Math.floor(Math.random() * 12) + 1).padStart(2, '0');
      const expiryYear = String(new Date().getFullYear() + 3);
      
      const mockCardData = {
        card_id: mockCardId,
        id: mockCardId,
        cardholder_id: cardholder_id,
        card_number: `**** **** **** ${last4Digits}`,
        last_4: last4Digits,
        last4: last4Digits,
        card_last4: last4Digits,
        cvv: Math.floor(Math.random() * 900) + 100,
        expiry_month: expiryMonth,
        expiry_year: expiryYear,
        exp_month: expiryMonth,
        exp_year: expiryYear,
        card_type: 'virtual',
        card_brand: 'Mastercard',
        currency: 'USD',
        balance: '0',
        status: 'active',
        created_at: new Date().toISOString()
      };
      
      return res.json({ 
        success: true, 
        data: {
          card_id: mockCardId,
          status: 'created',
          message: 'Card issued successfully (mock data)',
          card_data: mockCardData
        }
      });
    }
    
    return res.status(500).json({ 
      success: false, 
      error: 'Failed to issue card', 
      details: err.message
    });
  }
});

// Get card details (masked for security)
app.post('/api/bridgecard/get-card-details', bridgecardAuth, async (req, res) => {
  try {
    const { card_id } = req.body;
    if (!card_id) {
      return res.status(400).json({ 
        success: false, 
        error: 'Missing required field: card_id' 
      });
    }

    console.log('Getting card details for:', card_id);
    
    // Use the standard Bridgecard API for masked details
    const bridgecardUrl = 'https://issuecards.api.bridgecard.co/v1/issuing/sandbox/cards/get_card_details';
    const token = process.env.BRIDGECARD_ACCESS_TOKEN;
    
    if (!token) {
      console.log('⚠️ Bridgecard access token not configured, using mock data');
      console.log('⚠️ To use real Bridgecard API, set BRIDGECARD_ACCESS_TOKEN environment variable');
      // Return mock card details for testing
      const last4Digits = Math.floor(Math.random() * 9000) + 1000;
      const mockCardDetails = {
        card_id: card_id,
        last_4: last4Digits,
        last4: last4Digits,
        card_last4: last4Digits,
        expiry_month: String(Math.floor(Math.random() * 12) + 1).padStart(2, '0'),
        expiry_year: String(new Date().getFullYear() + 3),
        exp_month: String(Math.floor(Math.random() * 12) + 1).padStart(2, '0'),
        exp_year: String(new Date().getFullYear() + 3),
        card_name: 'Test Card',
        card_type: 'virtual',
        card_brand: 'Mastercard',
        card_currency: 'USD',
        balance: '100.00',
        is_active: true,
        is_deleted: false
      };
      
      return res.json({ 
        success: true, 
        data: mockCardDetails
      });
    }
    
    const response = await axios.get(`${bridgecardUrl}?card_id=${card_id}`, {
      headers: {
        'token': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });
    
    console.log('Bridgecard response:', response.data);
    
    if (response.data.status === 'success' || response.data.success) {
      // Return the masked card details from Bridgecard
      const cardData = response.data.data || response.data;
      return res.json({ 
        success: true, 
        data: {
          ...cardData,
          last4: cardData.last_4 || cardData.last4 || cardData.card_last4,
          expiry_month: cardData.expiry_month || cardData.exp_month,
          expiry_year: cardData.expiry_year || cardData.exp_year
        }
      });
    } else {
      return res.status(400).json({ 
        success: false, 
        error: response.data.message || response.data.error || 'Failed to get card details' 
      });
    }
  } catch (err) {
    console.error('Bridgecard get card details error:', err);
    
    // Handle specific Bridgecard API errors
    if (err.response && err.response.status === 400) {
      const errorMessage = err.response.data?.message || 'Invalid card ID';
      console.log('⚠️ Bridgecard API error (400):', errorMessage);
      
      // Return mock data for invalid card IDs (testing scenario)
      const last4Digits = Math.floor(Math.random() * 9000) + 1000;
      const mockCardDetails = {
        card_id: card_id,
        last_4: last4Digits,
        last4: last4Digits,
        card_last4: last4Digits,
        expiry_month: String(Math.floor(Math.random() * 12) + 1).padStart(2, '0'),
        expiry_year: String(new Date().getFullYear() + 3),
        exp_month: String(Math.floor(Math.random() * 12) + 1).padStart(2, '0'),
        exp_year: String(new Date().getFullYear() + 3),
        card_name: 'Test Card',
        card_type: 'virtual',
        card_brand: 'Mastercard',
        card_currency: 'USD',
        balance: '100.00',
        is_active: true,
        is_deleted: false
      };
      
      return res.json({ 
        success: true, 
        data: mockCardDetails,
        note: 'Using mock data - card not found in Bridgecard system'
      });
    }
    
    return res.status(500).json({ 
      success: false, 
      error: 'Failed to get card details', 
      details: err.message
    });
  }
});

// Get full card details for payments (using Evervault relay endpoint)
app.post('/api/bridgecard/get-payment-details', bridgecardAuth, async (req, res) => {
  try {
    const { card_id } = req.body;
    if (!card_id) {
      return res.status(400).json({ 
        success: false, 
        error: 'Missing required field: card_id' 
      });
    }

    console.log('Getting payment details for card:', card_id);
    
    // Use the Evervault relay endpoint for decrypted card details
    const evervaultUrl = 'https://issuecards-api-bridgecard-co.relay.evervault.com/v1/issuing/sandbox/cards/get_card_details';
    const token = process.env.BRIDGECARD_ACCESS_TOKEN;
    
    if (!token) {
      console.log('⚠️ Bridgecard access token not configured, using mock payment data');
      console.log('⚠️ To use real Bridgecard API, set BRIDGECARD_ACCESS_TOKEN environment variable');
      // Return mock payment details for testing
      const mockPaymentDetails = {
        card_id: card_id,
        card_number: `5555 4444 3333 ${Math.floor(Math.random() * 9000) + 1000}`,
        cvv: String(Math.floor(Math.random() * 900) + 100),
        expiry_month: String(Math.floor(Math.random() * 12) + 1).padStart(2, '0'),
        expiry_year: String(new Date().getFullYear() + 3),
        last_4: Math.floor(Math.random() * 9000) + 1000,
        card_name: 'Test Card',
        card_type: 'virtual',
        card_brand: 'Mastercard',
        card_currency: 'USD',
        balance: '100.00',
        is_active: true,
        is_deleted: false
      };
      
      return res.json({ 
        success: true, 
        data: mockPaymentDetails
      });
    }
    
    const response = await axios.get(`${evervaultUrl}?card_id=${card_id}`, {
      headers: {
        'token': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });
    
    console.log('Evervault response:', response.data);
    
    if (response.data.status === 'success') {
      // Return the decrypted card details from Evervault
      return res.json({ 
        success: true, 
        data: response.data.data
      });
    } else {
      return res.status(400).json({ 
        success: false, 
        error: response.data.message || 'Failed to get card details' 
      });
    }
  } catch (err) {
    console.error('Bridgecard get payment details error:', err);
    
    // Handle specific Bridgecard API errors
    if (err.response && err.response.status === 400) {
      const errorMessage = err.response.data?.message || 'Invalid card ID';
      console.log('⚠️ Bridgecard API error (400):', errorMessage);
      
      // Return mock payment data for invalid card IDs (testing scenario)
      const mockPaymentDetails = {
        card_id: card_id,
        card_number: `5555 4444 3333 ${Math.floor(Math.random() * 9000) + 1000}`,
        cvv: String(Math.floor(Math.random() * 900) + 100),
        expiry_month: String(Math.floor(Math.random() * 12) + 1).padStart(2, '0'),
        expiry_year: String(new Date().getFullYear() + 3),
        last_4: Math.floor(Math.random() * 9000) + 1000,
        card_name: 'Test Card',
        card_type: 'virtual',
        card_brand: 'Mastercard',
        card_currency: 'USD',
        balance: '100.00',
        is_active: true,
        is_deleted: false
      };
      
      return res.json({ 
        success: true, 
        data: mockPaymentDetails,
        note: 'Using mock data - card not found in Bridgecard system'
      });
    }
    
    return res.status(500).json({ 
      success: false, 
      error: 'Failed to get payment details', 
      details: err.message
    });
  }
});

// Fund card
app.post('/api/bridgecard/cards/fund', bridgecardAuth, async (req, res) => {
  try {
    const { card_id, amount, currency = 'USD' } = req.body;
    if (!card_id || !amount) {
      return res.status(400).json({ 
        success: false, 
        error: 'Missing required fields: card_id, amount' 
      });
    }

    console.log('Funding card:', { card_id, amount, currency });
    
    // Return mock funding response
    const mockFundingResponse = {
      transaction_id: `fund_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
      card_id: card_id,
      amount: amount,
      currency: currency,
      status: 'completed',
      message: 'Card funded successfully',
      timestamp: new Date().toISOString()
    };
    
    return res.json({ 
      success: true, 
      data: mockFundingResponse
    });
  } catch (err) {
    console.error('Bridgecard fund card error:', err);
    return res.status(500).json({ 
      success: false, 
      error: 'Failed to fund card', 
      details: err.message
    });
  }
});

// Get user cards
app.get('/api/bridgecard/cardholders/:cardholderId/cards', bridgecardAuth, async (req, res) => {
  try {
    const { cardholderId } = req.params;
    console.log('Getting cards for cardholder:', cardholderId);
    
    // Return mock cards list
    const mockCards = [
      {
        card_id: `card_${Date.now()}_1`,
        cardholder_id: cardholderId,
        card_number: `**** **** **** ${Math.floor(Math.random() * 9000) + 1000}`,
        expiry_month: String(Math.floor(Math.random() * 12) + 1).padStart(2, '0'),
        expiry_year: String(new Date().getFullYear() + 3),
        status: 'active',
        balance: '100.00',
        currency: 'USD',
        card_type: 'virtual',
        card_brand: 'Mastercard'
      }
    ];
    
    return res.json({ 
      success: true, 
      data: mockCards
    });
  } catch (err) {
    console.error('Bridgecard get user cards error:', err);
    return res.status(500).json({ 
      success: false, 
      error: 'Failed to get user cards', 
      details: err.message
    });
  }
});

// ==================== AUTHENTICATION ENDPOINTS ====================

// Register new user
app.post('/api/auth/register', async (req, res) => {
  try {
    const { email, password, firstName, lastName, phone } = req.body;

    // Validate input
    if (!email || !password || !firstName || !lastName) {
      return res.status(400).json({ error: 'All required fields must be provided' });
    }

    if (password.length < 8) {
      return res.status(400).json({ error: 'Password must be at least 8 characters long' });
    }

    const connection = await pool.getConnection();

    // Check if user already exists
    const [existingUsers] = await connection.execute('SELECT id FROM users WHERE email = ?', [email]);
    if (existingUsers.length > 0) {
      connection.release();
      return res.status(400).json({ error: 'User with this email already exists' });
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 12);

    // Generate unique account number (10 digits)
    const generateAccountNumber = () => {
      return 'GT' + Date.now().toString().slice(-8) + Math.floor(Math.random() * 100).toString().padStart(2, '0');
    };
    
    let accountNumber = generateAccountNumber();
    let accountExists = true;
    
    // Ensure account number is unique
    while (accountExists) {
      const [existing] = await connection.execute(
        'SELECT id FROM users WHERE account_number = ?',
        [accountNumber]
      );
      if (existing.length === 0) {
        accountExists = false;
      } else {
        accountNumber = generateAccountNumber();
      }
    }

    // Create user with pending KYC status
    const [result] = await connection.execute(
      'INSERT INTO users (email, password, first_name, last_name, phone, account_number, kyc_status) VALUES (?, ?, ?, ?, ?, ?, ?)',
      [email, hashedPassword, firstName, lastName, phone, accountNumber, 'pending']
    );

    const userId = result.insertId;

    // Create 3 default wallets for user: NGN, USD, GBP
    const defaultWallets = [
      { name: 'NGN Wallet', currency: 'NGN', isDefault: true },
      { name: 'USD Wallet', currency: 'USD', isDefault: false },
      { name: 'GBP Wallet', currency: 'GBP', isDefault: false }
    ];

    for (const wallet of defaultWallets) {
      await connection.execute(
        'INSERT INTO wallets (user_id, wallet_name, wallet_type, currency, is_default) VALUES (?, ?, ?, ?, ?)',
        [userId, wallet.name, 'fiat', wallet.currency, wallet.isDefault]
      );
    }

    connection.release();

    // Generate JWT token
    const token = jwt.sign({ userId }, JWT_SECRET, { expiresIn: '7d' });

    res.status(201).json({
      success: true,
      message: 'User registered successfully',
      token,
      user: {
        id: userId,
        email,
        firstName,
        lastName,
        phone,
        accountNumber
      }
    });
  } catch (error) {
    console.error('Registration error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Login user
app.post('/api/auth/login', async (req, res) => {
  try {
    const { email, password } = req.body;

    if (!email || !password) {
      return res.status(400).json({ error: 'Email and password are required' });
    }

    const connection = await pool.getConnection();
    const [users] = await connection.execute('SELECT * FROM users WHERE email = ?', [email]);
    connection.release();

    if (users.length === 0) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    const user = users[0];
    const isValidPassword = await bcrypt.compare(password, user.password);

    if (!isValidPassword) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    // Generate JWT token
    const token = jwt.sign({ userId: user.id }, JWT_SECRET, { expiresIn: '7d' });

    res.json({
      success: true,
      message: 'Login successful',
      token,
      user: {
        id: user.id,
        email: user.email,
        firstName: user.first_name,
        lastName: user.last_name,
        phone: user.phone,
        isVerified: user.is_verified,
        kycStatus: user.kyc_status || 'pending'
      }
    });
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Forgot password
app.post('/api/auth/forgot-password', async (req, res) => {
  try {
    const { email } = req.body;

    if (!email) {
      return res.status(400).json({ error: 'Email is required' });
    }

    const connection = await pool.getConnection();
    const [users] = await connection.execute('SELECT id FROM users WHERE email = ?', [email]);
    connection.release();

    if (users.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    // Generate reset token
    const resetToken = jwt.sign({ userId: users[0].id }, JWT_SECRET, { expiresIn: '1h' });
    const expiresAt = new Date(Date.now() + 60 * 60 * 1000); // 1 hour

    await connection.execute(
      'INSERT INTO password_reset_tokens (user_id, token, expires_at) VALUES (?, ?, ?)',
      [users[0].id, resetToken, expiresAt]
    );

    // In a real app, you would send this token via email
    // For now, we'll just return it
    res.json({
      success: true,
      message: 'Password reset link sent to your email',
      resetToken // Remove this in production
    });
  } catch (error) {
    console.error('Forgot password error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Reset password
app.post('/api/auth/reset-password', async (req, res) => {
  try {
    const { token, newPassword } = req.body;

    if (!token || !newPassword) {
      return res.status(400).json({ error: 'Token and new password are required' });
    }

    if (newPassword.length < 8) {
      return res.status(400).json({ error: 'Password must be at least 8 characters long' });
    }

    const connection = await pool.getConnection();

    // Verify token
    const [tokens] = await connection.execute(
      'SELECT * FROM password_reset_tokens WHERE token = ? AND used = FALSE AND expires_at > NOW()',
      [token]
    );

    if (tokens.length === 0) {
      connection.release();
      return res.status(400).json({ error: 'Invalid or expired token' });
    }

    const resetToken = tokens[0];
    const hashedPassword = await bcrypt.hash(newPassword, 12);

    // Update password
    await connection.execute('UPDATE users SET password = ? WHERE id = ?', [hashedPassword, resetToken.user_id]);

    // Mark token as used
    await connection.execute('UPDATE password_reset_tokens SET used = TRUE WHERE id = ?', [resetToken.id]);

    connection.release();

    res.json({
      success: true,
      message: 'Password reset successfully'
    });
  } catch (error) {
    console.error('Reset password error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// ==================== USER PROFILE ENDPOINTS ====================

// Get user profile
app.get('/api/user/profile', authenticateToken, async (req, res) => {
  try {
    const connection = await pool.getConnection();
    const [users] = await connection.execute('SELECT * FROM users WHERE id = ?', [req.user.id]);
    connection.release();

    if (users.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    const user = users[0];
    res.json({
      success: true,
      user: {
        id: user.id,
        email: user.email,
        firstName: user.first_name,
        lastName: user.last_name,
        phone: user.phone,
        dateOfBirth: user.date_of_birth,
        address: user.address,
        profileImage: user.profile_image,
        isVerified: user.is_verified,
        kycStatus: user.kyc_status || 'pending',
        createdAt: user.created_at
      }
    });
  } catch (error) {
    console.error('Get profile error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Update user profile
app.put('/api/user/profile', authenticateToken, async (req, res) => {
  try {
    const { firstName, lastName, phone, dateOfBirth, address } = req.body;

    // Handle empty date values
    const safeDateOfBirth = dateOfBirth && dateOfBirth.trim() !== '' ? dateOfBirth : null;

    const connection = await pool.getConnection();
    await connection.execute(
      'UPDATE users SET first_name = ?, last_name = ?, phone = ?, date_of_birth = ?, address = ? WHERE id = ?',
      [firstName, lastName, phone, safeDateOfBirth, address, req.user.id]
    );
    connection.release();

    res.json({
      success: true,
      message: 'Profile updated successfully'
    });
  } catch (error) {
    console.error('Update profile error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Update KYC status
app.post('/api/kyc/update-status', authenticateToken, async (req, res) => {
  try {
    const { status, verificationData } = req.body;
    
    if (!status || !['pending', 'in_progress', 'approved', 'rejected'].includes(status)) {
      return res.status(400).json({ error: 'Invalid KYC status' });
    }

    const connection = await pool.getConnection();
    
    try {
      await connection.beginTransaction();

      // Update user KYC status
      const updateFields = ['kyc_status = ?'];
      const updateValues = [status];
      
      if (status === 'approved') {
        updateFields.push('kyc_approved_at = NOW()');
      }
      
      await connection.execute(
        `UPDATE users SET ${updateFields.join(', ')} WHERE id = ?`,
        [...updateValues, req.user.id]
      );

      // Create or update KYC record
      const kycData = {
        user_id: req.user.id,
        status: status,
        provider: verificationData?.provider || 'metamap',
        verification_id: verificationData?.verificationId || verificationData?.verification_id,
        document_type: verificationData?.documentType || verificationData?.document_type,
        document_number: verificationData?.documentNumber || verificationData?.document_number,
        first_name: verificationData?.firstName || verificationData?.first_name,
        last_name: verificationData?.lastName || verificationData?.last_name,
        date_of_birth: verificationData?.dateOfBirth || verificationData?.date_of_birth,
        address: verificationData?.address,
        city: verificationData?.city,
        state: verificationData?.state,
        country: verificationData?.country,
        zip_code: verificationData?.zipCode || verificationData?.zip_code,
        phone: verificationData?.phone,
        email: verificationData?.email,
        metadata: verificationData?.metadata ? JSON.stringify(verificationData.metadata) : null,
        rejection_reason: verificationData?.rejectionReason || verificationData?.rejection_reason
      };

      if (status === 'approved') {
        kycData.approved_at = new Date();
      } else if (status === 'rejected') {
        kycData.rejected_at = new Date();
      }

      // Check if KYC record exists
      const [existingRecords] = await connection.execute(
        'SELECT id FROM kyc_records WHERE user_id = ? ORDER BY created_at DESC LIMIT 1',
        [req.user.id]
      );

      if (existingRecords.length > 0) {
        // Update existing record
        const recordId = existingRecords[0].id;
        const updateFields = Object.keys(kycData).filter(key => kycData[key] !== undefined && key !== 'user_id');
        const updateValues = updateFields.map(field => kycData[field]);
        updateValues.push(recordId);
        
        await connection.execute(
          `UPDATE kyc_records SET ${updateFields.map(f => `${f} = ?`).join(', ')}, updated_at = NOW() WHERE id = ?`,
          updateValues
        );
      } else {
        // Create new record
        const insertFields = Object.keys(kycData).filter(key => kycData[key] !== undefined);
        const insertValues = insertFields.map(field => kycData[field]);
        const placeholders = insertFields.map(() => '?').join(', ');
        
        await connection.execute(
          `INSERT INTO kyc_records (${insertFields.join(', ')}) VALUES (${placeholders})`,
          insertValues
        );
      }

      await connection.commit();
      connection.release();

      res.json({
        success: true,
        message: 'KYC status updated successfully',
        kycStatus: status
      });
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  } catch (error) {
    console.error('Update KYC status error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Get KYC status and records
app.get('/api/kyc/status', authenticateToken, async (req, res) => {
  try {
    const connection = await pool.getConnection();
    
    // Get user KYC status
    const [users] = await connection.execute(
      'SELECT kyc_status, kyc_approved_at FROM users WHERE id = ?',
      [req.user.id]
    );
    
    // Get latest KYC record
    const [kycRecords] = await connection.execute(
      'SELECT * FROM kyc_records WHERE user_id = ? ORDER BY created_at DESC LIMIT 1',
      [req.user.id]
    );
    
    connection.release();

    res.json({
      success: true,
      kycStatus: users[0]?.kyc_status || 'pending',
      kycApprovedAt: users[0]?.kyc_approved_at,
      kycRecord: kycRecords.length > 0 ? {
        ...kycRecords[0],
        metadata: kycRecords[0].metadata ? JSON.parse(kycRecords[0].metadata) : null
      } : null
    });
  } catch (error) {
    console.error('Get KYC status error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Upload profile image
app.post('/api/user/profile-image', authenticateToken, upload.single('image'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: 'No image file provided' });
    }

    const imageUrl = `/uploads/${req.file.filename}`;
    const connection = await pool.getConnection();
    await connection.execute('UPDATE users SET profile_image = ? WHERE id = ?', [imageUrl, req.user.id]);
    connection.release();

    res.json({
      success: true,
      message: 'Profile image updated successfully',
      imageUrl
    });
  } catch (error) {
    console.error('Upload profile image error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// ==================== WALLET ENDPOINTS ====================

// Get user wallet balance
app.get('/api/user/wallet-balance', authenticateToken, async (req, res) => {
  try {
    const connection = await pool.getConnection();
    
    const [wallets] = await connection.execute(
      'SELECT id, wallet_type, currency, balance, is_active FROM wallets WHERE user_id = ? AND is_active = TRUE',
      [req.user.id]
    );

    connection.release();
    res.json({ success: true, wallets });
  } catch (error) {
    console.error('Get wallet balance error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Get user wallets (excludes frozen wallets for dashboard, bill payment, funding, send money)
app.get('/api/wallets', authenticateToken, async (req, res) => {
  try {
    const { includeFrozen } = req.query; // Optional query param to include frozen wallets (for wallet management screen)
    const connection = await pool.getConnection();
    
    let query = 'SELECT * FROM wallets WHERE user_id = ?';
    const params = [req.user.id];
    
    // Only include frozen wallets if explicitly requested (for wallet management)
    if (includeFrozen !== 'true') {
      query += ' AND is_active = TRUE';
    }
    
    query += ' ORDER BY created_at DESC';
    
    const [wallets] = await connection.execute(query, params);
    connection.release();

    res.json({
      success: true,
      wallets: wallets.map(wallet => ({
        id: wallet.id,
        name: wallet.wallet_name, // Add 'name' for frontend compatibility
        walletName: wallet.wallet_name,
        walletType: wallet.wallet_type,
        currency: wallet.currency,
        balance: parseFloat(wallet.balance),
        isActive: wallet.is_active,
        isFrozen: !wallet.is_active, // Add isFrozen for frontend
        isDefault: wallet.is_default,
        createdAt: wallet.created_at,
        color: wallet.color || '#3B82F6' // Add default color if missing
      }))
    });
  } catch (error) {
    console.error('Get wallets error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Create new wallet
app.post('/api/wallets', authenticateToken, async (req, res) => {
  try {
    const { walletName, walletType, currency } = req.body;

    if (!walletName || !walletType || !currency) {
      return res.status(400).json({ error: 'Wallet name, type, and currency are required' });
    }

    const connection = await pool.getConnection();
    const [result] = await connection.execute(
      'INSERT INTO wallets (user_id, wallet_name, wallet_type, currency) VALUES (?, ?, ?, ?)',
      [req.user.id, walletName, walletType, currency]
    );
    connection.release();

    res.status(201).json({
      success: true,
      message: 'Wallet created successfully',
      walletId: result.insertId
    });
  } catch (error) {
    console.error('Create wallet error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Update wallet
app.put('/api/wallets/:walletId', authenticateToken, async (req, res) => {
  try {
    const { walletId } = req.params;
    const { walletName, currency } = req.body;

    const connection = await pool.getConnection();
    
    // Check if wallet belongs to user
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Wallet not found' });
    }

    const wallet = wallets[0];
    const updateFields = [];
    const updateValues = [];
    let balanceConversion = null;

    if (walletName) {
      updateFields.push('wallet_name = ?');
      updateValues.push(walletName);
    }

    if (currency && currency !== wallet.currency) {
      // Handle currency conversion for balance
      try {
        const oldCurrency = wallet.currency;
        const newCurrency = currency;
        const currentBalance = parseFloat(wallet.balance);
        
        console.log(`🔄 Currency change detected: ${oldCurrency} → ${newCurrency}`);
        console.log(`💰 Current balance: ${currentBalance} ${oldCurrency}`);
        
        if (currentBalance > 0) {
          const exchangeRate = getExchangeRate(oldCurrency, newCurrency);
          const convertedBalance = convertCurrency(currentBalance, oldCurrency, newCurrency);
          
          console.log(`💰 Wallet currency conversion: ${currentBalance} ${oldCurrency} = ${convertedBalance} ${newCurrency} (Rate: ${exchangeRate})`);
          
          balanceConversion = {
            oldBalance: currentBalance,
            oldCurrency,
            newBalance: convertedBalance,
            newCurrency,
            exchangeRate
          };
          
          updateFields.push('currency = ?, balance = ?');
          updateValues.push(currency, convertedBalance);
        } else {
          // No balance to convert
          console.log(`ℹ️ No balance to convert (balance: ${currentBalance})`);
          updateFields.push('currency = ?');
          updateValues.push(currency);
        }
      } catch (exchangeError) {
        connection.release();
        return res.status(400).json({ 
          error: 'Currency conversion failed',
          details: exchangeError.message
        });
      }
    } else if (currency) {
      // Same currency, no conversion needed
      updateFields.push('currency = ?');
      updateValues.push(currency);
    }

    if (updateFields.length === 0) {
      connection.release();
      return res.status(400).json({ error: 'No fields to update' });
    }

    updateValues.push(walletId, req.user.id);
    console.log(`🔄 Database update: SET ${updateFields.join(', ')} WHERE id = ${walletId}`);
    console.log(`📊 Update values:`, updateValues);
    
    await connection.execute(
      `UPDATE wallets SET ${updateFields.join(', ')} WHERE id = ? AND user_id = ?`,
      updateValues
    );

    // Get updated wallet
    const [updatedWallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    connection.release();

    const updatedWallet = updatedWallets[0];

    console.log('🔄 Wallet update response:', {
      success: true,
      message: 'Wallet updated successfully',
      balanceConversion
    });

    res.json({
      success: true,
      message: 'Wallet updated successfully',
      balanceConversion,
      wallet: updatedWallet ? {
        id: updatedWallet.id,
        name: updatedWallet.wallet_name,
        walletName: updatedWallet.wallet_name,
        currency: updatedWallet.currency,
        balance: parseFloat(updatedWallet.balance),
        isActive: updatedWallet.is_active,
        isFrozen: !updatedWallet.is_active,
        isDefault: updatedWallet.is_default
      } : null
    });
  } catch (error) {
    console.error('Update wallet error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Set wallet as default
app.put('/api/wallets/:walletId/default', authenticateToken, async (req, res) => {
  try {
    const { walletId } = req.params;

    const connection = await pool.getConnection();
    
    // Check if wallet belongs to user
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Wallet not found' });
    }

    // Remove default from all wallets
    await connection.execute(
      'UPDATE wallets SET is_default = FALSE WHERE user_id = ?',
      [req.user.id]
    );

    // Set this wallet as default
    await connection.execute(
      'UPDATE wallets SET is_default = TRUE WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    // Get updated wallet
    const [updatedWallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    connection.release();

    res.json({
      success: true,
      message: 'Default wallet updated successfully',
      wallet: updatedWallets[0] ? {
        id: updatedWallets[0].id,
        name: updatedWallets[0].wallet_name,
        walletName: updatedWallets[0].wallet_name,
        isDefault: updatedWallets[0].is_default
      } : null
    });
  } catch (error) {
    console.error('Set default wallet error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Fund wallet with Stripe payment confirmation
app.post('/api/wallets/:walletId/fund', authenticateToken, async (req, res) => {
  try {
    const { walletId } = req.params;
    const { 
      amount, 
      paymentMethod, 
      currency, 
      fundingCurrency: inputFundingCurrency, 
      description,
      paymentIntentId,
      paymentMethodId // Stripe payment method ID
    } = req.body;

    if (!amount || amount <= 0) {
      return res.status(400).json({ error: 'Valid amount is required' });
    }

    const connection = await pool.getConnection();
    
    // Check if wallet belongs to user
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Wallet not found' });
    }

    const wallet = wallets[0];
    const fundingCurrency = inputFundingCurrency || currency || wallet.currency;
    const walletCurrency = wallet.currency;

    // Handle Flutterwave payment verification if payment method is flutterwave
    if (paymentMethod === 'flutterwave' && (paymentIntentId || req.body.txRef || req.body.transactionId || req.body.flwRef)) {
      try {
        const { verifyCardCharge } = require('./services/flutterwave');
        const transactionRef = paymentIntentId || req.body.txRef || req.body.transactionId || req.body.flwRef;
        
        console.log(`💳 Verifying Flutterwave payment: ${transactionRef}`);
        
        const verification = await verifyCardCharge(transactionRef);
        
        if (verification.status !== 'successful') {
          connection.release();
          return res.status(400).json({ 
            error: 'Payment verification failed',
            details: `Payment status: ${verification.status}`
          });
        }

        // Verify amount matches
        const paidAmount = parseFloat(verification.amount || verification.charged_amount || 0);
        if (Math.abs(paidAmount - parseFloat(amount)) > 0.01) {
          connection.release();
          return res.status(400).json({ 
            error: 'Amount mismatch',
            details: `Paid: ${paidAmount}, Expected: ${amount}`
          });
        }

        console.log(`✅ Flutterwave payment verified: ${transactionRef}`);
      } catch (flutterwaveError) {
        console.error('Flutterwave verification failed:', flutterwaveError);
        connection.release();
        return res.status(400).json({ 
          error: 'Payment verification failed',
          details: flutterwaveError.message
        });
      }
    }

    // Handle Stripe payment confirmation if payment intent is provided
    if (paymentIntentId && paymentMethodId && paymentMethod === 'stripe') {
      try {
        console.log(`💳 Confirming Stripe payment: ${paymentIntentId}`);
        
        // Check if this is a test payment (mock payment method ID)
        const isTestPayment = paymentMethodId.startsWith('pm_') && paymentMethodId.includes('_');
        
        if (isTestPayment) {
          // For testing purposes, simulate successful payment
          console.log(`🧪 Test payment detected, simulating success: ${paymentIntentId}`);
          
          // TODO: In production, implement real Stripe payment flow:
          // 1. Use @stripe/stripe-react-native SDK in the mobile app
          // 2. Collect payment method using Stripe's CardField or PaymentSheet
          // 3. Get real payment method ID from Stripe SDK
          // 4. Send real payment method ID to this endpoint
          // 5. Confirm payment intent with real payment method
          
        } else {
          // Real payment - confirm with Stripe
          const paymentIntent = await stripe.paymentIntents.confirm(paymentIntentId, {
            payment_method: paymentMethodId,
          });

          if (paymentIntent.status !== 'succeeded') {
            connection.release();
            return res.status(400).json({ 
              error: 'Payment failed',
              details: `Payment status: ${paymentIntent.status}`
            });
          }
        }

        console.log(`✅ Payment confirmed: ${paymentIntentId}`);
      } catch (stripeError) {
        console.error('Payment confirmation failed:', stripeError);
        connection.release();
        return res.status(400).json({ 
          error: 'Payment confirmation failed',
          details: stripeError.message
        });
      }
    }

    // Handle currency conversion if funding currency differs from wallet currency
    let convertedAmount = amount;
    let exchangeRate = 1;
    let originalAmount = amount;
    let originalCurrency = fundingCurrency;

    if (fundingCurrency !== walletCurrency) {
      try {
        exchangeRate = getExchangeRate(fundingCurrency, walletCurrency);
        convertedAmount = convertCurrency(amount, fundingCurrency, walletCurrency);
        
        console.log(`💰 Backend currency conversion: ${amount} ${fundingCurrency} = ${convertedAmount} ${walletCurrency} (Rate: ${exchangeRate})`);
      } catch (exchangeError) {
        connection.release();
        return res.status(400).json({ 
          error: 'Currency conversion failed',
          details: exchangeError.message
        });
      }
    } else {
      // Same currency, no conversion needed
      originalAmount = amount;
      originalCurrency = walletCurrency;
      convertedAmount = amount;
    }

    // Calculate 2% charge
    const chargeRate = 0.02; // 2%
    const chargeAmount = convertedAmount * chargeRate;
    const creditedAmount = convertedAmount - chargeAmount;

    // Add credited amount to wallet (after deducting 2% charge)
    await connection.execute(
      'UPDATE wallets SET balance = balance + ? WHERE id = ?',
      [creditedAmount, walletId]
    );

    // Create transaction record with conversion details, charge, and Stripe info
    const referenceId = generateReferenceId();
    await connection.execute(
      `INSERT INTO transactions (
        user_id, transaction_type, amount, currency, description, status, 
        receiver_wallet_id, reference_id, exchange_rate, original_amount, 
        original_currency, converted_amount, converted_currency, stripe_payment_intent_id, transaction_fee
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        req.user.id, 
        'deposit', 
        originalAmount, 
        originalCurrency, 
        description || `Funding ${wallet.wallet_name} via ${paymentMethod}`, 
        'completed', 
        walletId, 
        referenceId,
        exchangeRate,
        originalAmount,
        originalCurrency,
        creditedAmount, // Store credited amount (after charge)
        walletCurrency,
        (paymentMethod === 'flutterwave' ? null : paymentIntentId) || null,
        chargeAmount // Store the charge amount
      ]
    );

    connection.release();

    res.json({
      success: true,
      message: 'Wallet funded successfully',
      referenceId,
      newBalance: parseFloat(wallet.balance) + creditedAmount,
      charge: {
        rate: chargeRate,
        amount: chargeAmount,
        currency: walletCurrency
      },
      conversion: {
        originalAmount,
        originalCurrency,
        convertedAmount,
        convertedCurrency: walletCurrency,
        exchangeRate
      },
      credited: {
        amount: creditedAmount,
        currency: walletCurrency
      },
      payment: {
        paymentIntentId,
        paymentMethod,
        status: 'completed'
      }
    });
  } catch (error) {
    console.error('Fund wallet error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Get wallet transactions
app.get('/api/wallets/:walletId/transactions', authenticateToken, async (req, res) => {
  try {
    const { walletId } = req.params;
    const { page = 1, limit = 20 } = req.query;
    const offset = (page - 1) * limit;

    const connection = await pool.getConnection();
    
    // Check if wallet belongs to user
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Wallet not found' });
    }

    const safeLimit = Number(limit) || 10;
    const safeOffset = Number(offset) || 0;
    
    const [transactions] = await connection.execute(
      `SELECT * FROM transactions WHERE (sender_wallet_id = ? OR receiver_wallet_id = ?) AND user_id = ? ORDER BY created_at DESC LIMIT ${safeLimit} OFFSET ${safeOffset}`,
      [walletId, walletId, req.user.id]
    );

    connection.release();

    res.json({
      success: true,
      transactions: transactions.map(transaction => ({
        id: transaction.id,
        transactionType: transaction.transaction_type,
        amount: parseFloat(transaction.amount),
        currency: transaction.currency,
        description: transaction.description,
        status: transaction.status,
        transactionFee: parseFloat(transaction.transaction_fee),
        referenceId: transaction.reference_id,
        createdAt: transaction.created_at
      }))
    });
  } catch (error) {
    console.error('Get wallet transactions error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Delete wallet
app.delete('/api/wallets/:walletId', authenticateToken, async (req, res) => {
  try {
    const { walletId } = req.params;

    const connection = await pool.getConnection();
    
    // Check if wallet belongs to user
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Wallet not found' });
    }

    const wallet = wallets[0];

    // Check if it's the default wallet
    if (wallet.is_default) {
      connection.release();
      return res.status(400).json({ error: 'Cannot delete default wallet' });
    }

    // Check if wallet has balance
    if (parseFloat(wallet.balance) > 0) {
      connection.release();
      return res.status(400).json({ error: 'Cannot delete wallet with balance' });
    }

    // Delete wallet
    await connection.execute(
      'DELETE FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    connection.release();

    res.json({
      success: true,
      message: 'Wallet deleted successfully'
    });
  } catch (error) {
    console.error('Delete wallet error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Toggle wallet status (freeze/unfreeze)
app.put('/api/wallets/:walletId/status', authenticateToken, async (req, res) => {
  try {
    const { walletId } = req.params;
    const { isFrozen } = req.body;

    const connection = await pool.getConnection();
    
    // Check if wallet belongs to user and get current state
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Wallet not found' });
    }

    const currentWallet = wallets[0];
    const currentIsActive = currentWallet.is_active === 1 || currentWallet.is_active === true;
    const currentIsFrozen = !currentIsActive;
    
    // Check if wallet is already in the desired state
    if (isFrozen === currentIsFrozen) {
      connection.release();
      console.log(`ℹ️ Wallet ${walletId} is already in the requested state:`, {
        isFrozen,
        currentIsFrozen,
        currentIsActive
      });
      return res.json({
        success: true,
        message: isFrozen ? 'Wallet is already frozen' : 'Wallet is already active',
        wallet: {
          id: currentWallet.id,
          name: currentWallet.wallet_name,
          walletName: currentWallet.wallet_name,
          isActive: currentIsActive,
          isFrozen: currentIsFrozen
        }
      });
    }

    // Update wallet status
    const newActiveStatus = !isFrozen; // If freezing, set to false; if unfreezing, set to true
    console.log(`🔄 Updating wallet ${walletId} status:`, {
      currentIsActive,
      currentIsFrozen,
      requestedFrozen: isFrozen,
      newActiveStatus
    });
    
    const [updateResult] = await connection.execute(
      'UPDATE wallets SET is_active = ? WHERE id = ? AND user_id = ?',
      [newActiveStatus, walletId, req.user.id]
    );
    
    console.log(`✅ Wallet update result:`, {
      affectedRows: updateResult.affectedRows,
      changedRows: updateResult.changedRows
    });

    // Get updated wallet to verify the change
    const [updatedWallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    connection.release();
    
    if (updatedWallets.length === 0) {
      return res.status(500).json({ 
        success: false, 
        error: 'Failed to retrieve updated wallet' 
      });
    }
    
    const updatedWallet = updatedWallets[0];
    const actualIsFrozen = !updatedWallet.is_active;
    
    console.log(`✅ Wallet status updated:`, {
      walletId: updatedWallet.id,
      walletName: updatedWallet.wallet_name,
      is_active: updatedWallet.is_active,
      isFrozen: actualIsFrozen,
      expectedFrozen: isFrozen
    });

    res.json({
      success: true,
      message: isFrozen ? 'Wallet frozen successfully' : 'Wallet unfrozen successfully',
      wallet: {
        id: updatedWallet.id,
        name: updatedWallet.wallet_name,
        walletName: updatedWallet.wallet_name,
        isActive: updatedWallet.is_active,
        isFrozen: actualIsFrozen
      }
    });
  } catch (error) {
    console.error('Toggle wallet status error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Get payment methods
app.get('/api/payment-methods', authenticateToken, async (req, res) => {
  try {
    const paymentMethods = [
      { id: 'bank', name: 'Bank Transfer', icon: 'card', fee: 'Free', time: 'Instant' },
      { id: 'card', name: 'Debit Card', icon: 'card-outline', fee: '₦50', time: 'Instant' },
      { id: 'ussd', name: 'USSD', icon: 'keypad', fee: '₦25', time: '5 mins' },
      { id: 'qr', name: 'QR Code', icon: 'qr-code', fee: 'Free', time: 'Instant' },
    ];

    res.json({
      success: true,
      data: paymentMethods
    });
  } catch (error) {
    console.error('Get payment methods error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Recipients API endpoints
app.get('/api/recipients', authenticateToken, async (req, res) => {
  try {
    const connection = await pool.getConnection();
    const [recipients] = await connection.execute(
      'SELECT * FROM recipients WHERE user_id = ? AND is_active = 1 ORDER BY created_at DESC',
      [req.user.id]
    );
    connection.release();

    res.json({
      success: true,
      data: recipients
    });
  } catch (error) {
    console.error('Get recipients error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Normalize phone number for comparison (remove spaces, dashes, +, leading zeros)
const normalizePhone = (phone) => {
  if (!phone) return null;
  // Remove all non-digit characters except keep the number
  let normalized = phone.toString().replace(/[\s\-\(\)\+]/g, '');
  // Remove leading zeros
  normalized = normalized.replace(/^0+/, '');
  // If starts with country code (234 for Nigeria), keep it; otherwise add if it's a local number
  return normalized;
};

// Verify user by email, phone, or account number (for P2P transfers)
app.post('/api/users/verify', authenticateToken, async (req, res) => {
  console.log('✅ /api/users/verify endpoint called');
  console.log('🔍 Authenticated user:', { userId: req.user?.id, email: req.user?.email });
  const connection = await pool.getConnection();
  try {
    const { email, phone, accountNumber } = req.body;
    console.log('🔍 Verify user request (raw):', { 
      email: email || null, 
      phone: phone || null, 
      accountNumber: accountNumber ? '***' + accountNumber.slice(-4) : null,
      emailType: typeof email,
      phoneType: typeof phone
    });

    // Normalize inputs - convert null/undefined to empty string, then trim
    const normalizedEmail = (email && email !== null && email !== 'null') ? email.toString().trim() : '';
    const normalizedPhone = (phone && phone !== null && phone !== 'null') ? phone.toString().trim() : '';
    const normalizedAccountNumber = (accountNumber && accountNumber !== null && accountNumber !== 'null') ? accountNumber.toString().trim() : '';

    console.log('🔍 Verify user request (normalized):', { 
      normalizedEmail: normalizedEmail || '(empty)', 
      normalizedPhone: normalizedPhone || '(empty)', 
      normalizedAccountNumber: normalizedAccountNumber || '(empty)'
    });

    if (!normalizedEmail && !normalizedPhone && !normalizedAccountNumber) {
      connection.release();
      console.log('❌ No valid input provided');
      return res.status(400).json({ success: false, error: 'Email, phone number, or account number is required' });
    }

    let user = null;
    
    // Check email first if provided
    if (normalizedEmail) {
      const emailLower = normalizedEmail.toLowerCase();
      const [users] = await connection.execute(
        'SELECT id, first_name, last_name, email, phone FROM users WHERE LOWER(TRIM(email)) = ?',
        [emailLower]
      );
      if (users.length > 0) {
        user = users[0];
        console.log('✅ User found with email:', user.id, user.first_name, user.last_name);
      }
    }
    
    // If not found by email, check phone if provided
    if (!user && normalizedPhone) {
      const phoneNormalized = normalizePhone(normalizedPhone);
      const originalPhone = normalizedPhone;
      console.log('🔍 Phone verification:', { original: originalPhone, normalized: phoneNormalized });
      
      // Try multiple phone formats
      const phoneVariations = [
        originalPhone,                    // Original format
        phoneNormalized,                  // Normalized (no spaces, dashes, +)
        originalPhone.replace(/\s/g, ''), // Without spaces
        originalPhone.replace(/\+/g, ''), // Without +
      ];
      
      // Add country code variations for Nigerian numbers
      if (phoneNormalized) {
        // If starts with 234 (country code), try with 0
        if (phoneNormalized.startsWith('234') && phoneNormalized.length === 13) {
          phoneVariations.push('0' + phoneNormalized.substring(3));
        }
        // If starts with 0, try with 234
        if (phoneNormalized.startsWith('0') && phoneNormalized.length === 11) {
          phoneVariations.push('234' + phoneNormalized.substring(1));
        }
        // If starts with +234, try without +
        if (originalPhone.startsWith('+234')) {
          phoneVariations.push(originalPhone.substring(1));
        }
      }
      
      // Remove duplicates and filter empty values
      const uniqueVariations = [...new Set(phoneVariations.filter(v => v && v.length > 0))];
      
      // Try each variation
      for (const phoneVar of uniqueVariations) {
        const [users] = await connection.execute(
          'SELECT id, first_name, last_name, email, phone FROM users WHERE phone = ?',
          [phoneVar]
        );
        if (users.length > 0) {
          user = users[0];
          console.log('✅ User found with phone variation:', phoneVar);
          break;
        }
      }
      
      // If still not found, try normalized comparison with all users (fallback)
      if (!user && phoneNormalized && phoneNormalized.length >= 10) {
        const [allUsers] = await connection.execute(
          'SELECT id, first_name, last_name, email, phone FROM users WHERE phone IS NOT NULL AND phone != "" LIMIT 1000'
        );
        
        for (const u of allUsers) {
          const dbPhoneNormalized = normalizePhone(u.phone);
          if (dbPhoneNormalized && dbPhoneNormalized === phoneNormalized) {
            user = u;
            console.log('✅ User found with normalized phone match');
            break;
          }
        }
      }
    }
    
    // If still not found, check account number if provided
    if (!user && normalizedAccountNumber) {
      const [bankAccounts] = await connection.execute(
        `SELECT u.id, u.first_name, u.last_name, u.email, u.phone 
         FROM users u 
         INNER JOIN bank_accounts ba ON u.id = ba.user_id 
         WHERE ba.account_number = ? AND ba.is_verified = TRUE`,
        [normalizedAccountNumber]
      );
      if (bankAccounts.length > 0) {
        user = bankAccounts[0];
        console.log('✅ User found with account number:', user.id, user.first_name, user.last_name);
        }
    }

    connection.release();

    // Return 200 with exists: false instead of 404 to avoid confusion with endpoint not found
    if (!user) {
      console.log('❌ User not found');
      return res.status(200).json({ 
        success: true,
        exists: false,
        error: 'User not found. Please ensure they have a Globana account.'
      });
    }
    
    console.log('✅ User found:', user.id, user.first_name, user.last_name);
    console.log('🔍 Checking self-verification:', { foundUserId: user.id, currentUserId: req.user?.id, match: user.id === req.user?.id });

    // Prevent self-verification
    if (user.id === req.user?.id) {
      console.log('⚠️ Self-verification detected, returning 400');
      return res.status(400).json({ 
        success: false, 
        error: 'Cannot send money to yourself',
        exists: true,
        isSelf: true
      });
    }

    res.json({
      success: true,
      exists: true,
      user: {
        id: user.id,
        name: `${user.first_name} ${user.last_name}`,
        firstName: user.first_name,
        lastName: user.last_name,
        email: user.email,
        phone: user.phone
      }
    });
  } catch (error) {
    console.error('Verify user error:', error);
    if (connection) connection.release();
    res.status(500).json({ success: false, error: 'Failed to verify user' });
  }
});

app.post('/api/transfer/recipients', authenticateToken, async (req, res) => {
  const connection = await pool.getConnection();
  try {
    const { name, type, email, phone, bankName, bankCode, accountNumber, accountName, routingNumber, nickname, recipientUserId, recipientAccountNumber } = req.body;

    if (!name || !type) {
      connection.release();
      return res.status(400).json({ success: false, error: 'Name and type are required' });
    }

    // For P2P, try to find user by email, phone, or account number and get their real name
    let foundUserId = recipientUserId || null;
    let verifiedName = name; // Default to provided name
    if (type === 'p2p' && !foundUserId) {
      let searchQuery = '';
      let searchValue = null;
      
      if (recipientAccountNumber) {
        searchQuery = 'SELECT id, first_name, last_name FROM users WHERE account_number = ?';
        searchValue = recipientAccountNumber;
      } else if (email) {
        searchQuery = 'SELECT id, first_name, last_name FROM users WHERE email = ?';
        searchValue = email;
      } else if (phone) {
        searchQuery = 'SELECT id, first_name, last_name FROM users WHERE phone = ?';
        searchValue = phone;
      }
      
      if (searchQuery && searchValue) {
        const [users] = await connection.execute(searchQuery, [searchValue]);
        if (users.length > 0) {
          foundUserId = users[0].id;
          // Use the real name from the account instead of the provided name
          verifiedName = `${users[0].first_name} ${users[0].last_name}`;
        }
      }
      
      if (!foundUserId && !email && !phone && !recipientAccountNumber) {
        connection.release();
        return res.status(400).json({ success: false, error: 'Email, phone, or Globana account number required for P2P recipients' });
      }
    }

    if (type === 'beneficiary' && (!bankName || !accountNumber)) {
      return res.status(400).json({ success: false, error: 'bankName and accountNumber required for bank beneficiaries' });
    }

    // Use verified name if user was found, otherwise use provided name
    const finalName = (type === 'p2p' && foundUserId) ? verifiedName : name;

    const [result] = await connection.execute(
      `INSERT INTO recipients (user_id, name, type, email, phone, bank_name, bank_code, account_number, account_name, routing_number, nickname, recipient_user_id) 
       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [req.user.id, finalName, type, email || null, phone || null, bankName || null, bankCode || null, accountNumber || null, accountName || null, routingNumber || null, nickname || null, foundUserId || null]
    );

    const [newRecipient] = await connection.execute(
      'SELECT * FROM recipients WHERE id = ?',
      [result.insertId]
    );

    res.json({
      success: true,
      recipient: newRecipient[0]
    });
  } catch (error) {
    console.error('Add recipient error:', error);
    res.status(500).json({ success: false, error: 'Failed to add recipient' });
  } finally {
    connection.release();
  }
});

app.post('/api/recipients', authenticateToken, async (req, res) => {
  try {
    const { name, type, email, phone, bankName, accountNumber, routingNumber, nickname } = req.body;

    if (!name || !type) {
      return res.status(400).json({ error: 'Name and type are required' });
    }

    const connection = await pool.getConnection();
    const [result] = await connection.execute(
      `INSERT INTO recipients (user_id, name, type, email, phone, bank_name, account_number, routing_number, nickname) 
       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [req.user.id, name, type, email || null, phone || null, bankName || null, accountNumber || null, routingNumber || null, nickname || null]
    );
    connection.release();

    res.json({
      success: true,
      data: { id: result.insertId, message: 'Recipient added successfully' }
    });
  } catch (error) {
    console.error('Add recipient error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.put('/api/recipients/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const { name, type, email, phone, bankName, accountNumber, routingNumber, nickname } = req.body;

    const connection = await pool.getConnection();
    const [result] = await connection.execute(
      `UPDATE recipients SET name = ?, type = ?, email = ?, phone = ?, bank_name = ?, account_number = ?, routing_number = ?, nickname = ? 
       WHERE id = ? AND user_id = ?`,
      [name, type, email || null, phone || null, bankName || null, accountNumber || null, routingNumber || null, nickname || null, id, req.user.id]
    );
    connection.release();

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: 'Recipient not found' });
    }

    res.json({
      success: true,
      data: { message: 'Recipient updated successfully' }
    });
  } catch (error) {
    console.error('Update recipient error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.delete('/api/recipients/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;

    const connection = await pool.getConnection();
    const [result] = await connection.execute(
      'UPDATE recipients SET is_active = 0 WHERE id = ? AND user_id = ?',
      [id, req.user.id]
    );
    connection.release();

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: 'Recipient not found' });
    }

    res.json({
      success: true,
      data: { message: 'Recipient deleted successfully' }
    });
  } catch (error) {
    console.error('Delete recipient error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Transaction Limits API endpoints
app.get('/api/transaction-limits', authenticateToken, async (req, res) => {
  try {
    const { walletId } = req.query;
    const connection = await pool.getConnection();
    
    let query = 'SELECT * FROM transaction_limits WHERE user_id = ?';
    let params = [req.user.id];
    
    if (walletId) {
      query += ' AND wallet_id = ?';
      params.push(walletId);
    }
    
    const [limits] = await connection.execute(query, params);
    connection.release();

    // If no limits found, return default values
    if (limits.length === 0) {
      return res.json({
        success: true,
        data: {
          dailyLimit: 1000,
          weeklyLimit: 5000,
          monthlyLimit: 20000,
          singleTransactionLimit: 500,
          isEnabled: true
        }
      });
    }

    res.json({
      success: true,
      data: limits[0]
    });
  } catch (error) {
    console.error('Get transaction limits error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.post('/api/transaction-limits', authenticateToken, async (req, res) => {
  try {
    const { walletId, dailyLimit, weeklyLimit, monthlyLimit, singleTransactionLimit, isEnabled } = req.body;

    if (!dailyLimit || !weeklyLimit || !monthlyLimit || !singleTransactionLimit) {
      return res.status(400).json({ error: 'All limit values are required' });
    }

    const connection = await pool.getConnection();
    
    // Check if limits already exist for this user/wallet combination
    const [existing] = await connection.execute(
      'SELECT id FROM transaction_limits WHERE user_id = ? AND wallet_id = ?',
      [req.user.id, walletId || null]
    );

    if (existing.length > 0) {
      // Update existing limits
      await connection.execute(
        `UPDATE transaction_limits SET daily_limit = ?, weekly_limit = ?, monthly_limit = ?, single_transaction_limit = ?, is_enabled = ? 
         WHERE user_id = ? AND wallet_id = ?`,
        [dailyLimit, weeklyLimit, monthlyLimit, singleTransactionLimit, isEnabled, req.user.id, walletId || null]
      );
    } else {
      // Create new limits
      await connection.execute(
        `INSERT INTO transaction_limits (user_id, wallet_id, daily_limit, weekly_limit, monthly_limit, single_transaction_limit, is_enabled) 
         VALUES (?, ?, ?, ?, ?, ?, ?)`,
        [req.user.id, walletId || null, dailyLimit, weeklyLimit, monthlyLimit, singleTransactionLimit, isEnabled]
      );
    }
    
    connection.release();

    res.json({
      success: true,
      data: { message: 'Transaction limits updated successfully' }
    });
  } catch (error) {
    console.error('Update transaction limits error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Create Stripe account for bank transfers
app.post('/api/create-stripe-account', authenticateToken, async (req, res) => {
  try {
    const { currency, walletId, userId } = req.body;
    
    console.log('🔄 Creating Stripe account for:', { currency, walletId, userId });
    
    // Check if Stripe Connect is enabled
    try {
      // Create Stripe account with country-specific capabilities
      const country = getCountryFromCurrency(currency);
      const capabilities = {
        transfers: { requested: true }
      };
      
      // Only request card_payments for supported countries
      if (['US', 'CA', 'GB', 'AU', 'DE', 'FR', 'IT', 'ES', 'NL', 'BE', 'AT', 'IE', 'FI', 'LU', 'PT', 'SI', 'SK', 'EE', 'LV', 'LT', 'MT', 'CY'].includes(country)) {
        capabilities.card_payments = { requested: true };
      }

      const accountData = {
        type: 'express',
        country: country,
        email: req.user.email,
        capabilities: capabilities,
        business_type: 'individual',
        individual: {
          email: req.user.email,
          first_name: req.user.firstName || 'User',
          last_name: req.user.lastName || 'Name'
        },
        metadata: {
          walletId: walletId.toString(),
          userId: userId.toString(),
          currency: currency,
          platform: 'GlobalTrust'
        }
      };

      // Note: service_agreement parameter is not supported in accounts.create API
      // The service agreement is handled differently in Stripe Connect

      const account = await stripe.accounts.create(accountData);

      // Create external account (bank account)
      const externalAccount = await stripe.accounts.createExternalAccount(account.id, {
        external_account: {
          object: 'bank_account',
          country: getCountryFromCurrency(currency),
          currency: currency,
          account_holder_type: 'individual',
          account_holder_name: `${req.user.firstName} ${req.user.lastName}`,
          // In production, you'd collect real bank account details
          account_number: generateTestAccountNumber(currency),
          routing_number: getRoutingNumber(currency)
        }
      });

      // Get account details
      const accountDetails = await stripe.accounts.retrieve(account.id);
      
      const bankDetails = {
        accountId: account.id,
        bankName: getBankName(currency),
        accountName: `GlobalTrust ${currency.toUpperCase()} Wallet`,
        accountNumber: externalAccount.last4 ? `****${externalAccount.last4}` : generateTestAccountNumber(currency),
        routingNumber: getRoutingNumber(currency),
        swiftCode: getSwiftCode(currency),
        iban: currency === 'EUR' ? generateIBAN() : null,
        sortCode: currency === 'GBP' ? getSortCode() : null,
        bankCode: currency === 'NGN' ? getBankCode() : null
      };

      console.log('✅ Stripe account created:', bankDetails);

      res.json({
        success: true,
        account: bankDetails
      });

      } catch (stripeError) {
        // Handle Stripe Connect not enabled error
        if (stripeError.type === 'StripeInvalidRequestError' && 
            stripeError.message.includes('Connect')) {
          console.log('⚠️ Stripe Connect not enabled, using fallback bank details');
          
          // Return fallback bank details
          const fallbackDetails = {
            accountId: 'fallback',
            bankName: getBankName(currency),
            accountName: `GlobalTrust ${currency.toUpperCase()} Wallet`,
            accountNumber: generateTestAccountNumber(currency),
            routingNumber: getRoutingNumber(currency),
            swiftCode: getSwiftCode(currency),
            iban: currency === 'EUR' ? generateIBAN() : null,
            sortCode: currency === 'GBP' ? getSortCode() : null,
            bankCode: currency === 'NGN' ? getBankCode() : null
          };

          res.json({
            success: true,
            account: fallbackDetails,
            message: 'Using fallback bank details (Stripe Connect not enabled)'
          });
        } 
        // Handle card_payments capability not supported error
        else if (stripeError.type === 'StripeInvalidRequestError' && 
                 stripeError.message.includes('card_payments')) {
          console.log('⚠️ Card payments not supported for this country, using fallback bank details');
          
          // Return fallback bank details
          const fallbackDetails = {
            accountId: 'fallback',
            bankName: getBankName(currency),
            accountName: `GlobalTrust ${currency.toUpperCase()} Wallet`,
            accountNumber: generateTestAccountNumber(currency),
            routingNumber: getRoutingNumber(currency),
            swiftCode: getSwiftCode(currency),
            iban: currency === 'EUR' ? generateIBAN() : null,
            sortCode: currency === 'GBP' ? getSortCode() : null,
            bankCode: currency === 'NGN' ? getBankCode() : null
          };

          res.json({
            success: true,
            account: fallbackDetails,
            message: 'Using fallback bank details (card payments not supported for this country)'
          });
        }
        // Handle service agreement required error
        else if (stripeError.type === 'StripeInvalidRequestError' && 
                 stripeError.message.includes('service agreement')) {
          console.log('⚠️ Service agreement required for this country, using fallback bank details');
          
          // Return fallback bank details
          const fallbackDetails = {
            accountId: 'fallback',
            bankName: getBankName(currency),
            accountName: `GlobalTrust ${currency.toUpperCase()} Wallet`,
            accountNumber: generateTestAccountNumber(currency),
            routingNumber: getRoutingNumber(currency),
            swiftCode: getSwiftCode(currency),
            iban: currency === 'EUR' ? generateIBAN() : null,
            sortCode: currency === 'GBP' ? getSortCode() : null,
            bankCode: currency === 'NGN' ? getBankCode() : null
          };

          res.json({
            success: true,
            account: fallbackDetails,
            message: 'Using fallback bank details (service agreement required for this country)'
          });
        }
        // Handle unknown parameter error
        else if (stripeError.type === 'StripeInvalidRequestError' && 
                 stripeError.code === 'parameter_unknown') {
          console.log('⚠️ Unknown parameter error, using fallback bank details');
          
          // Return fallback bank details
          const fallbackDetails = {
            accountId: 'fallback',
            bankName: getBankName(currency),
            accountName: `GlobalTrust ${currency.toUpperCase()} Wallet`,
            accountNumber: generateTestAccountNumber(currency),
            routingNumber: getRoutingNumber(currency),
            swiftCode: getSwiftCode(currency),
            iban: currency === 'EUR' ? generateIBAN() : null,
            sortCode: currency === 'GBP' ? getSortCode() : null,
            bankCode: currency === 'NGN' ? getBankCode() : null
          };

          res.json({
            success: true,
            account: fallbackDetails,
            message: 'Using fallback bank details (unknown parameter error)'
          });
        } 
        else {
          throw stripeError; // Re-throw other errors
        }
      }

  } catch (error) {
    console.error('❌ Stripe account creation error:', error);
    res.status(500).json({ 
      success: false,
      error: 'Failed to create account details' 
    });
  }
});

// Helper functions for Stripe account creation
function getCountryFromCurrency(currency) {
  const countryMap = {
    'usd': 'US',
    'eur': 'DE',
    'gbp': 'GB',
    'ngn': 'NG'
  };
  return countryMap[currency.toLowerCase()] || 'US';
}

function getBankName(currency) {
  const bankMap = {
    'usd': 'Chase Bank',
    'eur': 'Deutsche Bank',
    'gbp': 'Barclays Bank',
    'ngn': 'Access Bank'
  };
  return bankMap[currency.toLowerCase()] || 'Chase Bank';
}

function getRoutingNumber(currency) {
  const routingMap = {
    'usd': '021000021',
    'eur': 'DE89370400440532013000',
    'gbp': '20-00-00',
    'ngn': '044'
  };
  return routingMap[currency.toLowerCase()] || '021000021';
}

function getSwiftCode(currency) {
  const swiftMap = {
    'usd': 'CHASUS33',
    'eur': 'DEUTDEFF',
    'gbp': 'BARCGB22',
    'ngn': 'ACCESSNGLA'
  };
  return swiftMap[currency.toLowerCase()] || 'CHASUS33';
}

function generateTestAccountNumber(currency) {
  // In production, this would be the real account number from Stripe
  const prefixes = {
    'usd': '1234',
    'eur': '9876',
    'gbp': '1122',
    'ngn': '0012'
  };
  const prefix = prefixes[currency.toLowerCase()] || '1234';
  return prefix + Math.random().toString().slice(2, 8);
}

function generateIBAN() {
  return 'DE89370400440532013000';
}

function getSortCode() {
  return '20-00-00';
}

function getBankCode() {
  return '044';
}

// ==================== BANK ACCOUNT ENDPOINTS ====================

// Manual database migration endpoint (for development)
app.post('/api/admin/migrate-database', async (req, res) => {
  try {
    const connection = await pool.getConnection();
    
    // Add missing columns to bank_accounts table
    const bankAccountColumns = [
      { name: 'country', type: 'VARCHAR(10)' },
      { name: 'swift_code', type: 'VARCHAR(20)' },
      { name: 'is_default', type: 'BOOLEAN DEFAULT FALSE' }
    ];

    for (const column of bankAccountColumns) {
      try {
        await connection.execute(`ALTER TABLE bank_accounts ADD COLUMN ${column.name} ${column.type}`);
        console.log(`Successfully added column ${column.name} to bank_accounts table`);
      } catch (error) {
        if (error.code === 'ER_DUP_FIELDNAME') {
          console.log(`Column ${column.name} already exists in bank_accounts table`);
        } else {
          console.log(`Error adding column ${column.name}:`, error.message);
        }
      }
    }

    connection.release();
    res.json({ success: true, message: 'Database migration completed' });
  } catch (error) {
    console.error('Database migration error:', error);
    res.status(500).json({ error: 'Migration failed' });
  }
});

// Get user bank accounts
app.get('/api/bank-accounts', authenticateToken, async (req, res) => {
  try {
    const connection = await pool.getConnection();
    const [accounts] = await connection.execute(
      'SELECT * FROM bank_accounts WHERE user_id = ? ORDER BY created_at DESC',
      [req.user.id]
    );
    connection.release();

    res.json({
      success: true,
      bankAccounts: accounts.map(account => ({
        id: account.id,
        accountName: account.account_name,
        accountNumber: account.account_number,
        bankName: account.bank_name,
        routingNumber: account.routing_number,
        accountType: account.account_type,
        country: account.country || null,
        swiftCode: account.swift_code || null,
        isDefault: account.is_default || false,
        isVerified: account.is_verified,
        createdAt: account.created_at
      }))
    });
  } catch (error) {
    console.error('Get bank accounts error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Add bank account
app.post('/api/bank-accounts', authenticateToken, async (req, res) => {
  try {
    const { accountName, accountNumber, bankName, routingNumber, accountType, country, swiftCode, isDefault } = req.body;

    if (!accountName || !accountNumber || !bankName) {
      return res.status(400).json({ error: 'Account name, number, and bank name are required' });
    }

    const connection = await pool.getConnection();
    
    // Check if is_default column exists, if not, skip the default logic
    try {
      // If this is set as default, unset other default accounts for this user
      if (isDefault) {
        await connection.execute(
          'UPDATE bank_accounts SET is_default = FALSE WHERE user_id = ?',
          [req.user.id]
        );
      }
    } catch (error) {
      console.log('is_default column not available, skipping default logic:', error.message);
    }

    // Try to insert with all columns, fallback to basic columns if needed
    let result;
    try {
      [result] = await connection.execute(
        'INSERT INTO bank_accounts (user_id, account_name, account_number, bank_name, routing_number, account_type, country, swift_code, is_default) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)',
        [req.user.id, accountName, accountNumber, bankName, routingNumber, accountType, country, swiftCode, isDefault || false]
      );
    } catch (error) {
      console.log('Full insert failed, trying basic insert:', error.message);
      // Fallback to basic insert without new columns
      [result] = await connection.execute(
        'INSERT INTO bank_accounts (user_id, account_name, account_number, bank_name, routing_number, account_type) VALUES (?, ?, ?, ?, ?, ?)',
        [req.user.id, accountName, accountNumber, bankName, routingNumber, accountType]
      );
    }
    
    connection.release();

    res.status(201).json({
      success: true,
      message: 'Bank account added successfully',
      accountId: result.insertId
    });
  } catch (error) {
    console.error('Add bank account error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// ==================== TRANSACTION ENDPOINTS ====================

// Get user transactions
app.get('/api/transactions', authenticateToken, async (req, res) => {
  try {
    const { page = 1, limit = 20, type } = req.query;
    const offset = (page - 1) * limit;

    const connection = await pool.getConnection();
    
    let query = 'SELECT * FROM transactions WHERE user_id = ?';
    let params = [req.user.id];

    if (type) {
      query += ' AND transaction_type = ?';
      params.push(type);
    }

    const safeLimit = Number(limit) || 20;
    const safeOffset = Number(offset) || 0;
    
    query += ` ORDER BY created_at DESC LIMIT ${safeLimit} OFFSET ${safeOffset}`;

    const [transactions] = await connection.execute(query, params);
    connection.release();

    res.json({
      success: true,
      transactions: transactions.map(transaction => ({
        id: transaction.id,
        transactionType: transaction.transaction_type,
        amount: parseFloat(transaction.amount),
        currency: transaction.currency,
        description: transaction.description,
        status: transaction.status,
        transactionFee: parseFloat(transaction.transaction_fee || 0),
        referenceId: transaction.reference_id,
        senderWalletId: transaction.sender_wallet_id,
        receiverWalletId: transaction.receiver_wallet_id,
        recipientPhone: transaction.recipient_phone,
        recipientName: transaction.recipient_name,
        recipientBank: transaction.recipient_bank,
        recipientAccountNumber: transaction.recipient_account_number,
        senderPhone: transaction.sender_phone,
        senderName: transaction.sender_name,
        exchangeRate: parseFloat(transaction.exchange_rate || 1),
        originalAmount: parseFloat(transaction.original_amount || transaction.amount),
        originalCurrency: transaction.original_currency || transaction.currency,
        convertedAmount: parseFloat(transaction.converted_amount || transaction.amount),
        convertedCurrency: transaction.converted_currency || transaction.currency,
        stripePaymentIntentId: transaction.stripe_payment_intent_id,
        createdAt: transaction.created_at
      }))
    });
  } catch (error) {
    console.error('Get transactions error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// P2P Transfer - Send money by email or phone (no fees, within platform)
app.post('/api/transactions/p2p-transfer', authenticateToken, requireKYCApproval, async (req, res) => {
  const connection = await pool.getConnection();
  try {
    const { amount, currency, description, receiverEmail, receiverPhone, walletId } = req.body;

    if (!amount || !currency) {
      connection.release();
      return res.status(400).json({ success: false, error: 'Amount and currency are required' });
    }

    if (!receiverEmail && !receiverPhone) {
      connection.release();
      return res.status(400).json({ success: false, error: 'Receiver email or phone number is required' });
    }

    // Validate amount is a valid number
    const transferAmount = parseFloat(amount);
    if (isNaN(transferAmount) || !isFinite(transferAmount)) {
      connection.release();
      return res.status(400).json({ success: false, error: 'Invalid amount format. Please enter a valid number.' });
    }

    if (transferAmount <= 0) {
      connection.release();
      return res.status(400).json({ success: false, error: 'Amount must be greater than 0' });
    }

    // Check for maximum amount (prevent extremely large numbers)
    if (transferAmount > 1000000000) {
      connection.release();
      return res.status(400).json({ success: false, error: 'Amount is too large. Maximum transfer amount is 1,000,000,000.' });
    }

    await connection.beginTransaction();

    try {
      // Get sender's wallet
      let senderWallet;
      if (walletId) {
        const [wallets] = await connection.execute(
          'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND currency = ? AND is_active = TRUE',
          [walletId, req.user.id, currency.toUpperCase()]
        );
        if (wallets.length === 0) {
          await connection.rollback();
          connection.release();
          return res.status(400).json({ success: false, error: 'Sender wallet not found or invalid' });
        }
        senderWallet = wallets[0];
      } else {
        const [wallets] = await connection.execute(
          'SELECT * FROM wallets WHERE user_id = ? AND wallet_type = "fiat" AND currency = ? AND is_active = TRUE LIMIT 1',
          [req.user.id, currency.toUpperCase()]
        );
        if (wallets.length === 0) {
          await connection.rollback();
          connection.release();
          return res.status(400).json({ success: false, error: 'No active wallet found for this currency' });
        }
        senderWallet = wallets[0];
      }

      // transferAmount already validated above
      if (parseFloat(senderWallet.balance) < transferAmount) {
        await connection.rollback();
      connection.release();
        return res.status(400).json({ success: false, error: 'Insufficient balance' });
      }

      // Find receiver by email or phone
      let receiver;
      if (receiverEmail) {
        const [receivers] = await connection.execute('SELECT id, first_name, last_name, email, phone FROM users WHERE email = ?', [receiverEmail]);
        if (receivers.length === 0) {
      await connection.rollback();
      connection.release();
          return res.status(404).json({ success: false, error: 'Receiver not found. Please ensure they have an account with this email.' });
        }
        receiver = receivers[0];
      } else if (receiverPhone) {
        const [receivers] = await connection.execute('SELECT id, first_name, last_name, email, phone FROM users WHERE phone = ?', [receiverPhone]);
        if (receivers.length === 0) {
          await connection.rollback();
          connection.release();
          return res.status(404).json({ success: false, error: 'Receiver not found. Please ensure they have an account with this phone number.' });
        }
        receiver = receivers[0];
      }

      // Prevent self-transfer
      if (receiver.id === req.user.id) {
        await connection.rollback();
        connection.release();
        return res.status(400).json({ success: false, error: 'Cannot send money to yourself' });
      }

      // Get or create receiver's wallet
      const [receiverWallets] = await connection.execute(
        'SELECT * FROM wallets WHERE user_id = ? AND currency = ? AND is_active = TRUE',
        [receiver.id, currency.toUpperCase()]
      );

      let receiverWallet;
      if (receiverWallets.length > 0) {
        receiverWallet = receiverWallets[0];
      } else {
        // Create wallet for receiver if it doesn't exist
        const walletName = `${currency.toUpperCase()} Wallet`;
        const [newWallet] = await connection.execute(
          'INSERT INTO wallets (user_id, wallet_name, currency, balance, wallet_type, is_active, is_default) VALUES (?, ?, ?, 0, "fiat", TRUE, FALSE)',
          [receiver.id, walletName, currency.toUpperCase()]
        );
        receiverWallet = { id: newWallet.insertId, balance: 0 };
      }

      // Generate base reference ID for the transfer
      const baseReferenceId = generateReferenceId();
      const senderReferenceId = `${baseReferenceId}-S`;
      const receiverReferenceId = `${baseReferenceId}-R`;
      const transactionFee = 0; // No fees for P2P transfers

      // Helper function to create transaction with retry on duplicate reference_id
      const createTransactionWithRetry = async (transactionData, refId, maxRetries = 3) => {
        let currentReferenceId = refId;
        for (let attempt = 0; attempt < maxRetries; attempt++) {
          try {
            await connection.execute(
              'INSERT INTO transactions (user_id, transaction_type, amount, currency, description, status, sender_wallet_id, receiver_wallet_id, transaction_fee, reference_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
              [
                transactionData.userId,
                transactionData.type,
                transactionData.amount,
                transactionData.currency,
                transactionData.description,
                transactionData.status,
                transactionData.senderWalletId,
                transactionData.receiverWalletId,
                transactionData.fee,
                currentReferenceId
              ]
            );
            return currentReferenceId; // Success
          } catch (error) {
            if (error.code === 'ER_DUP_ENTRY' && attempt < maxRetries - 1) {
              // Duplicate reference_id - generate a new one and retry
              console.log(`⚠️ Duplicate reference_id ${currentReferenceId}, generating new one...`);
              currentReferenceId = `${generateReferenceId()}-${refId.endsWith('-S') ? 'S' : 'R'}`;
              continue;
            }
            throw error; // Re-throw if not a duplicate error or max retries reached
          }
        }
      };

      // Debit from sender wallet
      await connection.execute(
        'UPDATE wallets SET balance = balance - ? WHERE id = ?',
        [transferAmount, senderWallet.id]
      );

      // Credit to receiver wallet
      await connection.execute(
        'UPDATE wallets SET balance = balance + ? WHERE id = ?',
        [transferAmount, receiverWallet.id]
      );

      // Create transaction record for sender
      await createTransactionWithRetry({
        userId: req.user.id,
        type: 'transfer',
        amount: transferAmount,
        currency: currency.toUpperCase(),
        description: description || `P2P transfer to ${receiver.first_name} ${receiver.last_name}`,
        status: 'completed',
        senderWalletId: senderWallet.id,
        receiverWalletId: receiverWallet.id,
        fee: transactionFee
      }, senderReferenceId);

      // Create transaction record for receiver
      await createTransactionWithRetry({
        userId: receiver.id,
        type: 'transfer',
        amount: transferAmount,
        currency: currency.toUpperCase(),
        description: description || `P2P transfer from ${req.user.first_name} ${req.user.last_name}`,
        status: 'completed',
        senderWalletId: senderWallet.id,
        receiverWalletId: receiverWallet.id,
        fee: 0
      }, receiverReferenceId);

      // Get the transaction ID using the sender's reference ID
      const [senderTransactions] = await connection.execute(
        'SELECT id FROM transactions WHERE reference_id = ? AND user_id = ? ORDER BY id DESC LIMIT 1',
        [senderReferenceId, req.user.id]
      );
      const transactionId = senderTransactions[0]?.id || null;

      await connection.commit();
      connection.release();

      res.json({
        success: true,
        message: 'Money sent successfully',
        referenceId: baseReferenceId, // Return base reference ID (without suffix)
        transactionId,
        transactionFee: 0,
        receiver: {
          id: receiver.id,
          name: `${receiver.first_name} ${receiver.last_name}`,
          email: receiver.email,
          phone: receiver.phone
        }
      });
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  } catch (error) {
    console.error('P2P transfer error:', error);
    if (connection) connection.release();
    res.status(500).json({ success: false, error: 'Internal server error' });
  }
});

// Send money by email (deprecated - use p2p-transfer instead)
app.post('/api/transactions/send-money', authenticateToken, requireKYCApproval, async (req, res) => {
  try {
    const { amount, currency, description, receiverEmail, walletId } = req.body;

    if (!amount || !currency || !receiverEmail) {
      return res.status(400).json({ success: false, error: 'Amount, currency, and receiver email are required' });
    }

    // Forward to P2P transfer endpoint by modifying request body
    req.body.receiverEmail = receiverEmail;
    req.body.receiverPhone = undefined;
    req.body.walletId = walletId;
    // Re-route to P2P endpoint handler
    return app._router.handle({ ...req, url: '/api/transactions/p2p-transfer', method: 'POST' }, res);
  } catch (error) {
    console.error('Send money error:', error);
    res.status(500).json({ success: false, error: 'Internal server error' });
  }
});

// Send money by phone number (deprecated - use p2p-transfer instead)
app.post('/api/transactions/send-money-phone', authenticateToken, async (req, res) => {
  try {
    const { amount, currency, description, recipientPhone, walletId } = req.body;

    if (!amount || !currency || !recipientPhone) {
      return res.status(400).json({ success: false, error: 'Amount, currency, and recipient phone are required' });
    }

    // Forward to P2P transfer endpoint by modifying request body
    req.body.receiverPhone = recipientPhone;
    req.body.receiverEmail = undefined;
    req.body.walletId = walletId;
    // Re-route to P2P endpoint handler
    return app._router.handle({ ...req, url: '/api/transactions/p2p-transfer', method: 'POST' }, res);
  } catch (error) {
    console.error('Send money by phone error:', error);
    res.status(500).json({ success: false, error: 'Internal server error' });
  }
});

// Add money to wallet
app.post('/api/transactions/add-money', authenticateToken, requireKYCApproval, async (req, res) => {
  try {
    const { amount, currency, walletId } = req.body;

    if (!amount || !currency || !walletId) {
      return res.status(400).json({ error: 'Amount, currency, and wallet ID are required' });
    }

    if (amount <= 0) {
      return res.status(400).json({ error: 'Amount must be greater than 0' });
    }

    const connection = await pool.getConnection();

    // Verify wallet belongs to user and is active (not frozen)
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND currency = ? AND is_active = TRUE',
      [walletId, req.user.id, currency]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Wallet not found or frozen. Please unfreeze the wallet to add money.' });
    }

    const referenceId = generateReferenceId();

    // Update wallet balance
    await connection.execute(
      'UPDATE wallets SET balance = balance + ? WHERE id = ?',
      [amount, walletId]
    );

    // Create transaction record
    await connection.execute(
      'INSERT INTO transactions (user_id, transaction_type, amount, currency, description, status, receiver_wallet_id, reference_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
      [req.user.id, 'deposit', amount, currency, 'Money added to wallet', 'completed', walletId, referenceId]
    );

    connection.release();

    res.json({
      success: true,
      message: 'Money added successfully',
      referenceId
    });
  } catch (error) {
    console.error('Add money error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Verify bank transfer and add money to wallet
app.post('/api/transactions/verify-bank-transfer', authenticateToken, async (req, res) => {
  try {
    const { amount, currency, walletId, transferReference, paymentMethod = 'Bank Transfer' } = req.body;

    if (!amount || !currency || !walletId || !transferReference) {
      return res.status(400).json({ 
        error: 'Amount, currency, wallet ID, and transfer reference are required' 
      });
    }

    if (amount <= 0) {
      return res.status(400).json({ error: 'Amount must be greater than 0' });
    }

    const connection = await pool.getConnection();

    try {
      // Verify wallet belongs to user and is active (not frozen)
      const [wallets] = await connection.execute(
        'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND currency = ? AND is_active = TRUE',
        [walletId, req.user.id, currency]
      );

      if (wallets.length === 0) {
        connection.release();
        return res.status(404).json({ error: 'Wallet not found or frozen. Please unfreeze the wallet to add money.' });
      }

      const wallet = wallets[0];

      // Check if transfer reference already exists to prevent duplicate processing
      const [existingTransactions] = await connection.execute(
        'SELECT id FROM transactions WHERE reference_id = ? AND user_id = ?',
        [transferReference, req.user.id]
      );

      if (existingTransactions.length > 0) {
        connection.release();
        return res.status(400).json({ 
          error: 'Transfer reference already processed',
          referenceId: transferReference
        });
      }

      // Start database transaction
      await connection.beginTransaction();

      try {
        // Update wallet balance
        await connection.execute(
          'UPDATE wallets SET balance = balance + ? WHERE id = ?',
          [amount, walletId]
        );

        // Create transaction record
        const description = `Bank transfer to ${wallet.wallet_name} - Ref: ${transferReference}`;
        await connection.execute(
          `INSERT INTO transactions (
            user_id, transaction_type, amount, currency, description, status, 
            receiver_wallet_id, reference_id, transaction_fee
          ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
          [
            req.user.id, 
            'deposit', 
            amount, 
            currency, 
            description, 
            'completed', 
            walletId, 
            transferReference,
            0 // Bank transfers are typically free
          ]
        );

        await connection.commit();
        
        console.log(`✅ Bank transfer verified: ${amount} ${currency} added to wallet ${walletId} (Ref: ${transferReference})`);

        res.json({
          success: true,
          message: 'Bank transfer verified and wallet funded successfully',
          referenceId: transferReference,
          amount: amount,
          currency: currency,
          walletName: wallet.wallet_name
        });

      } catch (transactionError) {
        await connection.rollback();
        throw transactionError;
      }

    } finally {
      connection.release();
    }

  } catch (error) {
    console.error('Verify bank transfer error:', error);
    
    // Record failed transaction for audit trail
    try {
      const connection = await pool.getConnection();
      await connection.execute(
        `INSERT INTO transactions (
          user_id, transaction_type, amount, currency, description, status, 
          reference_id, transaction_fee
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          req.user.id, 
          'deposit', 
          req.body.amount || 0, 
          req.body.currency || 'USD', 
          `Failed bank transfer verification - Ref: ${req.body.transferReference || 'N/A'}`, 
          'failed', 
          req.body.transferReference || `FAILED-${Date.now()}`,
          0
        ]
      );
      connection.release();
    } catch (logError) {
      console.error('Failed to log failed transaction:', logError);
    }

    res.status(500).json({ 
      error: 'Failed to verify bank transfer. Please try again or contact support.',
      referenceId: req.body.transferReference
    });
  }
});

// Verify USSD payment and add money to wallet
app.post('/api/transactions/verify-ussd-payment', authenticateToken, async (req, res) => {
  try {
    const { amount, currency, walletId, ussdReference, bankName, paymentMethod = 'USSD' } = req.body;

    if (!amount || !currency || !walletId || !ussdReference) {
      return res.status(400).json({ 
        error: 'Amount, currency, wallet ID, and USSD reference are required' 
      });
    }

    if (amount <= 0) {
      return res.status(400).json({ error: 'Amount must be greater than 0' });
    }

    const connection = await pool.getConnection();

    try {
      // Verify wallet belongs to user and is active (not frozen)
      const [wallets] = await connection.execute(
        'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND currency = ? AND is_active = TRUE',
        [walletId, req.user.id, currency]
      );

      if (wallets.length === 0) {
        connection.release();
        return res.status(404).json({ error: 'Wallet not found or frozen. Please unfreeze the wallet to add money.' });
      }

      const wallet = wallets[0];

      // Check if USSD reference already exists to prevent duplicate processing
      const [existingTransactions] = await connection.execute(
        'SELECT id FROM transactions WHERE reference_id = ? AND user_id = ?',
        [ussdReference, req.user.id]
      );

      if (existingTransactions.length > 0) {
        connection.release();
        return res.status(400).json({ 
          error: 'USSD reference already processed',
          referenceId: ussdReference
        });
      }

      // Start database transaction
      await connection.beginTransaction();

      try {
        // Update wallet balance
        await connection.execute(
          'UPDATE wallets SET balance = balance + ? WHERE id = ?',
          [amount, walletId]
        );

        // Create transaction record
        const description = `USSD payment to ${wallet.wallet_name}${bankName ? ` via ${bankName}` : ''} - Ref: ${ussdReference}`;
        await connection.execute(
          `INSERT INTO transactions (
            user_id, transaction_type, amount, currency, description, status, 
            receiver_wallet_id, reference_id, transaction_fee
          ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
          [
            req.user.id, 
            'deposit', 
            amount, 
            currency, 
            description, 
            'completed', 
            walletId, 
            ussdReference,
            0 // USSD payments typically have small fixed fees, but we'll keep it free for now
          ]
        );

        await connection.commit();
        
        console.log(`✅ USSD payment verified: ${amount} ${currency} added to wallet ${walletId} (Ref: ${ussdReference})`);

        res.json({
          success: true,
          message: 'USSD payment verified and wallet funded successfully',
          referenceId: ussdReference,
          amount: amount,
          currency: currency,
          walletName: wallet.wallet_name,
          bankName: bankName
        });

      } catch (transactionError) {
        await connection.rollback();
        throw transactionError;
      }

    } finally {
      connection.release();
    }

  } catch (error) {
    console.error('Verify USSD payment error:', error);
    
    // Record failed transaction for audit trail
    try {
      const connection = await pool.getConnection();
      await connection.execute(
        `INSERT INTO transactions (
          user_id, transaction_type, amount, currency, description, status, 
          reference_id, transaction_fee
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          req.user.id, 
          'deposit', 
          req.body.amount || 0, 
          req.body.currency || 'USD', 
          `Failed USSD payment verification - Ref: ${req.body.ussdReference || 'N/A'}${req.body.bankName ? ` via ${req.body.bankName}` : ''}`, 
          'failed', 
          req.body.ussdReference || `FAILED-USSD-${Date.now()}`,
          0
        ]
      );
      connection.release();
    } catch (logError) {
      console.error('Failed to log failed USSD transaction:', logError);
    }

    res.status(500).json({ 
      error: 'Failed to verify USSD payment. Please try again or contact support.',
      referenceId: req.body.ussdReference
    });
  }
});

// ==================== STRIPE CONNECT ENDPOINTS ====================

// Create or get Stripe Connect account for user
app.post('/api/stripe/connect-account', authenticateToken, async (req, res) => {
  try {
    const { country = 'US', currency = 'USD' } = req.body;
    const connection = await pool.getConnection();

    try {
      // Check if user already has a Stripe Connect account
      const [existingAccounts] = await connection.execute(
        'SELECT * FROM stripe_connect_accounts WHERE user_id = ?',
        [req.user.id]
      );

      if (existingAccounts.length > 0) {
        const account = existingAccounts[0];
        
        // Get latest account info from Stripe
        const stripeAccount = await stripe.accounts.retrieve(account.stripe_account_id);
        
        // Update local database with latest info
        await connection.execute(
          `UPDATE stripe_connect_accounts SET 
            charges_enabled = ?, payouts_enabled = ?, details_submitted = ?,
            requirements_currently_due = ?, requirements_eventually_due = ?, requirements_past_due = ?
           WHERE user_id = ?`,
          [
            stripeAccount.charges_enabled,
            stripeAccount.payouts_enabled,
            stripeAccount.details_submitted,
            JSON.stringify(stripeAccount.requirements?.currently_due || []),
            JSON.stringify(stripeAccount.requirements?.eventually_due || []),
            JSON.stringify(stripeAccount.requirements?.past_due || []),
            req.user.id
          ]
        );

        connection.release();
        
        return res.json({
          success: true,
          account: {
            id: account.stripe_account_id,
            chargesEnabled: stripeAccount.charges_enabled,
            payoutsEnabled: stripeAccount.payouts_enabled,
            detailsSubmitted: stripeAccount.details_submitted,
            requirementsCurrentlyDue: stripeAccount.requirements?.currently_due || [],
            requirementsEventuallyDue: stripeAccount.requirements?.eventually_due || [],
            requirementsPastDue: stripeAccount.requirements?.past_due || []
          }
        });
      }

      // Create new Stripe Connect account
      // For Nigeria, Express accounts cannot be created via API due to service agreement restrictions
      // We need to handle Nigeria accounts differently - they must be created through onboarding
      let stripeAccount;
      
      if (country === 'NG') {
        // For Nigeria, Stripe Express accounts cannot be created via API
        // Stripe doesn't support creating Express accounts for Nigeria due to service agreement restrictions
        // We should use Flutterwave for Nigeria instead
        try {
          return res.status(400).json({
            success: false,
            error: 'Nigeria Stripe accounts not supported',
            message: 'Stripe Express accounts are not available for Nigeria due to service agreement restrictions. Please use Flutterwave for Nigeria withdrawals.',
            country: 'NG',
            alternative: 'flutterwave',
            useFlutterwave: true
          });
        } finally {
          connection.release();
        }
      } else {
        // For other countries, create Express account normally
        const accountData = {
        type: 'express',
        country: country,
        email: req.user.email,
        business_type: 'individual',
        capabilities: {
          transfers: { requested: true }
        }
        };
        
        stripeAccount = await stripe.accounts.create(accountData);
      }

      // Save to database
      await connection.execute(
        `INSERT INTO stripe_connect_accounts (
          user_id, stripe_account_id, account_type, country, currency,
          charges_enabled, payouts_enabled, details_submitted
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          req.user.id,
          stripeAccount.id,
          'express',
          country,
          currency,
          stripeAccount.charges_enabled,
          stripeAccount.payouts_enabled,
          stripeAccount.details_submitted
        ]
      );

      connection.release();

      res.json({
        success: true,
        message: country === 'NG' 
          ? 'Stripe Connect account created. Please complete onboarding to enable transfers with recipient service agreement.'
          : 'Stripe Connect account created successfully',
        account: {
          id: stripeAccount.id,
          chargesEnabled: stripeAccount.charges_enabled,
          payoutsEnabled: stripeAccount.payouts_enabled,
          detailsSubmitted: stripeAccount.details_submitted,
          requirementsCurrentlyDue: stripeAccount.requirements?.currently_due || [],
          requirementsEventuallyDue: stripeAccount.requirements?.eventually_due || [],
          requirementsPastDue: stripeAccount.requirements?.past_due || [],
          // For Nigeria, transfers capability will be requested during onboarding
          requiresOnboarding: country === 'NG',
          country: country
        }
      });

    } finally {
      if (connection) connection.release();
    }

  } catch (error) {
    console.error('Create Stripe Connect account error:', error);
    
    // Handle Nigeria-specific service agreement errors
    if (error.type === 'StripeInvalidRequestError' && 
        error.message && 
        (error.message.includes('service agreement') || 
         error.message.includes('full service agreement'))) {
      
      const requestCountry = req.body.country || 'US';
      if (requestCountry === 'NG') {
        // Nigeria accounts cannot be created via Stripe API
        return res.status(400).json({ 
          success: false,
          error: 'Nigeria Stripe accounts not supported',
          message: 'Stripe Express accounts are not available for Nigeria due to service agreement restrictions. Please use Flutterwave for Nigeria withdrawals.',
          country: 'NG',
          alternative: 'flutterwave',
          useFlutterwave: true
        });
      }
      
      // For other countries with service agreement errors
      return res.status(400).json({ 
        success: false,
        error: 'Service agreement configuration required',
        message: 'Please complete Stripe onboarding to configure the service agreement properly.',
        details: error.message,
        requiresOnboarding: true,
        country: requestCountry
      });
    }
    
    // Handle other Stripe errors
    if (error.type && error.type.includes('Stripe')) {
      return res.status(400).json({ 
        success: false,
        error: 'Stripe API error',
        message: error.message || 'Failed to create Stripe Connect account',
        details: error.message
      });
    }
    
    // Generic error
    res.status(500).json({ 
      success: false,
      error: 'Failed to create Stripe Connect account',
      message: error.message || 'An unexpected error occurred',
      details: error.message 
    });
  }
});

// Create account link for onboarding
app.post('/api/stripe/account-link', authenticateToken, async (req, res) => {
  try {
    const { country = 'US', currency = 'USD' } = req.body;
    const connection = await pool.getConnection();
    
    let accountId;
    let accountCountry = country;
    
    const [accounts] = await connection.execute(
      'SELECT stripe_account_id FROM stripe_connect_accounts WHERE user_id = ?',
      [req.user.id]
    );

    if (accounts.length === 0) {
      // No account exists
      if (country === 'NG') {
        // For Nigeria, Express accounts cannot be created via API
        // We need to inform the user that they should use Flutterwave or another payment method
        // Or we can try to create a Custom account instead of Express for Nigeria
      connection.release();
        
        return res.status(400).json({ 
          error: 'Nigeria Express accounts not supported',
          message: 'Stripe Express accounts for Nigeria require special handling. Please use Flutterwave for Nigeria withdrawals, or contact support for alternative payment methods.',
          country: 'NG',
          alternative: 'flutterwave'
        });
      } else {
        connection.release();
        return res.status(404).json({ error: 'No Stripe Connect account found. Please create one first.' });
      }
    } else {
      accountId = accounts[0].stripe_account_id;
      
      // Get account info to check country
      try {
        const accountInfo = await stripe.accounts.retrieve(accountId);
        accountCountry = accountInfo.country;
      } catch (retrieveError) {
        // Account might not exist in Stripe, use provided country
        accountCountry = country;
      }
    }

    // Create account link for onboarding
    // For Nigeria, the account will be created during onboarding with recipient service agreement
    const accountLinkData = {
      refresh_url: `${process.env.CLIENT_URL || 'http://localhost:3000'}/stripe/reauth`,
      return_url: `${process.env.CLIENT_URL || 'http://localhost:3000'}/stripe/return`,
      type: 'account_onboarding',
    };

    // If account exists, specify it; otherwise account will be created during onboarding
    if (accountId) {
      accountLinkData.account = accountId;
    } else {
      // For Nigeria, create account during onboarding
      accountLinkData.type = 'account_onboarding';
      // Note: Stripe will create the account with recipient service agreement during onboarding
    }

    const accountLink = await stripe.accountLinks.create(accountLinkData);

    // Update database with link info
    await connection.execute(
      `UPDATE stripe_connect_accounts SET 
        account_link_url = ?, account_link_expires_at = ?
       WHERE user_id = ?`,
      [accountLink.url, new Date(accountLink.expires_at * 1000), req.user.id]
    );

    connection.release();

    res.json({
      success: true,
      accountLinkUrl: accountLink.url,
      expiresAt: accountLink.expires_at
    });

  } catch (error) {
    console.error('Create account link error:', error);
    res.status(500).json({ 
      error: 'Failed to create account link',
      details: error.message 
    });
  }
});

// Add bank account to Stripe Connect account
app.post('/api/stripe/add-bank-account', authenticateToken, async (req, res) => {
  try {
    const { 
      accountName, 
      accountNumber, 
      routingNumber, 
      bankName,
      country = 'US',
      currency = 'USD'
    } = req.body;

    if (!accountName || !accountNumber || !routingNumber) {
      return res.status(400).json({ 
        error: 'Account name, number, and routing number are required' 
      });
    }

    const connection = await pool.getConnection();
    
    try {
      // Get user's Stripe Connect account
      const [accounts] = await connection.execute(
        'SELECT stripe_account_id FROM stripe_connect_accounts WHERE user_id = ?',
        [req.user.id]
      );

      if (accounts.length === 0) {
        connection.release();
        return res.status(404).json({ 
          error: 'No Stripe Connect account found. Please create one first.' 
        });
      }

      const stripeAccountId = accounts[0].stripe_account_id;

      // Create external account in Stripe
      const externalAccount = await stripe.accounts.createExternalAccount(
        stripeAccountId,
        {
          external_account: {
            object: 'bank_account',
            country: country,
            currency: currency.toLowerCase(),
            account_holder_name: accountName,
            account_number: accountNumber,
            routing_number: routingNumber,
          }
        }
      );

      // Save to local database
      await connection.execute(
        `INSERT INTO bank_accounts (
          user_id, account_name, account_number, bank_name, routing_number,
          account_type, country, is_verified
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          req.user.id,
          accountName,
          accountNumber,
          bankName || 'Bank Account',
          routingNumber,
          'checking',
          country,
          externalAccount.status === 'verified'
        ]
      );

      connection.release();

      res.json({
        success: true,
        message: 'Bank account added successfully',
        externalAccount: {
          id: externalAccount.id,
          status: externalAccount.status,
          currency: externalAccount.currency,
          last4: externalAccount.last4
        }
      });

    } finally {
      if (connection) connection.release();
    }

  } catch (error) {
    console.error('Add bank account error:', error);
    res.status(500).json({ 
      error: 'Failed to add bank account',
      details: error.message 
    });
  }
});

// Add bank account via Flutterwave (for Nigeria)
app.post('/api/flutterwave/add-bank-account', authenticateToken, async (req, res) => {
  try {
    const { 
      accountName, 
      accountNumber, 
      bankCode, 
      bankName,
      country = 'NG',
      currency = 'NGN'
    } = req.body;

    if (!accountName || !accountNumber || !bankCode) {
      return res.status(400).json({ 
        success: false,
        error: 'Account name, number, and bank code are required' 
      });
    }

    const connection = await pool.getConnection();
    
    try {
      // Verify account via Flutterwave
      const { resolveAccount } = require('./services/flutterwave');
      let accountInfo;
      try {
        accountInfo = await resolveAccount({ 
          accountNumber: String(accountNumber).trim(), 
          bankCode: String(bankCode).trim() 
        });
      } catch (resolveError) {
        connection.release();
        return res.status(400).json({ 
          success: false,
          error: 'Failed to verify account',
          message: resolveError.message || 'Could not verify bank account. Please check account number and bank code.',
          details: resolveError.details
        });
      }

      // Verify account name matches
      const resolvedName = accountInfo.account_name?.toLowerCase().trim();
      const providedName = accountName.toLowerCase().trim();
      
      if (resolvedName !== providedName && !resolvedName.includes(providedName) && !providedName.includes(resolvedName)) {
        connection.release();
        return res.status(400).json({ 
          success: false,
          error: 'Account name mismatch',
          message: `Account name does not match. Resolved name: ${accountInfo.account_name}`,
          resolvedName: accountInfo.account_name
        });
      }

      // Create Flutterwave subaccount (like Stripe Connect)
      const { createSubaccount } = require('./services/flutterwave');
      let subaccount;
      let subaccountId = null;
      
      try {
        subaccount = await createSubaccount({
          accountBank: bankCode,
          accountNumber: accountNumber,
          businessName: `${accountInfo.account_name || accountName} - Globana`,
          businessEmail: req.user.email,
          splitType: 'percentage',
          splitValue: 90 // User gets 90%, platform keeps 10%
        });
        
        subaccountId = subaccount.id;
        console.log(`✅ Flutterwave subaccount created: ${subaccountId} for user ${req.user.id}`);
      } catch (subaccountError) {
        console.error('⚠️ Failed to create subaccount, will use manual transfers:', subaccountError.message);
        // Continue without subaccount - can use manual transfers instead
      }

      // Save to local database
      await connection.execute(
        `INSERT INTO bank_accounts (
          user_id, account_name, account_number, bank_name, routing_number,
          account_type, country, is_verified, provider, flutterwave_subaccount_id
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          req.user.id,
          accountInfo.account_name || accountName,
          accountNumber,
          bankName || 'Bank Account',
          bankCode, // Store bank code as routing number
          'checking',
          country,
          true, // Verified via Flutterwave
          'flutterwave',
          subaccountId
        ]
      );

      connection.release();

      res.json({
        success: true,
        message: subaccountId 
          ? 'Bank account added, verified, and Flutterwave subaccount created successfully'
          : 'Bank account added and verified successfully (manual transfers available)',
        account: {
          accountName: accountInfo.account_name,
          accountNumber: accountNumber,
          bankCode: bankCode,
          bankName: bankName,
          isVerified: true,
          provider: 'flutterwave',
          subaccountId: subaccountId
        }
      });

    } finally {
      if (connection) connection.release();
    }

  } catch (error) {
    console.error('Add Flutterwave bank account error:', error);
    res.status(500).json({ 
      success: false,
      error: 'Failed to add bank account',
      details: error.message 
    });
  }
});

// ==================== WITHDRAWAL ENDPOINTS ====================

// Process withdrawal to bank account with Stripe payout
app.post('/api/withdrawals', authenticateToken, async (req, res) => {
  try {
    const { amount, currency, walletId, bankAccountId, description } = req.body;

    if (!amount || !currency || !walletId || !bankAccountId) {
      return res.status(400).json({ error: 'Amount, currency, wallet ID, and bank account ID are required' });
    }

    if (amount <= 0) {
      return res.status(400).json({ error: 'Amount must be greater than 0' });
    }

    const connection = await pool.getConnection();
    await connection.beginTransaction();

    try {
      // Verify wallet belongs to user and has sufficient balance
      const [wallets] = await connection.execute(
        'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND currency = ? AND is_active = TRUE',
        [walletId, req.user.id, currency]
      );

      if (wallets.length === 0) {
        await connection.rollback();
        connection.release();
        return res.status(404).json({ error: 'Wallet not found or inactive' });
      }

      const wallet = wallets[0];
      const currentBalance = parseFloat(wallet.balance);

      // Verify bank account belongs to user
      const [bankAccounts] = await connection.execute(
        'SELECT * FROM bank_accounts WHERE id = ? AND user_id = ?',
        [bankAccountId, req.user.id]
      );

      if (bankAccounts.length === 0) {
        await connection.rollback();
        connection.release();
        return res.status(404).json({ error: 'Bank account not found' });
      }

      const bankAccount = bankAccounts[0];

      // Use Flutterwave for all bank accounts (we're using Flutterwave fully now)
      const isFlutterwave = bankAccount.provider === 'flutterwave' || !bankAccount.provider || bankAccount.country === 'NG';
      
      if (isFlutterwave) {
        // Use Flutterwave for withdrawals (supports all countries via Flutterwave)
        // Calculate withdrawal fee
      let withdrawalFee = 0;
        if (currency.toUpperCase() === 'NGN') {
          withdrawalFee = 25;
        } else {
          withdrawalFee = 1; // $1 or equivalent for other currencies
        }
        
        const totalAmount = parseFloat(amount) + withdrawalFee;

      if (currentBalance < totalAmount) {
        await connection.rollback();
        connection.release();
        return res.status(400).json({ 
            error: `Insufficient balance. Required: ${totalAmount} ${currency}, Available: ${currentBalance} ${currency}`,
            required: totalAmount,
            available: currentBalance,
            fee: withdrawalFee
          });
        }
        
        // Debit wallet immediately
        await connection.execute(
          'UPDATE wallets SET balance = balance - ? WHERE id = ?',
          [totalAmount, walletId]
        );
        
        // Create withdrawal record
        const [withdrawalResult] = await connection.execute(
          `INSERT INTO withdrawals (
            user_id, wallet_id, bank_account_id, amount, fee, currency, status, description
          ) VALUES (?, ?, ?, ?, ?, ?, 'pending', ?)`,
          [req.user.id, walletId, bankAccountId, amount, withdrawalFee, currency, description || 'Withdrawal via Flutterwave']
        );
        
        const withdrawalId = withdrawalResult.insertId;
        const referenceId = `flw_wd_${withdrawalId}_${Date.now()}`;
        
        // Process Flutterwave transfer
        try {
          const { createTransfer } = require('./services/flutterwave');
          const flwResponse = await createTransfer({
            amount: parseFloat(amount),
            currency: currency.toUpperCase(),
            bankCode: bankAccount.routing_number, // Bank code stored in routing_number
            accountNumber: bankAccount.account_number,
            narration: description || `Withdrawal from ${wallet.wallet_name}`,
            reference: referenceId
          });
          
          // Update withdrawal with Flutterwave reference
          await connection.execute(
            `UPDATE withdrawals SET 
              reference_id = ?, 
              provider_reference = ?,
              status = ?
            WHERE id = ?`,
            [
              referenceId,
              flwResponse.reference || flwResponse.id,
              flwResponse.status === 'SUCCESSFUL' ? 'completed' : 'pending',
              withdrawalId
            ]
          );
          
          await connection.commit();
        connection.release();
        
          return res.json({
            success: true,
            withdrawalId,
            referenceId,
            status: flwResponse.status === 'SUCCESSFUL' ? 'completed' : 'pending',
            message: flwResponse.status === 'SUCCESSFUL' 
              ? 'Withdrawal completed successfully' 
              : 'Withdrawal initiated, pending confirmation',
            provider: 'flutterwave',
            providerReference: flwResponse.reference || flwResponse.id
          });
          
        } catch (flwError) {
          // Refund on failure
          await connection.execute(
            'UPDATE wallets SET balance = balance + ? WHERE id = ?',
            [totalAmount, walletId]
          );
          await connection.rollback();
          connection.release();
          console.error('Flutterwave transfer error:', flwError);
          return res.status(500).json({
            error: 'Withdrawal failed',
            message: flwError.message || 'Failed to process withdrawal via Flutterwave'
          });
        }
      } else {
        await connection.rollback();
        connection.release();
        return res.status(400).json({ 
          error: 'Bank account must be connected via Flutterwave. Please connect your bank account first.',
          requiresFlutterwaveAccount: true
        });
      }
    } catch (error) {
      console.error('Withdrawal error:', error);
      if (connection) {
        await connection.rollback();
        connection.release();
      }
      res.status(500).json({
        error: 'Withdrawal processing failed',
        message: error.message
      });
    }
  } catch (error) {
    console.error('Withdrawal endpoint error:', error);
    res.status(500).json({
      error: 'Withdrawal processing failed',
      message: error.message
    });
  }
});

// Get Flutterwave public key (safe to expose to frontend)
app.get('/api/flutterwave/public-key', authenticateToken, async (req, res) => {
  try {
    const publicKey = process.env.FLW_PUBLIC_KEY?.trim();
    if (!publicKey) {
      return res.status(500).json({
        success: false,
        error: 'Flutterwave public key not configured'
      });
    }
    return res.json({
      success: true,
      publicKey
    });
  } catch (error) {
    console.error('Get Flutterwave public key error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to get Flutterwave public key'
    });
  }
});

// Card withdrawal endpoint - verify Flutterwave payment and credit wallet
app.post('/api/withdrawals/card', authenticateToken, async (req, res) => {
  const connection = await pool.getConnection();
  try {
    const { 
      txRef,
      transactionId,
      flwRef,
      amount, 
      currency, 
      walletId, 
      fee 
    } = req.body;

    if (!txRef && !transactionId && !flwRef) {
        connection.release();
        return res.status(400).json({ 
        success: false,
        error: 'Transaction reference is required' 
      });
    }

    if (!amount || parseFloat(amount) <= 0) {
      connection.release();
      return res.status(400).json({ 
        success: false,
        error: 'Valid amount is required' 
      });
    }

    if (!walletId) {
      connection.release();
      return res.status(400).json({ 
        success: false,
        error: 'Wallet ID is required' 
      });
    }

    await connection.beginTransaction();

    try {
      // Get wallet
      const [wallets] = await connection.execute(
        'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND is_active = TRUE',
        [walletId, req.user.id]
      );

      if (wallets.length === 0) {
        await connection.rollback();
        connection.release();
        return res.status(404).json({ 
          success: false,
          error: 'Wallet not found' 
        });
      }

      const wallet = wallets[0];
      const withdrawalAmount = parseFloat(amount);
      const withdrawalFee = parseFloat(fee || 0);

      // Verify payment with Flutterwave
      const { verifyCardCharge } = require('./services/flutterwave');
      
      let verification;
      try {
        // Flutterwave verification can use either:
        // 1. transaction_id (numeric) - preferred
        // 2. tx_ref (transaction reference) - fallback
        // Try transaction_id first, then fall back to tx_ref
        const refToVerify = transactionId || txRef || flwRef;
        if (!refToVerify) {
          throw new Error('No Flutterwave transaction reference to verify');
        }

        console.log('🔍 Verifying Flutterwave transaction:', {
          transactionId,
          txRef,
          flwRef,
          using: refToVerify
        });
        verification = await verifyCardCharge(refToVerify);
        console.log('✅ Verification result:', { 
          status: verification.status, 
          amount: verification.amount,
          tx_ref: verification.tx_ref,
          transaction_id: verification.id
        });
        
        if (verification.status !== 'successful') {
          throw new Error('Payment verification failed - payment not successful');
        }

        // Double-check amount matches
        const paidAmount = parseFloat(verification.amount || verification.charged_amount || 0);
        const expectedAmount = withdrawalAmount + withdrawalFee;
        
        if (Math.abs(paidAmount - expectedAmount) > 0.01) {
          throw new Error(`Amount mismatch. Expected: ${expectedAmount}, Paid: ${paidAmount}`);
        }

        // Credit wallet
      await connection.execute(
          'UPDATE wallets SET balance = balance + ? WHERE id = ?',
          [withdrawalAmount, walletId]
      );

        // Create transaction record
        const referenceId = `CARD_WD_${Date.now()}`;
        const [transactionResult] = await connection.execute(
        `INSERT INTO transactions (
            user_id, transaction_type, amount, currency, description, status, reference_id, 
            sender_wallet_id, receiver_wallet_id, transaction_fee
          ) VALUES (?, 'deposit', ?, ?, ?, 'completed', ?, NULL, ?, ?)`,
        [
          req.user.id,
            withdrawalAmount,
            currency.toUpperCase(),
            `Card withdrawal: ${withdrawalAmount.toFixed(2)} ${currency.toUpperCase()} added to ${wallet.wallet_name}`,
          referenceId,
            walletId,
            withdrawalFee
        ]
      );

      // Create withdrawal record
        // Note: For card withdrawals, we don't have a bank_account_id, so we'll use NULL
        // Also, the table uses withdrawal_fee, not fee, and doesn't have provider_reference or method columns
      const [withdrawalResult] = await connection.execute(
        `INSERT INTO withdrawals (
            user_id, wallet_id, bank_account_id, amount, withdrawal_fee, total_amount, currency, status, description, 
            reference_id, bank_transfer_id
          ) VALUES (?, ?, NULL, ?, ?, ?, ?, 'completed', ?, ?, ?)`,
        [
          req.user.id,
          walletId,
            withdrawalAmount,
          withdrawalFee,
            withdrawalAmount + withdrawalFee, // total_amount
            currency.toUpperCase(),
            `Card withdrawal: ${withdrawalAmount.toFixed(2)} ${currency.toUpperCase()} added to ${wallet.wallet_name}`,
          referenceId,
            verification.flw_ref || flwRef || transactionId || txRef || 'N/A' // Store in bank_transfer_id for reference
        ]
      );

      await connection.commit();
      connection.release();

        return res.json({
        success: true,
          withdrawalId: withdrawalResult.insertId,
          transactionId: transactionResult.insertId,
          referenceId,
          message: 'Card withdrawal successful. Funds added to wallet.',
          amount: withdrawalAmount,
          fee: withdrawalFee,
          currency: currency.toUpperCase()
        });
      } catch (verifyError) {
        await connection.rollback();
        connection.release();
        console.error('Payment verification error:', verifyError);
        return res.status(400).json({
          success: false,
          error: 'Payment verification failed',
          message: verifyError.message || 'Failed to verify payment. Please contact support.'
        });
      }
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  } catch (error) {
    console.error('Card withdrawal error:', error);
    if (connection) {
      await connection.rollback();
      connection.release();
    }
    res.status(500).json({ 
      success: false,
      error: 'Card withdrawal processing failed',
      message: error.message
    });
  }
});

// Check withdrawal status
app.get('/api/withdrawals/:withdrawalId/status', authenticateToken, async (req, res) => {
  try {
    const { withdrawalId } = req.params;
    const connection = await pool.getConnection();

    const [withdrawals] = await connection.execute(
      `SELECT w.*, ba.bank_name, ba.account_number, ba.account_name, ba.provider, ba.flutterwave_subaccount_id
       FROM withdrawals w
       LEFT JOIN bank_accounts ba ON w.bank_account_id = ba.id
       WHERE w.id = ? AND w.user_id = ?`,
      [withdrawalId, req.user.id]
    );

    if (withdrawals.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Withdrawal not found' });
    }

    const withdrawal = withdrawals[0];

    // If we have a Stripe payout ID, get the latest status from Stripe
    if (withdrawal.stripe_payout_id) {
      try {
        const stripeTransfer = await stripe.transfers.retrieve(withdrawal.stripe_payout_id);
        
        // Update local status if different
        if (stripeTransfer.status !== withdrawal.status) {
          await connection.execute(
            'UPDATE withdrawals SET status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
            [stripeTransfer.status === 'paid' ? 'completed' : stripeTransfer.status, withdrawalId]
          );
          withdrawal.status = stripeTransfer.status === 'paid' ? 'completed' : stripeTransfer.status;
        }

      } catch (stripeError) {
        console.error('Error fetching Stripe transfer status:', stripeError);
      }
    }

    connection.release();

    res.json({
      success: true,
      withdrawal: {
        id: withdrawal.id,
        amount: parseFloat(withdrawal.amount),
        currency: withdrawal.currency,
        withdrawalFee: parseFloat(withdrawal.withdrawal_fee),
        totalAmount: parseFloat(withdrawal.total_amount),
        status: withdrawal.status,
        referenceId: withdrawal.reference_id,
        stripePayoutId: withdrawal.stripe_payout_id,
        failureReason: withdrawal.failure_reason,
        estimatedArrival: withdrawal.estimated_arrival,
        completedAt: withdrawal.completed_at,
        bankAccount: {
          bankName: withdrawal.bank_name,
          accountNumber: withdrawal.account_number,
          accountName: withdrawal.account_name
        },
        createdAt: withdrawal.created_at,
        updatedAt: withdrawal.updated_at
      }
    });

  } catch (error) {
    console.error('Get withdrawal status error:', error);
    res.status(500).json({ error: 'Failed to get withdrawal status' });
  }
});

// Get withdrawal history
app.get('/api/withdrawals', authenticateToken, async (req, res) => {
  try {
    const { page = 1, limit = 10 } = req.query;
    const offset = (page - 1) * limit;

    const connection = await pool.getConnection();
    const safeLimit = Number(limit) || 10;
    const safeOffset = Number(offset) || 0;
    
    const [withdrawals] = await connection.execute(
      `SELECT w.*, ba.bank_name, ba.account_number, ba.account_name
       FROM withdrawals w
       JOIN bank_accounts ba ON w.bank_account_id = ba.id
       WHERE w.user_id = ?
       ORDER BY w.created_at DESC
       LIMIT ${safeLimit} OFFSET ${safeOffset}`,
      [req.user.id]
    );

    const [total] = await connection.execute(
      'SELECT COUNT(*) as count FROM withdrawals WHERE user_id = ?',
      [req.user.id]
    );

    connection.release();

    res.json({
      success: true,
      withdrawals: withdrawals.map(withdrawal => ({
        id: withdrawal.id,
        amount: parseFloat(withdrawal.amount),
        currency: withdrawal.currency,
        withdrawalFee: parseFloat(withdrawal.withdrawal_fee),
        totalAmount: parseFloat(withdrawal.total_amount),
        status: withdrawal.status,
        referenceId: withdrawal.reference_id,
        description: withdrawal.description,
        bankAccount: {
          bankName: withdrawal.bank_name,
          accountNumber: withdrawal.account_number,
          accountName: withdrawal.account_name
        },
        createdAt: withdrawal.created_at,
        updatedAt: withdrawal.updated_at
      })),
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total: total[0].count,
        totalPages: Math.ceil(total[0].count / limit)
      }
    });
  } catch (error) {
    console.error('Get withdrawal history error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// ==================== CRYPTO ENDPOINTS ====================

// Get crypto transactions
app.get('/api/crypto/transactions', authenticateToken, async (req, res) => {
  try {
    const { page = 1, limit = 20, type } = req.query;
    const offset = (page - 1) * limit;

    const connection = await pool.getConnection();
    
    let query = 'SELECT * FROM crypto_transactions WHERE user_id = ?';
    let params = [req.user.id];

    if (type) {
      query += ' AND transaction_type = ?';
      params.push(type);
    }

    const safeLimit = Number(limit) || 20;
    const safeOffset = Number(offset) || 0;
    
    query += ` ORDER BY created_at DESC LIMIT ${safeLimit} OFFSET ${safeOffset}`;

    const [transactions] = await connection.execute(query, params);
    connection.release();

    res.json({
      success: true,
      transactions: transactions.map(transaction => ({
        id: transaction.id,
        cryptoType: transaction.crypto_type,
        transactionType: transaction.transaction_type,
        amount: parseFloat(transaction.amount),
        pricePerUnit: parseFloat(transaction.price_per_unit),
        totalValue: parseFloat(transaction.total_value),
        status: transaction.status,
        transactionHash: transaction.transaction_hash,
        createdAt: transaction.created_at
      }))
    });
  } catch (error) {
    console.error('Get crypto transactions error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Buy crypto
app.post('/api/crypto/buy', authenticateToken, async (req, res) => {
  try {
    const { cryptoType, amount, pricePerUnit, walletId } = req.body;

    if (!cryptoType || !amount || !pricePerUnit || !walletId) {
      return res.status(400).json({ error: 'Crypto type, amount, price per unit, and wallet ID are required' });
    }

    if (amount <= 0 || pricePerUnit <= 0) {
      return res.status(400).json({ error: 'Amount and price must be greater than 0' });
    }

    const connection = await pool.getConnection();

    // Verify wallet belongs to user
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND wallet_type = "crypto"',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Crypto wallet not found' });
    }

    const totalValue = amount * pricePerUnit;

    // Check if user has enough balance in fiat wallet
    const [fiatWallets] = await connection.execute(
      'SELECT * FROM wallets WHERE user_id = ? AND wallet_type = "fiat" AND currency = "USD" AND is_active = TRUE',
      [req.user.id]
    );

    if (fiatWallets.length === 0) {
      connection.release();
      return res.status(400).json({ error: 'No active fiat wallet found' });
    }

    const fiatWallet = fiatWallets[0];

    if (parseFloat(fiatWallet.balance) < totalValue) {
      connection.release();
      return res.status(400).json({ error: 'Insufficient balance' });
    }

    // Start transaction
    await connection.beginTransaction();

    try {
      // Deduct from fiat wallet
      await connection.execute(
        'UPDATE wallets SET balance = balance - ? WHERE id = ?',
        [totalValue, fiatWallet.id]
      );

      // Add to crypto wallet
      await connection.execute(
        'UPDATE wallets SET balance = balance + ? WHERE id = ?',
        [amount, walletId]
      );

      // Create crypto transaction record
      await connection.execute(
        'INSERT INTO crypto_transactions (user_id, crypto_type, transaction_type, amount, price_per_unit, total_value, wallet_id, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
        [req.user.id, cryptoType, 'buy', amount, pricePerUnit, totalValue, walletId, 'completed']
      );

      await connection.commit();
      connection.release();

      res.json({
        success: true,
        message: 'Crypto purchased successfully',
        totalValue
      });
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  } catch (error) {
    console.error('Buy crypto error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Sell crypto
app.post('/api/crypto/sell', authenticateToken, async (req, res) => {
  try {
    const { cryptoType, amount, pricePerUnit, walletId } = req.body;

    if (!cryptoType || !amount || !pricePerUnit || !walletId) {
      return res.status(400).json({ error: 'Crypto type, amount, price per unit, and wallet ID are required' });
    }

    if (amount <= 0 || pricePerUnit <= 0) {
      return res.status(400).json({ error: 'Amount and price must be greater than 0' });
    }

    const connection = await pool.getConnection();

    // Verify wallet belongs to user
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND wallet_type = "crypto"',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Crypto wallet not found' });
    }

    const wallet = wallets[0];

    if (parseFloat(wallet.balance) < amount) {
      connection.release();
      return res.status(400).json({ error: 'Insufficient crypto balance' });
    }

    const totalValue = amount * pricePerUnit;

    // Get fiat wallet
    const [fiatWallets] = await connection.execute(
      'SELECT * FROM wallets WHERE user_id = ? AND wallet_type = "fiat" AND currency = "USD" AND is_active = TRUE',
      [req.user.id]
    );

    if (fiatWallets.length === 0) {
      connection.release();
      return res.status(400).json({ error: 'No active fiat wallet found' });
    }

    const fiatWallet = fiatWallets[0];

    // Start transaction
    await connection.beginTransaction();

    try {
      // Deduct from crypto wallet
      await connection.execute(
        'UPDATE wallets SET balance = balance - ? WHERE id = ?',
        [amount, walletId]
      );

      // Add to fiat wallet
      await connection.execute(
        'UPDATE wallets SET balance = balance + ? WHERE id = ?',
        [totalValue, fiatWallet.id]
      );

      // Create crypto transaction record
      await connection.execute(
        'INSERT INTO crypto_transactions (user_id, crypto_type, transaction_type, amount, price_per_unit, total_value, wallet_id, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
        [req.user.id, cryptoType, 'sell', amount, pricePerUnit, totalValue, walletId, 'completed']
      );

      await connection.commit();
      connection.release();

      res.json({
        success: true,
        message: 'Crypto sold successfully',
        totalValue
      });
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  } catch (error) {
    console.error('Sell crypto error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// ==================== FLUTTERWAVE WEBHOOK HANDLERS ====================

// Flutterwave webhook endpoint for payment events
app.post('/api/webhooks/flutterwave', express.json(), async (req, res) => {
  try {
    // Verify webhook signature (Flutterwave sends verif-hash header)
    // Note: Secret hash is optional - if not set, webhook will still process but without verification
    const secretHash = process.env.FLW_SECRET_HASH;
    const signature = req.headers['verif-hash'];

    // Only verify if secret hash is configured
    if (secretHash) {
      if (!signature || signature !== secretHash) {
        console.error('⚠️ Invalid Flutterwave webhook signature');
        return res.status(401).json({ error: 'Invalid signature' });
      }
    } else {
      console.log('⚠️ FLW_SECRET_HASH not set - webhook processing without signature verification');
    }

    const event = req.body;
    console.log('📥 Flutterwave webhook event:', event);

    // Handle different event types
    if (event.event === 'charge.completed' || event.event === 'charge.successful') {
      const transaction = event.data;
      
      // Update transaction status in database
      const connection = await pool.getConnection();
      try {
        await connection.execute(
          `UPDATE transactions 
           SET status = 'completed', 
               reference_id = ?,
               updated_at = CURRENT_TIMESTAMP
           WHERE reference_id = ? OR description LIKE ?`,
          [
            transaction.tx_ref || transaction.flw_ref,
            transaction.tx_ref || transaction.flw_ref,
            `%${transaction.tx_ref || transaction.flw_ref}%`
          ]
        );

        // If this is a wallet funding, credit the wallet
        if (transaction.meta && transaction.meta.walletId) {
          await connection.execute(
            'UPDATE wallets SET balance = balance + ? WHERE id = ?',
            [transaction.amount, transaction.meta.walletId]
          );
        }

        await connection.commit();
        console.log('✅ Transaction updated from Flutterwave webhook');
      } catch (dbError) {
        await connection.rollback();
        console.error('❌ Database update error:', dbError);
      } finally {
        connection.release();
      }
    }

    // Handle transfer events
    if (event.event === 'transfer.completed' || event.event === 'transfer.successful') {
      const transfer = event.data;
      
      const connection = await pool.getConnection();
      try {
        // Update withdrawal status
        await connection.execute(
          `UPDATE withdrawals 
           SET status = 'completed',
               provider_reference = ?,
               updated_at = CURRENT_TIMESTAMP
           WHERE reference_id = ? OR provider_reference = ?`,
          [
            transfer.reference || transfer.id,
            transfer.reference || transfer.id,
            transfer.reference || transfer.id
          ]
        );

        await connection.commit();
        console.log('✅ Withdrawal updated from Flutterwave webhook');
      } catch (dbError) {
        await connection.rollback();
        console.error('❌ Withdrawal update error:', dbError);
      } finally {
        connection.release();
      }
    }

    res.status(200).json({ status: 'success', message: 'Webhook processed' });
  } catch (error) {
    console.error('❌ Flutterwave webhook error:', error);
    res.status(500).json({ error: 'Webhook processing failed' });
  }
});

// ==================== STRIPE WEBHOOK HANDLERS ====================

// Webhook endpoint for Stripe events (especially payout updates)
app.post('/api/webhooks/stripe-connect', express.raw({ type: 'application/json' }), async (req, res) => {
  const sig = req.headers['stripe-signature'];
  const endpointSecret = process.env.STRIPE_CONNECT_WEBHOOK_SECRET || 'whsec_test_secret';

  let event;

  try {
    event = stripe.webhooks.constructEvent(req.body, sig, endpointSecret);
  } catch (err) {
    console.error('⚠️ Webhook signature verification failed:', err.message);
    return res.status(400).send(`Webhook Error: ${err.message}`);
  }

  console.log('📥 Stripe webhook received:', event.type);

  try {
    const connection = await pool.getConnection();

    switch (event.type) {
      case 'transfer.created':
        console.log('✅ Transfer created:', event.data.object.id);
        // Transfer created - already handled in withdrawal endpoint
        break;

      case 'transfer.updated':
        await handleTransferUpdated(connection, event.data.object);
        break;

      case 'transfer.paid':
        await handleTransferPaid(connection, event.data.object);
        break;

      case 'transfer.failed':
        await handleTransferFailed(connection, event.data.object);
        break;

      case 'account.updated':
        await handleAccountUpdated(connection, event.data.object);
        break;

      case 'payout.created':
      case 'payout.updated':
      case 'payout.paid':
      case 'payout.failed':
        // Note: These are for Express/Custom accounts, we're using transfers to Standard accounts
        console.log(`📦 Payout event: ${event.type} - ${event.data.object.id}`);
        break;

      default:
        console.log(`🔔 Unhandled event type: ${event.type}`);
    }

    connection.release();
    res.json({ received: true });

  } catch (error) {
    console.error('❌ Webhook processing error:', error);
    res.status(500).json({ error: 'Webhook processing failed' });
  }
});

// Handle transfer status updates
async function handleTransferUpdated(connection, transfer) {
  try {
    console.log(`🔄 Transfer updated: ${transfer.id} - Status: ${transfer.status || 'unknown'}`);
    
    // Update withdrawal status in database
    await connection.execute(
      `UPDATE withdrawals SET 
        status = ?, updated_at = CURRENT_TIMESTAMP
       WHERE stripe_payout_id = ?`,
      [transfer.status === 'in_transit' ? 'processing' : 'pending', transfer.id]
    );

    // Update transaction status
    await connection.execute(
      `UPDATE transactions SET 
        status = ?, updated_at = CURRENT_TIMESTAMP
       WHERE stripe_payout_id = ?`,
      [transfer.status === 'in_transit' ? 'processing' : 'pending', transfer.id]
    );

  } catch (error) {
    console.error('❌ Error handling transfer update:', error);
  }
}

// Handle successful transfer completion
async function handleTransferPaid(connection, transfer) {
  try {
    console.log(`✅ Transfer completed: ${transfer.id} - Amount: ${transfer.amount}`);
    
    await connection.beginTransaction();

    try {
      // Update withdrawal status to completed
      await connection.execute(
        `UPDATE withdrawals SET 
          status = 'completed', completed_at = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP
         WHERE stripe_payout_id = ?`,
        [transfer.id]
      );

      // Update transaction status to completed
      await connection.execute(
        `UPDATE transactions SET 
          status = 'completed', updated_at = CURRENT_TIMESTAMP
         WHERE stripe_payout_id = ?`,
        [transfer.id]
      );

      await connection.commit();
      console.log(`🎉 Withdrawal ${transfer.id} marked as completed`);

    } catch (error) {
      await connection.rollback();
      throw error;
    }

  } catch (error) {
    console.error('❌ Error handling transfer completion:', error);
  }
}

// Handle failed transfers
async function handleTransferFailed(connection, transfer) {
  try {
    console.log(`❌ Transfer failed: ${transfer.id} - Failure message: ${transfer.failure_message || 'Unknown error'}`);
    
    await connection.beginTransaction();

    try {
      // Get withdrawal details to reverse the wallet deduction
      const [withdrawals] = await connection.execute(
        'SELECT * FROM withdrawals WHERE stripe_payout_id = ?',
        [transfer.id]
      );

      if (withdrawals.length > 0) {
        const withdrawal = withdrawals[0];

        // Reverse the wallet deduction (add money back)
        await connection.execute(
          'UPDATE wallets SET balance = balance + ? WHERE id = ?',
          [withdrawal.total_amount, withdrawal.wallet_id]
        );

        // Update withdrawal status to failed
        await connection.execute(
          `UPDATE withdrawals SET 
            status = 'failed', failure_reason = ?, updated_at = CURRENT_TIMESTAMP
           WHERE stripe_payout_id = ?`,
          [transfer.failure_message || 'Transfer failed', transfer.id]
        );

        // Update transaction status to failed
        await connection.execute(
          `UPDATE transactions SET 
            status = 'failed', updated_at = CURRENT_TIMESTAMP
           WHERE stripe_payout_id = ?`,
          [transfer.id]
        );

        console.log(`🔄 Wallet balance restored for failed withdrawal: ${withdrawal.total_amount} ${withdrawal.currency}`);
      }

      await connection.commit();

    } catch (error) {
      await connection.rollback();
      throw error;
    }

  } catch (error) {
    console.error('❌ Error handling transfer failure:', error);
  }
}

// Handle Stripe account updates (for capability changes)
async function handleAccountUpdated(connection, account) {
  try {
    console.log(`🔄 Account updated: ${account.id} - Charges: ${account.charges_enabled}, Payouts: ${account.payouts_enabled}`);
    
    // Update local database with latest account capabilities
    await connection.execute(
      `UPDATE stripe_connect_accounts SET 
        charges_enabled = ?, payouts_enabled = ?, details_submitted = ?,
        requirements_currently_due = ?, requirements_eventually_due = ?, requirements_past_due = ?,
        updated_at = CURRENT_TIMESTAMP
       WHERE stripe_account_id = ?`,
      [
        account.charges_enabled,
        account.payouts_enabled,
        account.details_submitted,
        JSON.stringify(account.requirements?.currently_due || []),
        JSON.stringify(account.requirements?.eventually_due || []),
        JSON.stringify(account.requirements?.past_due || []),
        account.id
      ]
    );

  } catch (error) {
    console.error('❌ Error handling account update:', error);
  }
}

// ==================== BILL PAYMENT ENDPOINTS ====================

// Get all bill providers (for debugging)
app.get('/api/bill-payments/providers', authenticateToken, async (req, res) => {
  try {
    const connection = await pool.getConnection();
    const [providers] = await connection.execute('SELECT * FROM bill_providers');
    connection.release();

    res.json({
      success: true,
      providers: providers.map(p => ({
        id: p.id,
        name: p.name,
        code: p.code,
        category_id: p.category_id,
        currency: p.currency
      }))
    });
  } catch (error) {
    console.error('Get providers error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Get bill categories
app.get('/api/bill-categories', async (req, res) => {
  try {
    const connection = await pool.getConnection();
    const [categories] = await connection.execute(
      'SELECT * FROM bill_categories WHERE is_active = TRUE ORDER BY name'
    );
    connection.release();

    res.json({
      success: true,
      data: categories
    });
  } catch (error) {
    console.error('Get bill categories error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Get providers for a bill category from Stripe
app.get('/api/bill-categories/:categoryId/providers', async (req, res) => {
  try {
    const { categoryId } = req.params;
    const { currency, serviceType } = req.query; // Optional currency and service type filter
    
    // Map category ID to service type
    const categoryServiceMap = {
      1: 'electricity',
      2: 'cable_tv', 
      3: 'water',
      4: 'internet',
      5: 'streaming'
    };
    
    const serviceTypeFromCategory = categoryServiceMap[categoryId] || serviceType;
    
    // Default to NGN if no currency specified
    const targetCurrency = currency || 'NGN';
    
    console.log('🌍 Fetching Stripe providers for category:', categoryId, 'currency:', targetCurrency, 'serviceType:', serviceTypeFromCategory);
    
    // Use Stripe to fetch providers
    const StripeBillProviders = require('./stripe-bill-providers');
    const stripeProviders = new StripeBillProviders();
    
    const response = await stripeProviders.getBillProviders(targetCurrency, serviceTypeFromCategory);
    
    // Filter by currency if specified
    let providers = response.data;
    if (targetCurrency) {
      providers = providers.filter(provider => provider.currency === targetCurrency);
    }
    
    console.log('✅ Stripe providers found:', providers.length);

    res.json({
      success: true,
      data: providers
    });
  } catch (error) {
    console.error('Get Stripe bill providers error:', error);
    
    // Fallback to database if Stripe fails
    try {
      const { categoryId } = req.params;
      const { currency } = req.query;
      
      const connection = await pool.getConnection();
      
      let query = 'SELECT * FROM bill_providers WHERE category_id = ? AND is_active = TRUE';
      let params = [categoryId];
      
      if (currency) {
        query += ' AND currency = ?';
        params.push(currency);
      }
      
      query += ' ORDER BY name';
      
      const [providers] = await connection.execute(query, params);
      connection.release();

      console.log('🔄 Fallback to database providers:', providers.length);

      res.json({
        success: true,
        data: providers
      });
    } catch (fallbackError) {
      console.error('Fallback error:', fallbackError);
      res.status(500).json({ error: 'Internal server error' });
    }
  }
});

// Verify customer number
app.post('/api/bill-payments/verify', authenticateToken, async (req, res) => {
  try {
    const { providerId, customerNumber } = req.body;

    if (!providerId || !customerNumber) {
      return res.status(400).json({ error: 'Provider ID and customer number are required' });
    }

    // Mock verification - in real implementation, this would call the provider's API
    // For demo purposes, accept any customer number with at least 6 characters
    const isValid = customerNumber.length >= 6;
    const customerInfo = isValid ? {
      customerName: `Customer ${customerNumber.slice(-4)}`,
      address: '123 Sample Street, Lagos, Nigeria',
      isValid: true
    } : {
      isValid: false,
      message: 'Customer number must be at least 6 characters long'
    };

    res.json({
      success: true,
      data: customerInfo
    });
  } catch (error) {
    console.error('Verify customer error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Get bill amount
app.post('/api/bill-payments/amount', authenticateToken, async (req, res) => {
  try {
    const { providerId, customerNumber } = req.body;

    if (!providerId || !customerNumber) {
      return res.status(400).json({ error: 'Provider ID and customer number are required' });
    }

    // Get provider details to determine currency
    console.log('🔍 Looking for provider (amount):', providerId);
    const connection = await pool.getConnection();
    const [providers] = await connection.execute(
      'SELECT * FROM bill_providers WHERE id = ? OR code = ?',
      [providerId, providerId]
    );
    connection.release();

    if (providers.length === 0) {
      console.log('❌ Provider not found (amount):', providerId);
      return res.status(404).json({ error: `Provider not found: ${providerId}` });
    }

    const provider = providers[0];
    const currency = provider.currency || 'NGN';

    // Mock bill amount - in real implementation, this would call the provider's API
    const amount = Math.floor(Math.random() * 50000) + 1000; // Random amount between 1000-51000

    res.json({
      success: true,
      data: {
        amount,
        currency,
        dueDate: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString(), // 30 days from now
        billNumber: `BILL${Date.now()}`
      }
    });
  } catch (error) {
    console.error('Get bill amount error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Create payment intent (Stripe integration)
app.post('/api/bill-payments/create-intent', authenticateToken, async (req, res) => {
  try {
    const { billType, providerId, customerNumber, amount, paymentMethodId, customerInfo, currency = 'NGN' } = req.body;

    if (!billType || !providerId || !customerNumber || !amount || !paymentMethodId) {
      return res.status(400).json({ error: 'All fields are required' });
    }

    // Check if this is a VTPass service (has special format)
    const isVTPassService = providerId.includes('-electric') || 
                           providerId.includes('-airtime') || 
                           providerId.includes('dstv') || 
                           providerId.includes('gotv') || 
                           providerId.includes('startimes');

    let provider;
    let billCurrency = currency || 'NGN';

    if (isVTPassService) {
      // For VTPass services, create a mock provider
      provider = {
        id: providerId,
        name: providerId.replace('-', ' ').toUpperCase(),
        currency: 'NGN',
        type: 'vtpass'
      };
      billCurrency = 'NGN'; // VTPass only supports NGN
    } else {
      // Get provider details from Stripe
      const StripeBillProviders = require('./stripe-bill-providers.js');
      const stripeProviders = new StripeBillProviders();
      
      try {
        const providerResponse = await stripeProviders.getProviderDetails(providerId);
        provider = providerResponse.data;
      } catch (stripeError) {
        console.error('Stripe provider fetch failed, falling back to database:', stripeError);
        
        // Fallback to database
        console.log('🔍 Looking for provider (create-intent):', providerId);
        const connection = await pool.getConnection();
        const [providers] = await connection.execute(
          'SELECT * FROM bill_providers WHERE id = ? OR code = ?',
          [providerId, providerId]
        );
        connection.release();

        if (providers.length === 0) {
          console.log('❌ Provider not found (create-intent):', providerId);
          return res.status(404).json({ error: `Provider not found: ${providerId}` });
        }
        provider = providers[0];
      }
      
      // Use provider's currency or default to NGN
      billCurrency = provider.currency || currency || 'NGN';
    }
    
    // Calculate processing fee (1% of amount)
    const processingFee = amount * 0.01;
    const totalAmount = amount + processingFee;

    let paymentIntentId;
    let clientSecret;

    if (paymentMethodId.startsWith('wallet_')) {
      // For wallet payments, create a mock payment intent ID
      paymentIntentId = `pi_wallet_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      clientSecret = `pi_${paymentIntentId}_secret_${Math.random().toString(36).substr(2, 9)}`;
    } else {
      // For card payments, create payment intent through YOUR platform
      try {
        const paymentIntent = await stripe.paymentIntents.create({
          amount: Math.round(totalAmount * 100), // Convert to cents
          currency: billCurrency.toLowerCase(),
          payment_method_types: ['card'],
          metadata: {
            billType,
            providerId,
            customerNumber,
            userId: req.user.id.toString(),
            customerName: customerInfo?.customerName || 'Unknown',
            currency: billCurrency,
            bill_type: 'utility_payment',
            platform_payment: 'true' // Mark as platform payment
          },
          description: `Bill payment for ${customerNumber}`,
          receipt_email: req.user.email,
        });

        paymentIntentId = paymentIntent.id;
        clientSecret = paymentIntent.client_secret;
      } catch (stripeError) {
        console.error('Payment intent creation failed:', stripeError);
        throw stripeError;
      }
    }

    res.json({
      success: true,
      data: {
        paymentIntentId,
        clientSecret,
        amount: totalAmount,
        currency: billCurrency,
        processingFee
      }
    });
  } catch (error) {
    console.error('Create payment intent error:', error);
    res.status(500).json({ 
      error: 'Failed to create payment intent',
      details: error.message 
    });
  }
});

// Process bill payment
app.post('/api/bill-payments/process', authenticateToken, async (req, res) => {
  try {
    console.log('🔄 Server: Processing bill payment request');
    console.log('📦 Server: Request body received:', JSON.stringify(req.body, null, 2));
    
    const connection = await pool.getConnection();
    await connection.beginTransaction();

    const {
      billType,
      providerId,
      customerNumber,
      amount,
      paymentMethodId,
      paymentIntentId,
      customerInfo,
      currency = 'NGN'
    } = req.body;

    console.log('📋 Server: Extracted fields:', {
      billType,
      providerId,
      customerNumber,
      amount,
      paymentMethodId,
      paymentIntentId,
      hasCustomerInfo: !!customerInfo
    });

    if (!billType || !providerId || !customerNumber || !amount || !paymentIntentId) {
      console.log('❌ Server: Missing required fields');
      console.log('   billType:', !!billType);
      console.log('   providerId:', !!providerId);
      console.log('   customerNumber:', !!customerNumber);
      console.log('   amount:', !!amount);
      console.log('   paymentIntentId:', !!paymentIntentId);
      return res.status(400).json({ error: 'All required fields are missing' });
    }

    // Handle different payment methods
    let paymentIntent;
    
    if (paymentMethodId.startsWith('wallet_')) {
      // For wallet payments, we don't need to confirm with Stripe
      // Just mark the payment as successful
      paymentIntent = { status: 'succeeded' };
    } else {
      // For card payments, confirm with Stripe
      paymentIntent = await stripe.paymentIntents.confirm(paymentIntentId, {
        payment_method: paymentMethodId,
      });
    }

    if (paymentIntent.status !== 'succeeded') {
      await connection.rollback();
      connection.release();
      return res.status(400).json({ 
        error: 'Payment failed',
        details: paymentIntent.last_payment_error?.message || 'Payment could not be processed'
      });
    }

    // Calculate processing fee (1% of amount)
    const processingFee = amount * 0.01;
    const totalAmount = amount + processingFee;

    // Generate transaction ID
    const transactionId = `BILL${Date.now()}${Math.random().toString(36).substr(2, 6)}`;

    // Get provider info
    console.log('🔍 Looking for provider:', providerId);
    const [providers] = await connection.execute(
      'SELECT * FROM bill_providers WHERE id = ? OR code = ?',
      [providerId, providerId]
    );

    console.log('🔍 Provider lookup results:', {
      searchTerm: providerId,
      foundCount: providers.length,
      providers: providers.map(p => ({ id: p.id, code: p.code, name: p.name }))
    });

    if (providers.length === 0) {
      await connection.rollback();
      connection.release();
      console.log('❌ Provider not found:', providerId);
      return res.status(404).json({ error: `Provider not found: ${providerId}` });
    }

    const provider = providers[0];
    
    // Ensure provider.id is a number
    if (typeof provider.id !== 'number') {
      await connection.rollback();
      connection.release();
      console.log('❌ Invalid provider ID type:', typeof provider.id, provider.id);
      return res.status(500).json({ 
        error: 'Invalid provider configuration',
        details: 'Provider ID must be a number'
      });
    }
    
    // Use provider's currency or default to NGN
    const billCurrency = provider.currency || currency || 'NGN';
    
    console.log('🔍 Provider found:', {
      providerId: provider.id,
      providerCode: provider.code,
      providerName: provider.name,
      providerCurrency: provider.currency
    });
    
    console.log('💰 Bill Payment Details:', {
      billType,
      providerName: provider.name,
      billCurrency,
      amount,
      totalAmount,
      paymentMethodId,
      userCurrency: currency
    });

    // Create bill payment record
    const insertValues = [
      req.user.id,
      provider.id, // Use the actual provider ID from database
      customerNumber,
      customerInfo?.customerName || 'Unknown',
      customerInfo?.address || '',
      amount,
      billCurrency,
      processingFee,
      totalAmount,
      'completed',
      paymentIntentId,
      transactionId,
      paymentMethodId
    ];
    
    console.log('📝 Inserting bill payment with values:', {
      userId: insertValues[0],
      providerId: insertValues[1],
      customerNumber: insertValues[2],
      customerName: insertValues[3],
      amount: insertValues[5],
      currency: insertValues[6],
      totalAmount: insertValues[8]
    });
    
    const [result] = await connection.execute(
      `INSERT INTO bill_payments (
        user_id, provider_id, customer_number, customer_name, customer_address,
        amount, currency, processing_fee, total_amount, status,
        stripe_payment_intent_id, transaction_id, payment_method
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      insertValues
    );

    const paymentId = result.insertId;

    // Handle money deduction based on payment method
    if (paymentMethodId.startsWith('wallet_')) {
      // Extract wallet ID from payment method
      const walletId = paymentMethodId.replace('wallet_', '');
      
      // Get wallet details to check currency - also check if wallet is active (not frozen)
      const [wallets] = await connection.execute(
        'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND is_active = TRUE',
        [walletId, req.user.id]
      );

      if (wallets.length === 0) {
        await connection.rollback();
        connection.release();
        return res.status(400).json({ 
          success: false,
          error: 'Wallet not found or frozen',
          details: 'Selected wallet does not exist or is frozen. Please unfreeze the wallet to perform bill payments.'
        });
      }

      const wallet = wallets[0];
      
      console.log('💳 Wallet Details:', {
        walletId,
        walletCurrency: wallet.currency,
        walletBalance: wallet.balance,
        billCurrency,
        totalAmount
      });
      
      // Check if wallet currency matches bill currency
      if (wallet.currency !== billCurrency) {
        await connection.rollback();
        connection.release();
        return res.status(400).json({ 
          error: 'Currency mismatch',
          details: `Bill requires ${billCurrency} but wallet has ${wallet.currency}. Please use a wallet with ${billCurrency} currency.`
        });
      }
      
      // Deduct money from user's wallet
      const [walletUpdate] = await connection.execute(
        `UPDATE wallets 
         SET balance = balance - ? 
         WHERE id = ? AND user_id = ? AND balance >= ? AND currency = ?`,
        [totalAmount, walletId, req.user.id, totalAmount, billCurrency]
      );

      if (walletUpdate.affectedRows === 0) {
        await connection.rollback();
        connection.release();
        return res.status(400).json({ 
          error: 'Insufficient wallet balance or currency mismatch',
          details: `Please ensure you have sufficient ${billCurrency} funds in your wallet`
        });
      }

      console.log(`💰 Deducted ${totalAmount} ${billCurrency} from wallet ${walletId}`);
    } else {
      // For card payments, Stripe handles the deduction
      // We just need to ensure the payment was successful
      if (paymentIntent.status !== 'succeeded') {
        await connection.rollback();
        connection.release();
        return res.status(400).json({
          error: 'Payment failed',
          details: paymentIntent.last_payment_error?.message || 'Payment could not be processed'
        });
      }
      
      console.log(`💳 Card payment processed via Stripe: ${totalAmount} NGN`);
    }

    // Create transaction record
    await connection.execute(
      `INSERT INTO transactions (
        user_id, transaction_type, amount, currency, status, reference_id, description
      ) VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [
        req.user.id,
        'withdrawal', // Use 'withdrawal' instead of 'bill_payment' to avoid ENUM issues
        totalAmount,
        billCurrency,
        'completed',
        transactionId,
        `${provider.name} bill payment for ${customerNumber}`
      ]
    );

    await connection.commit();
    connection.release();

    res.json({
      success: true,
      message: 'Bill payment processed successfully',
      data: {
        paymentId,
        transactionId,
        amount: totalAmount,
        currency: billCurrency,
        provider: provider.name,
        customerNumber,
        status: 'completed'
      }
    });
  } catch (error) {
    console.error('Process bill payment error:', error);
    res.status(500).json({ 
      error: 'Payment processing failed',
      details: error.message 
    });
  }
});

// Get payment history
app.get('/api/bill-payments/history', authenticateToken, async (req, res) => {
  try {
    const { page = 1, limit = 10 } = req.query;
    const offset = (page - 1) * limit;

    const connection = await pool.getConnection();
    const safeLimit = Number(limit) || 10;
    const safeOffset = Number(offset) || 0;
    
    const [payments] = await connection.execute(
      `SELECT bp.*, bp_providers.name as provider_name, bp_providers.code as provider_code
       FROM bill_payments bp
       JOIN bill_providers bp_providers ON bp.provider_id = bp_providers.id
       WHERE bp.user_id = ?
       ORDER BY bp.created_at DESC
       LIMIT ${safeLimit} OFFSET ${safeOffset}`,
      [req.user.id]
    );

    const [total] = await connection.execute(
      'SELECT COUNT(*) as count FROM bill_payments WHERE user_id = ?',
      [req.user.id]
    );

    connection.release();

    res.json({
      success: true,
      data: payments,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total: total[0].count,
        totalPages: Math.ceil(total[0].count / limit)
      }
    });
  } catch (error) {
    console.error('Get payment history error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Stripe webhook endpoint
app.post('/api/webhooks/stripe', express.raw({ type: 'application/json' }), async (req, res) => {
  const sig = req.headers['stripe-signature'];
  const endpointSecret = process.env.STRIPE_WEBHOOK_SECRET || 'whsec_test_webhook_secret';

  let event;

  try {
    event = stripe.webhooks.constructEvent(req.body, sig, endpointSecret);
  } catch (err) {
    console.error('Webhook signature verification failed:', err.message);
    return res.status(400).send(`Webhook Error: ${err.message}`);
  }

  // Handle the event
  switch (event.type) {
    case 'payment_intent.succeeded':
      const paymentIntent = event.data.object;
      console.log('PaymentIntent was successful:', paymentIntent.id);
      
      // Update payment status in database if needed
      try {
        const connection = await pool.getConnection();
        await connection.execute(
          'UPDATE bill_payments SET status = ? WHERE stripe_payment_intent_id = ?',
          ['completed', paymentIntent.id]
        );
        connection.release();
      } catch (error) {
        console.error('Error updating payment status:', error);
      }
      break;
      
    case 'payment_intent.payment_failed':
      const failedPayment = event.data.object;
      console.log('PaymentIntent failed:', failedPayment.id);
      
      // Update payment status in database
      try {
        const connection = await pool.getConnection();
        await connection.execute(
          'UPDATE bill_payments SET status = ? WHERE stripe_payment_intent_id = ?',
          ['failed', failedPayment.id]
        );
        connection.release();
      } catch (error) {
        console.error('Error updating payment status:', error);
      }
      break;
      
    default:
      console.log(`Unhandled event type ${event.type}`);
  }

  res.json({ received: true });
});

// Get payment receipt
app.get('/api/bill-payments/:paymentId/receipt', authenticateToken, async (req, res) => {
  try {
    const { paymentId } = req.params;

    // Check if this is a VTPass transaction (starts with TXN_)
    if (paymentId.startsWith('TXN_')) {
      // Look for VTPass payment in database
      const connection = await pool.getConnection();
      const [payments] = await connection.execute(
        `SELECT * FROM bill_payments 
         WHERE transaction_id = ? AND user_id = ? AND payment_method = 'vtpass'`,
        [paymentId, req.user.id]
      );
      connection.release();

      if (payments.length > 0) {
        const payment = payments[0];
        return res.json({
          success: true,
          data: {
            transactionId: payment.transaction_id,
            amount: payment.amount,
            currency: payment.currency,
            providerName: payment.provider_id, // This will be the serviceId
            customerName: payment.customer_name,
            customerNumber: payment.customer_number,
            paymentDate: payment.created_at,
            status: payment.status,
            processingFee: payment.processing_fee,
            totalAmount: payment.total_amount,
            paymentMethod: 'VTPass',
            receiptUrl: payment.receipt_url
          }
        });
      } else {
        // Fallback to mock receipt if not found in database
        const mockReceipt = {
          transactionId: paymentId,
          amount: 3500, // Mock amount
          currency: 'NGN',
          providerName: 'VTPass Service',
          customerName: 'Adebayo Johnson',
          customerNumber: '1234567890',
          paymentDate: new Date().toISOString(),
          status: 'successful',
          processingFee: 35,
          totalAmount: 3535,
          paymentMethod: 'Wallet',
          receiptUrl: null
        };

        return res.json({
          success: true,
          data: mockReceipt
        });
      }
    }

    // For regular payments, check database
    const connection = await pool.getConnection();
    const [payments] = await connection.execute(
      `SELECT bp.*, bp_providers.name as provider_name, bp_providers.code as provider_code
       FROM bill_payments bp
       JOIN bill_providers bp_providers ON bp.provider_id = bp_providers.id
       WHERE bp.id = ? AND bp.user_id = ?`,
      [paymentId, req.user.id]
    );

    connection.release();

    if (payments.length === 0) {
      return res.status(404).json({ error: 'Payment not found' });
    }

    const payment = payments[0];

    res.json({
      success: true,
      data: {
        transactionId: payment.transaction_id,
        amount: payment.amount,
        currency: payment.currency,
        providerName: payment.provider_name,
        customerName: payment.customer_name,
        customerNumber: payment.customer_number,
        paymentDate: payment.created_at,
        status: payment.status,
        processingFee: payment.processing_fee,
        totalAmount: payment.total_amount,
        paymentMethod: payment.payment_method,
        receiptUrl: payment.receipt_url
      }
    });
  } catch (error) {
    console.error('Get payment receipt error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Get saved billers
app.get('/api/bill-payments/saved-billers', authenticateToken, async (req, res) => {
  try {
    // Mock saved billers - in real implementation, this would be stored in database
    const savedBillers = [
      {
        id: 1,
        name: 'My Electricity Bill',
        providerName: 'Eko Electricity (EKEDC)',
        customerNumber: '1234567890',
        customerName: 'John Doe',
        billType: 'electricity'
      },
      {
        id: 2,
        name: 'DSTV Subscription',
        providerName: 'DSTV',
        customerNumber: 'DSTV123456',
        customerName: 'John Doe',
        billType: 'cable'
      }
    ];

    res.json({
      success: true,
      data: savedBillers
    });
  } catch (error) {
    console.error('Get saved billers error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Save biller
app.post('/api/bill-payments/save-biller', authenticateToken, async (req, res) => {
  try {
    const { name, providerId, customerNumber, customerName, billType } = req.body;

    if (!name || !providerId || !customerNumber || !customerName || !billType) {
      return res.status(400).json({ error: 'All fields are required' });
    }

    // Mock save biller - in real implementation, this would be stored in database
    const savedBiller = {
      id: Date.now(),
      name,
      providerId,
      customerNumber,
      customerName,
      billType,
      createdAt: new Date().toISOString()
    };

    res.json({
      success: true,
      message: 'Biller saved successfully',
      data: savedBiller
    });
  } catch (error) {
    console.error('Save biller error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Delete saved biller
app.delete('/api/bill-payments/saved-billers/:billerId', authenticateToken, async (req, res) => {
  try {
    const { billerId } = req.params;

    // Mock delete biller - in real implementation, this would delete from database
    res.json({
      success: true,
      message: 'Biller deleted successfully'
    });
  } catch (error) {
    console.error('Delete saved biller error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Get payment methods for bill payment
app.get('/api/bill-payments/payment-methods', authenticateToken, async (req, res) => {
  try {
    const { currency } = req.query; // Optional currency filter
    const connection = await pool.getConnection();
    const [paymentMethods] = await connection.execute(
      'SELECT * FROM payment_methods WHERE user_id = ? AND is_active = TRUE ORDER BY is_default DESC, created_at DESC',
      [req.user.id]
    );

    // Add wallet as payment method with optional currency filter
    let walletQuery = 'SELECT * FROM wallets WHERE user_id = ? AND is_active = TRUE ORDER BY is_default DESC';
    let walletParams = [req.user.id];
    
    if (currency) {
      walletQuery = 'SELECT * FROM wallets WHERE user_id = ? AND is_active = TRUE AND currency = ? ORDER BY is_default DESC';
      walletParams.push(currency);
    }
    
    const [wallets] = await connection.execute(walletQuery, walletParams);

    const walletPaymentMethods = wallets.map(wallet => ({
      id: `wallet_${wallet.id}`,
      type: 'wallet',
      name: `${wallet.wallet_name} (${wallet.currency})`,
      details: {
        walletId: wallet.id,
        balance: wallet.balance,
        currency: wallet.currency
      },
      isDefault: wallet.is_default
    }));

    connection.release();

    const allPaymentMethods = [...walletPaymentMethods, ...paymentMethods];

    console.log('💳 Payment methods for user:', {
      userId: req.user.id,
      currencyFilter: currency,
      walletCount: wallets.length,
      totalMethods: allPaymentMethods.length
    });

    res.json({
      success: true,
      data: allPaymentMethods
    });
  } catch (error) {
    console.error('Get payment methods error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Validate payment method
app.post('/api/bill-payments/validate-payment-method', authenticateToken, async (req, res) => {
  try {
    const { paymentMethodId } = req.body;

    if (!paymentMethodId) {
      return res.status(400).json({ error: 'Payment method ID is required' });
    }

    // Mock validation - in real implementation, this would validate with Stripe
    const isValid = paymentMethodId.length > 0;

    res.json({
      success: true,
      data: {
        isValid,
        message: isValid ? 'Payment method is valid' : 'Invalid payment method'
      }
    });
  } catch (error) {
    console.error('Validate payment method error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// ==================== VTPASS INTEGRATION ENDPOINTS ====================

// Get VTPass services for Nigeria
app.get('/api/vtpass/services', authenticateToken, async (req, res) => {
  try {
    console.log('🚀 Server: Getting VTPass services...');
    
    // Add timeout to prevent hanging
    const timeoutPromise = new Promise((_, reject) => {
      setTimeout(() => reject(new Error('VTPass API timeout')), 10000);
    });
    
    const servicesPromise = vtpass.getServices();
    const result = await Promise.race([servicesPromise, timeoutPromise]);
    
    if (!result.success) {
      console.warn('⚠️ VTPass returned unsuccessful result:', result.error);
      // Return mock data instead of error for better UX
      const mockServices = vtpass.getMockServices();
      const categorizedServices = vtpass.mapServicesToCategories(mockServices);
      return res.json({
        success: true,
        data: categorizedServices
      });
    }

    // Map services to categories
    const categorizedServices = vtpass.mapServicesToCategories(result.data);
    
    console.log('✅ VTPass services categorized successfully');
    res.json({
      success: true,
      data: categorizedServices
    });
  } catch (error) {
    console.error('💥 Server: VTPass services error:', error.message);
    
    // Always return mock data on error to prevent loading state from hanging
    try {
      const mockServices = vtpass.getMockServices();
      const categorizedServices = vtpass.mapServicesToCategories(mockServices);
      console.log('✅ Returning mock VTPass services due to error');
      return res.json({
        success: true,
        data: categorizedServices
      });
    } catch (mockError) {
      console.error('💥 Failed to generate mock services:', mockError);
      // Return empty categorized structure as last resort
      return res.json({
        success: true,
        data: {
          'Airtime & Data': [],
          'Electricity': [],
          'Water': [],
          'Cable TV': []
        }
      });
    }
  }
});

// Get VTPass service variations
app.get('/api/vtpass/service-variations/:serviceId', authenticateToken, async (req, res) => {
  try {
    const { serviceId } = req.params;
    console.log('🚀 Server: Getting VTPass service variations for:', serviceId);
    
    const result = await vtpass.getServiceVariations(serviceId);
    
    if (!result.success) {
      return res.status(400).json({ 
        error: result.error || 'Failed to fetch service variations' 
      });
    }

    res.json({
      success: true,
      data: result.data
    });
  } catch (error) {
    console.error('💥 Server: VTPass service variations error:', error);
    res.status(500).json({ error: 'Failed to fetch service variations' });
  }
});

// Verify customer with VTPass
app.post('/api/vtpass/verify', authenticateToken, async (req, res) => {
  try {
    const { serviceId, customerNumber } = req.body;

    if (!serviceId || !customerNumber) {
      return res.status(400).json({ error: 'Service ID and customer number are required' });
    }

    console.log('🚀 Server: Verifying customer with VTPass:', { serviceId, customerNumber });
    const result = await vtpass.verifyCustomer(serviceId, customerNumber);
    
    if (!result.success) {
      return res.status(400).json({ 
        error: result.error || 'Customer verification failed' 
      });
    }

    res.json({
      success: true,
      data: result.data
    });
  } catch (error) {
    console.error('💥 Server: VTPass verify error:', error);
    res.status(500).json({ error: 'Failed to verify customer' });
  }
});

// Get bill amount from VTPass
app.post('/api/vtpass/bill-amount', authenticateToken, async (req, res) => {
  try {
    const { serviceId, customerNumber } = req.body;

    if (!serviceId || !customerNumber) {
      return res.status(400).json({ error: 'Service ID and customer number are required' });
    }

    console.log('🚀 Server: Getting bill amount from VTPass:', { serviceId, customerNumber });
    const result = await vtpass.getBillAmount(serviceId, customerNumber);
    
    if (!result.success) {
      return res.status(400).json({ 
        error: result.error || 'Failed to get bill amount' 
      });
    }

    res.json({
      success: true,
      data: result.data
    });
  } catch (error) {
    console.error('💥 Server: VTPass bill amount error:', error);
    res.status(500).json({ error: 'Failed to get bill amount' });
  }
});

// Process payment with VTPass
app.post('/api/vtpass/process-payment', authenticateToken, async (req, res) => {
  try {
    const {
      serviceId,
      customerNumber,
      amount,
      phone,
      reference,
      customerInfo,
      paymentMethodId
    } = req.body;

    if (!serviceId || !customerNumber || !amount) {
      return res.status(400).json({ 
        error: 'Service ID, customer number, and amount are required' 
      });
    }

    console.log('🚀 Server: Processing payment with VTPass:', { 
      serviceId, customerNumber, amount, phone, reference, paymentMethodId
    });
    
    const result = await vtpass.processPayment({
      serviceId,
      customerNumber,
      amount,
      phone: phone || '08012345678',
      reference: reference || `VT_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
    });
    
    if (!result.success) {
      return res.status(400).json({ 
        error: result.error || 'Payment processing failed' 
      });
    }

    // Save payment to database
    const connection = await pool.getConnection();
    await connection.beginTransaction();

    try {
      // Calculate processing fee (1% of amount)
      const processingFee = amount * 0.01;
      const totalAmount = amount + processingFee;

      // Generate transaction ID
      const transactionId = result.data.transaction_id || result.data.transactionId || `TXN_${Date.now()}`;

      // For VTPass payments, we need to handle the provider_id differently
      // since VTPass services are not in the bill_providers table
      let providerId = null;
      
      // Check if this service exists in bill_providers
      const [providers] = await connection.execute(
        'SELECT id FROM bill_providers WHERE code = ? OR name LIKE ?',
        [serviceId || '', `%${serviceId || ''}%`]
      );
      
      if (providers.length > 0 && providers[0].id) {
        providerId = parseInt(providers[0].id);
      } else {
        // Create a temporary provider entry for VTPass services
        try {
          const [insertResult] = await connection.execute(
            `INSERT INTO bill_providers (category_id, name, code, description, currency) 
             VALUES (1, ?, ?, ?, ?)`,
            [
              (serviceId || 'VTPass Service').replace('-', ' ').toUpperCase(), // name
              serviceId || 'vtpass-service', // code
              `VTPass ${serviceId || 'service'} service`, // description
              'NGN' // currency
            ]
          );
          providerId = insertResult?.insertId ? parseInt(insertResult.insertId) : null;
        } catch (insertError) {
          console.error('💥 Error creating provider entry:', insertError);
          // Try to get existing provider as fallback
          const [fallbackProviders] = await connection.execute(
            'SELECT id FROM bill_providers WHERE code LIKE ? LIMIT 1',
            ['%vtpass%']
          );
          if (fallbackProviders.length > 0 && fallbackProviders[0].id) {
            providerId = parseInt(fallbackProviders[0].id);
          }
        }
      }
      
      // Ensure providerId is a valid integer
      if (!providerId || isNaN(providerId)) {
        throw new Error('Failed to resolve provider ID for VTPass service');
      }
      
      // Create bill payment record for VTPass
      // Ensure all values are defined (use null instead of undefined for SQL)
      const paymentReference = reference || result.data?.requestId || result.data?.reference || null;
      const finalTransactionId = transactionId || result.data?.transaction_id || result.data?.transactionId || null;
      
      // Validate all required fields before insert
      if (!req.user?.id) {
        throw new Error('User ID is required');
      }
      if (!providerId || isNaN(providerId)) {
        throw new Error('Provider ID is invalid');
      }
      if (!customerNumber) {
        throw new Error('Customer number is required');
      }
      if (!amount || isNaN(amount)) {
        throw new Error('Amount is invalid');
      }
      
      console.log('📝 Inserting bill payment with values:', {
        userId: req.user.id,
        providerId: providerId,
        customerNumber: customerNumber || null,
        customerName: customerInfo?.customerName || 'Unknown',
        customerAddress: customerInfo?.address || null,
        amount: amount,
        currency: 'NGN',
        processingFee: processingFee || 0,
        totalAmount: totalAmount || amount,
        status: 'completed',
        reference: paymentReference,
        transactionId: finalTransactionId,
        paymentMethod: 'vtpass'
      });
      
      const [dbResult] = await connection.execute(
        `INSERT INTO bill_payments (
          user_id, provider_id, customer_number, customer_name, customer_address,
          amount, currency, processing_fee, total_amount, status,
          stripe_payment_intent_id, transaction_id, payment_method
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          req.user.id,
          providerId, // Already validated as integer
          customerNumber || null,
          customerInfo?.customerName || 'Unknown',
          customerInfo?.address || null,
          parseFloat(amount) || 0,
          'NGN', // VTPass only supports NGN
          parseFloat(processingFee) || 0,
          parseFloat(totalAmount) || parseFloat(amount) || 0,
          'completed',
          paymentReference, // Use reference as payment intent ID (null if undefined)
          finalTransactionId, // Use transaction ID (null if undefined)
          'vtpass' // Mark as VTPass payment
        ]
      );

      const paymentId = dbResult.insertId;

      // Deduct money from user's wallet (assuming wallet payment)
      if (paymentMethodId && paymentMethodId.startsWith('wallet_')) {
        const walletId = paymentMethodId.replace('wallet_', '');
        
        // Get wallet details - check if it's active (not frozen)
        const [wallets] = await connection.execute(
          'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND is_active = TRUE',
          [walletId, req.user.id]
        );

        if (wallets.length === 0) {
          await connection.rollback();
          connection.release();
          return res.status(400).json({ 
            error: 'Wallet not found or frozen',
            details: 'Selected wallet does not exist or is frozen. Please unfreeze the wallet to perform bill payments.'
          });
        }

        const wallet = wallets[0];
        
        // Check if wallet has NGN currency
        if (wallet.currency !== 'NGN') {
          await connection.rollback();
          connection.release();
          return res.status(400).json({ 
            error: 'Currency mismatch',
            details: 'VTPass payments require NGN currency. Please use a wallet with NGN currency.'
          });
        }
        
        // Deduct money from user's wallet
        const [walletUpdate] = await connection.execute(
          `UPDATE wallets 
           SET balance = balance - ? 
           WHERE id = ? AND user_id = ? AND balance >= ? AND currency = ?`,
          [totalAmount, walletId, req.user.id, totalAmount, 'NGN']
        );

        if (walletUpdate.affectedRows === 0) {
          await connection.rollback();
          connection.release();
          return res.status(400).json({ 
            error: 'Insufficient wallet balance',
            details: 'Please ensure you have sufficient NGN funds in your wallet'
          });
        }

        console.log(`💰 Deducted ${totalAmount} NGN from wallet ${walletId}`);

        // Create transaction record
        const referenceId = `VT_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
        await connection.execute(
          `INSERT INTO transactions (
            user_id, transaction_type, amount, currency, description, status,
            sender_wallet_id, transaction_fee, reference_id
          ) VALUES (?, 'bill_payment', ?, 'NGN', ?, 'completed', ?, ?, ?)`,
          [
            req.user.id,
            amount,
            `Bill payment: ${serviceId} - ${customerNumber}`,
            walletId,
            processingFee,
            referenceId
          ]
        );
      }

      await connection.commit();
      connection.release();

      console.log('✅ VTPass payment saved to database:', {
        paymentId,
        transactionId: finalTransactionId,
        amount,
        totalAmount
      });

      res.json({
        success: true,
        data: {
          ...result.data,
          paymentId,
          transactionId: finalTransactionId,
          transaction_id: finalTransactionId,
          reference: paymentReference,
          requestId: paymentReference,
          savedToDatabase: true,
          status: 'completed'
        }
      });
    } catch (dbError) {
      if (connection) {
        await connection.rollback().catch(rollbackErr => {
          console.error('Rollback error:', rollbackErr);
        });
        connection.release();
      }
      console.error('💥 Database error during VTPass payment:', dbError);
      console.error('💥 Database error details:', {
        message: dbError.message,
        code: dbError.code,
        sqlState: dbError.sqlState,
        sqlMessage: dbError.sqlMessage
      });
      
      // Return error response instead of throwing to prevent navigation issues
      // Payment was processed by VTPass, but database save failed
      return res.status(500).json({ 
        success: false,
        error: 'Payment processed but failed to save to database. Please contact support.',
        details: dbError.message,
        // Still return the payment result so user knows payment went through
        data: result?.data ? {
          ...result.data,
          savedToDatabase: false,
          databaseError: true
        } : null
      });
    }
  } catch (error) {
    console.error('💥 Server: VTPass payment error:', error);
    console.error('💥 Error details:', {
      message: error.message,
      stack: error.stack,
      response: error.response?.data
    });
    
    // Return proper error response
    res.status(500).json({ 
      success: false,
      error: error.message || 'Failed to process payment',
      details: error.response?.data || error.message
    });
  }
});

// Get transaction status from VTPass
app.get('/api/vtpass/transaction-status/:reference', authenticateToken, async (req, res) => {
  try {
    const { reference } = req.params;
    
    console.log('🚀 Server: Getting transaction status from VTPass:', reference);
    const result = await vtpass.getTransactionStatus(reference);
    
    if (!result.success) {
      return res.status(400).json({ 
        error: result.error || 'Failed to get transaction status' 
      });
    }

    res.json({
      success: true,
      data: result.data
    });
  } catch (error) {
    console.error('💥 Server: VTPass transaction status error:', error);
    res.status(500).json({ error: 'Failed to get transaction status' });
  }
});

// ==================== UTILITY ENDPOINTS ====================

// Health check
app.get('/api/health', (req, res) => {
  res.json({
    success: true,
    message: 'GlobanaMobile API is running',
    timestamp: new Date().toISOString()
  });
});

// Get supported currencies
app.get('/api/currencies', (req, res) => {
  res.json({
    success: true,
    currencies: [
      { code: 'USD', name: 'US Dollar', symbol: '$', country: 'USA' },
      { code: 'GBP', name: 'British Pound', symbol: '£', country: 'UK' },
      { code: 'NGN', name: 'Nigerian Naira', symbol: '₦', country: 'Nigeria' },
      { code: 'BTC', name: 'Bitcoin', symbol: '₿', country: 'Global' },
      { code: 'ETH', name: 'Ethereum', symbol: 'Ξ', country: 'Global' }
    ]
  });
});

// Get supported crypto types
app.get('/api/crypto/types', (req, res) => {
  res.json({
    success: true,
    cryptoTypes: [
      { code: 'BTC', name: 'Bitcoin', symbol: '₿' },
      { code: 'ETH', name: 'Ethereum', symbol: 'Ξ' },
      { code: 'USDT', name: 'Tether', symbol: '₮' },
      { code: 'USDC', name: 'USD Coin', symbol: '💵' },
      { code: 'BNB', name: 'Binance Coin', symbol: 'BNB' },
      { code: 'ADA', name: 'Cardano', symbol: '₳' },
      { code: 'SOL', name: 'Solana', symbol: '◎' }
    ]
  });
});

const getExchangeRate = (fromCurrency, toCurrency) => {
  // Normalize currency codes to uppercase
  const from = fromCurrency.toUpperCase();
  const to = toCurrency.toUpperCase();
  
  // Updated exchange rates as of 2024 (approximate market rates) - US, UK, Nigeria
  const rates = {
    USD: {
      NGN: 1500,  // Updated: 1 USD = ~1500 NGN
      GBP: 0.79,  // Updated: 1 USD = ~0.79 GBP
      USD: 1
    },
    NGN: {
      USD: 0.00067,  // Updated: 1 NGN = ~0.00067 USD (1/1500)
      GBP: 0.00053,  // Updated: 1 NGN = ~0.00053 GBP
      NGN: 1
    },
    GBP: {
      USD: 1.27,   // Updated: 1 GBP = ~1.27 USD
      NGN: 1900,   // Updated: 1 GBP = ~1900 NGN
      GBP: 1
    }
  };

  if (from === to) return 1;
  
  const rate = rates[from]?.[to];
  if (!rate) {
    throw new Error(`Exchange rate not available for ${from} to ${to}`);
  }
  
  return rate;
};

// Convert amount between currencies
const convertCurrency = (amount, fromCurrency, toCurrency) => {
  const rate = getExchangeRate(fromCurrency, toCurrency);
  return amount * rate;
};

// Create payment intent (general endpoint)
app.post('/api/create-payment-intent', authenticateToken, async (req, res) => {
  try {
    const { amount, currency = 'NGN', paymentMethod, walletId } = req.body;

    if (!amount || amount <= 0) {
      return res.status(400).json({ error: 'Valid amount is required' });
    }

    if (!walletId) {
      return res.status(400).json({ error: 'Wallet ID is required' });
    }

    const connection = await pool.getConnection();
    
    // Check if wallet belongs to user
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Wallet not found' });
    }

    const wallet = wallets[0];
    connection.release();

    // Calculate processing fee based on payment method
    let processingFee = 0;
    if (paymentMethod === 'card') {
      processingFee = Math.max(50, amount * 0.015); // 1.5% or minimum ₦50
    } else if (paymentMethod === 'ussd') {
      processingFee = 25; // Fixed ₦25 for USSD
    }

    const totalAmount = amount + processingFee;

    // Create Stripe payment intent
    let paymentIntentId, clientSecret;

    try {
      const paymentIntent = await stripe.paymentIntents.create({
        amount: Math.round(totalAmount * 100), // Convert to kobo (smallest currency unit)
        currency: currency.toLowerCase(),
        payment_method_types: ['card'],
        metadata: {
          walletId: walletId.toString(),
          userId: req.user.id.toString(),
          walletCurrency: wallet.currency,
          fundingCurrency: currency,
          paymentMethod: paymentMethod,
          wallet_name: wallet.wallet_name,
          platform_payment: 'wallet_funding'
        },
        description: `Wallet funding for ${wallet.wallet_name}`,
        receipt_email: req.user.email,
      });

      paymentIntentId = paymentIntent.id;
      clientSecret = paymentIntent.client_secret;

      console.log(`💳 Created payment intent: ${paymentIntentId}`);
      console.log(`💰 Amount: ${amount} ${currency}, Fee: ${processingFee} ${currency}, Total: ${totalAmount} ${currency}`);

    } catch (stripeError) {
      console.error('Payment intent creation failed:', stripeError);
      return res.status(400).json({ 
        error: 'Payment processing failed',
        details: stripeError.message 
      });
    }

    const responseData = {
      success: true,
      data: {
        paymentIntentId,
        clientSecret,
        amount: amount,
        processingFee,
        totalAmount,
        currency,
        walletCurrency: wallet.currency,
        walletName: wallet.wallet_name
      }
    };
    
    console.log('📤 Sending payment intent response:', responseData);
    
    res.json(responseData);
  } catch (error) {
    console.error('Create payment intent error:', error);
    res.status(500).json({ 
      error: 'Failed to create payment intent',
      details: error.message 
    });
  }
});

// Create SetupIntent for saving cards
app.post('/api/create-setup-intent', authenticateToken, async (req, res) => {
  try {
    const { customerEmail } = req.body;

    if (!customerEmail) {
      return res.status(400).json({ error: 'Customer email is required' });
    }

    // First, find or create customer in Stripe using email
    let customer;
    const existingCustomers = await stripe.customers.list({
      email: customerEmail,
      limit: 1
    });

    if (existingCustomers.data.length > 0) {
      customer = existingCustomers.data[0];
      console.log(`👤 Found existing customer: ${customer.id} for email: ${customerEmail}`);
    } else {
      // Create new customer in Stripe
      customer = await stripe.customers.create({
        email: customerEmail,
        name: `${req.user.first_name} ${req.user.last_name}`,
        metadata: {
          userId: req.user.id.toString(),
          platform_user: 'globaltrustmobile'
        }
      });
      console.log(`👤 Created new customer: ${customer.id} for email: ${customerEmail}`);
    }

    // Create SetupIntent in Stripe
    const setupIntent = await stripe.setupIntents.create({
      customer: customer.id,
      payment_method_types: ['card'],
      metadata: {
        userId: req.user.id.toString(),
        customerEmail: customerEmail,
        platform_payment: 'card_setup'
      }
    });

    console.log(`💳 Created SetupIntent: ${setupIntent.id} for customer: ${customer.id}`);

    res.json({
      success: true,
      data: {
        clientSecret: setupIntent.client_secret,
        setupIntentId: setupIntent.id,
        customerId: customer.id
      }
    });

  } catch (error) {
    console.error('Create SetupIntent error:', error);
    res.status(500).json({ 
      error: 'Failed to create SetupIntent',
      details: error.message 
    });
  }
});

// Create payment intent for wallet funding
app.post('/api/wallets/:walletId/create-payment-intent', authenticateToken, async (req, res) => {
  try {
    const { walletId } = req.params;
    const { amount, currency = 'NGN', paymentMethod } = req.body;

    if (!amount || amount <= 0) {
      return res.status(400).json({ error: 'Valid amount is required' });
    }

    const connection = await pool.getConnection();
    
    // Check if wallet belongs to user and is active (not frozen)
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND is_active = TRUE',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Wallet not found or frozen. Please unfreeze the wallet to create a payment intent.' });
    }

    const wallet = wallets[0];
    connection.release();

    // Calculate processing fee based on payment method
    let processingFee = 0;
    if (paymentMethod === 'card') {
      processingFee = Math.max(50, amount * 0.015); // 1.5% or minimum ₦50
    } else if (paymentMethod === 'ussd') {
      processingFee = 25; // Fixed ₦25 for USSD
    }

    const totalAmount = amount + processingFee;

    // Create Stripe payment intent
    let paymentIntentId, clientSecret;

    try {
      const paymentIntent = await stripe.paymentIntents.create({
        amount: Math.round(totalAmount * 100), // Convert to kobo (smallest currency unit)
        currency: currency.toLowerCase(),
        payment_method_types: ['card'],
        metadata: {
          walletId: walletId.toString(),
          userId: req.user.id.toString(),
          walletCurrency: wallet.currency,
          fundingCurrency: currency,
          paymentMethod: paymentMethod,
          wallet_name: wallet.wallet_name,
          platform_payment: 'wallet_funding'
        },
        description: `Wallet funding for ${wallet.wallet_name}`,
        receipt_email: req.user.email,
      });

      paymentIntentId = paymentIntent.id;
      clientSecret = paymentIntent.client_secret;

      console.log(`💳 Created payment intent for wallet funding: ${paymentIntentId}`);
      console.log(`💰 Amount: ${amount} ${currency}, Fee: ${processingFee} ${currency}, Total: ${totalAmount} ${currency}`);

    } catch (stripeError) {
      console.error('Payment intent creation failed:', stripeError);
      return res.status(400).json({ 
        error: 'Payment processing failed',
        details: stripeError.message 
      });
    }

    const responseData = {
      success: true,
      data: {
        paymentIntentId,
        clientSecret,
        amount: amount,
        processingFee,
        totalAmount,
        currency,
        walletCurrency: wallet.currency,
        walletName: wallet.wallet_name
      }
    };
    
    console.log('📤 Sending payment intent response:', responseData);
    
    res.json(responseData);
  } catch (error) {
    console.error('Create payment intent error:', error);
    res.status(500).json({ 
      error: 'Failed to create payment intent',
      details: error.message 
    });
  }
});
// Stripe SetupIntent endpoints for saving cards

// Create SetupIntent for saving a card
app.post('/api/setup-intent/create', authenticateToken, async (req, res) => {
  try {
    const { customerEmail } = req.body;

    if (!customerEmail) {
      return res.status(400).json({ error: 'Customer email is required' });
    }

    // First, find or create customer in Stripe using email
    let customer;
    const existingCustomers = await stripe.customers.list({
      email: customerEmail,
      limit: 1
    });

    if (existingCustomers.data.length > 0) {
      customer = existingCustomers.data[0];
      console.log(`👤 Found existing customer: ${customer.id} for email: ${customerEmail}`);
    } else {
      // Create new customer in Stripe
      customer = await stripe.customers.create({
        email: customerEmail,
        name: `${req.user.first_name} ${req.user.last_name}`,
        metadata: {
          userId: req.user.id.toString(),
          platform_user: 'globaltrustmobile'
        }
      });
      console.log(`👤 Created new customer: ${customer.id} for email: ${customerEmail}`);
    }

    // Create SetupIntent in Stripe
    const setupIntent = await stripe.setupIntents.create({
      customer: customer.id,
      payment_method_types: ['card'],
      metadata: {
        userId: req.user.id.toString(),
        customerEmail: customerEmail,
        platform_payment: 'card_setup'
      }
    });

    console.log(`💳 Created SetupIntent: ${setupIntent.id} for customer: ${customer.id}`);

    res.json({
      success: true,
      data: {
        clientSecret: setupIntent.client_secret,
        setupIntentId: setupIntent.id,
        customerId: customer.id
      }
    });

  } catch (error) {
    console.error('Create SetupIntent error:', error);
    res.status(500).json({ 
      error: 'Failed to create SetupIntent',
      details: error.message 
    });
  }
});

// Get saved cards for a customer by email
app.get('/api/saved-cards/:customerEmail', authenticateToken, async (req, res) => {
  try {
    const { customerEmail } = req.params;

    console.log(`🔍 Looking for saved cards for email: ${customerEmail}`);
    console.log(`👤 Authenticated user ID: ${req.user.id}`);

    if (!customerEmail) {
      return res.status(400).json({ error: 'Customer email is required' });
    }

    // First, find customer in Stripe using email
    console.log(`🔍 Searching for Stripe customer with email: ${customerEmail}`);
    const existingCustomers = await stripe.customers.list({
      email: customerEmail,
      limit: 1
    });

    console.log(`👥 Found ${existingCustomers.data.length} customers in Stripe`);

    if (existingCustomers.data.length === 0) {
      console.log(`👤 No customer found for email: ${customerEmail}`);
      return res.json({
        success: true,
        data: [],
        debug: {
          searchedEmail: customerEmail,
          customersFound: 0
        }
      });
    }

    const customer = existingCustomers.data[0];
    console.log(`👤 Found customer: ${customer.id} for email: ${customerEmail}`);

    // Get payment methods from Stripe
    console.log(`💳 Fetching payment methods for customer: ${customer.id}`);
    const paymentMethods = await stripe.paymentMethods.list({
      customer: customer.id,
      type: 'card',
    });

    console.log(`💳 Retrieved ${paymentMethods.data.length} saved cards for customer: ${customer.id} (${customerEmail})`);

    res.json({
      success: true,
      data: paymentMethods.data,
      debug: {
        customerId: customer.id,
        customerEmail: customerEmail,
        cardsFound: paymentMethods.data.length
      }
    });

  } catch (error) {
    console.error('Get saved cards error:', error);
    res.status(500).json({ 
      error: 'Failed to get saved cards',
      details: error.message 
    });
  }
});

// Debug endpoint to test card saving
app.post('/api/debug/save-test-card', authenticateToken, async (req, res) => {
  try {
    const { customerEmail } = req.body;

    if (!customerEmail) {
      return res.status(400).json({ error: 'Customer email is required' });
    }

    console.log(`🧪 Debug: Testing card save for email: ${customerEmail}`);

    // First, find or create customer in Stripe using email
    let customer;
    const existingCustomers = await stripe.customers.list({
      email: customerEmail,
      limit: 1
    });

    if (existingCustomers.data.length > 0) {
      customer = existingCustomers.data[0];
      console.log(`👤 Found existing customer: ${customer.id} for email: ${customerEmail}`);
    } else {
      // Create new customer in Stripe
      customer = await stripe.customers.create({
        email: customerEmail,
        name: `${req.user.first_name} ${req.user.last_name}`,
        metadata: {
          userId: req.user.id.toString(),
          platform_user: 'globaltrustmobile'
        }
      });
      console.log(`👤 Created new customer: ${customer.id} for email: ${customerEmail}`);
    }

    // Create a test payment method (this would normally come from Stripe's frontend)
    const testPaymentMethod = await stripe.paymentMethods.create({
      type: 'card',
      card: {
        token: 'tok_visa', // Stripe test token
      },
    });

    console.log(`💳 Created test payment method: ${testPaymentMethod.id}`);

    // Attach the payment method to the customer
    await stripe.paymentMethods.attach(testPaymentMethod.id, {
      customer: customer.id,
    });

    console.log(`💳 Attached payment method ${testPaymentMethod.id} to customer ${customer.id}`);

    // Get all payment methods for the customer
    const paymentMethods = await stripe.paymentMethods.list({
      customer: customer.id,
      type: 'card',
    });

    console.log(`💳 Customer now has ${paymentMethods.data.length} saved cards`);

    res.json({
      success: true,
      data: {
        customerId: customer.id,
        paymentMethodId: testPaymentMethod.id,
        totalCards: paymentMethods.data.length,
        cards: paymentMethods.data
      }
    });

  } catch (error) {
    console.error('Debug save test card error:', error);
    res.status(500).json({ 
      error: 'Failed to save test card',
      details: error.message 
    });
  }
});

// Test endpoint for debugging
app.get('/api/test-charge-endpoint', (req, res) => {
  console.log('🔍 Test endpoint hit');
  res.json({ 
    success: true, 
    message: 'Charge endpoint is accessible',
    timestamp: new Date().toISOString()
  });
});

// Charge a saved card
app.post('/api/charge-saved-card', authenticateToken, async (req, res) => {
  console.log('🔍 Charge saved card endpoint hit');
  console.log('🔍 Request body:', req.body);
  console.log('🔍 User:', req.user);
  
  try {
    const { customerEmail, paymentMethodId, amount, currency = 'NGN', walletId } = req.body;

    console.log('🔍 Parsed parameters:', { customerEmail, paymentMethodId, amount, currency, walletId });

    if (!customerEmail || !paymentMethodId || !amount || !walletId) {
      console.log('❌ Missing required parameters');
      return res.status(400).json({ error: 'Customer email, payment method ID, amount, and wallet ID are required' });
    }

    // First, find customer in Stripe using email
    const existingCustomers = await stripe.customers.list({
      email: customerEmail,
      limit: 1
    });

    if (existingCustomers.data.length === 0) {
      return res.status(404).json({ error: 'Customer not found' });
    }

    const customer = existingCustomers.data[0];

    const connection = await pool.getConnection();
    
    // Check if wallet belongs to user
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ?',
      [walletId, req.user.id]
    );

    if (wallets.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Wallet not found' });
    }

    const wallet = wallets[0];
    connection.release();

    // Calculate processing fee
    const processingFee = Math.max(50, amount * 0.015); // 1.5% or minimum ₦50
    const totalAmount = amount + processingFee;

    // Create payment intent with saved card
    const paymentIntent = await stripe.paymentIntents.create({
      amount: Math.round(totalAmount * 100), // Convert to kobo
      currency: currency.toLowerCase(),
      customer: customer.id,
      payment_method: paymentMethodId,
      off_session: true, // No immediate user interaction needed
      confirm: true,
      metadata: {
        walletId: walletId.toString(),
        userId: req.user.id.toString(),
        customerEmail: customerEmail,
        walletCurrency: wallet.currency,
        fundingCurrency: currency,
        paymentMethod: 'saved_card',
        wallet_name: wallet.wallet_name,
        platform_payment: 'wallet_funding'
      },
      description: `Wallet funding for ${wallet.wallet_name} using saved card`,
      receipt_email: customerEmail,
    });

    console.log(`💳 Charged saved card: ${paymentMethodId} for customer: ${customer.id} (${customerEmail}) for amount: ${totalAmount} ${currency}`);

    // If payment successful, fund the wallet
    if (paymentIntent.status === 'succeeded') {
      const connection = await pool.getConnection();
      
      try {
        // Convert amount to wallet currency if different
        let fundingAmount = amount;
        if (currency !== wallet.currency) {
          fundingAmount = convertCurrency(amount, currency, wallet.currency);
        }

        // Update wallet balance
        await connection.execute(
          'UPDATE wallets SET balance = balance + ? WHERE id = ?',
          [fundingAmount, walletId]
        );

        // Record transaction
        await connection.execute(
          `INSERT INTO transactions (
            user_id, receiver_wallet_id, transaction_type, amount, currency, 
            description, status, reference_id, stripe_payment_intent_id
          ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
          [
            req.user.id,
            walletId,
            'deposit',
            fundingAmount,
            wallet.currency,
            `Wallet funding via saved card - ${wallet.wallet_name}`,
            'completed',
            paymentIntent.id,
            paymentIntent.id
          ]
        );

        console.log(`✅ Wallet funded successfully: ${fundingAmount} ${wallet.currency} added to wallet ${walletId}`);

      } catch (dbError) {
        console.error('Database error during wallet funding:', dbError);
        throw new Error('Failed to update wallet balance');
      } finally {
        connection.release();
      }
    }

    res.json({
      success: true,
      data: {
        paymentIntentId: paymentIntent.id,
        status: paymentIntent.status,
        amount: amount,
        processingFee,
        totalAmount,
        currency,
        walletCurrency: wallet.currency
      }
    });

  } catch (error) {
    console.error('Charge saved card error:', error);
    
    // Handle specific Stripe errors
    if (error.type === 'StripeCardError') {
      return res.status(400).json({ 
        error: 'Card payment failed',
        details: error.message,
        code: error.code
      });
    }
    
    res.status(500).json({ 
      error: 'Failed to charge saved card',
      details: error.message 
    });
  }
});

// Serve uploaded files
app.use('/uploads', express.static('uploads'));

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('Error:', error);
  res.status(500).json({ error: 'Internal server error' });
});

// ==================== CRYPTO ENDPOINTS ====================

// Get user's crypto portfolio
app.get('/api/crypto/portfolio', authenticateToken, async (req, res) => {
  try {
    const connection = await pool.getConnection();
    
    // Check if crypto tables exist, create them if they don't
    const [tables] = await connection.execute("SHOW TABLES LIKE 'crypto_holdings'");
    
    if (tables.length === 0) {
      console.log('📊 Creating crypto tables...');
      
      // Create crypto_holdings table
      await connection.execute(`
        CREATE TABLE IF NOT EXISTS crypto_holdings (
          id INT PRIMARY KEY AUTO_INCREMENT,
          user_id INT NOT NULL,
          symbol VARCHAR(10) NOT NULL,
          name VARCHAR(100) NOT NULL,
          amount DECIMAL(20, 8) NOT NULL DEFAULT 0,
          icon VARCHAR(10) DEFAULT '₿',
          color VARCHAR(7) DEFAULT '#F7931A',
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
          UNIQUE KEY unique_user_crypto (user_id, symbol)
        )
      `);
      
      // Create crypto_prices table
      await connection.execute(`
        CREATE TABLE IF NOT EXISTS crypto_prices (
          id INT PRIMARY KEY AUTO_INCREMENT,
          symbol VARCHAR(10) NOT NULL,
          name VARCHAR(100) NOT NULL,
          price_usd DECIMAL(20, 8) NOT NULL,
          change_24h DECIMAL(10, 4) DEFAULT 0,
          market_cap BIGINT DEFAULT 0,
          volume_24h BIGINT DEFAULT 0,
          last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          UNIQUE KEY unique_symbol (symbol)
        )
      `);
      
      // Create crypto_transactions table
      await connection.execute(`
        CREATE TABLE IF NOT EXISTS crypto_transactions (
          id INT PRIMARY KEY AUTO_INCREMENT,
          user_id INT NOT NULL,
          type ENUM('buy', 'sell', 'send', 'receive', 'swap') NOT NULL,
          symbol VARCHAR(10) NOT NULL,
          amount DECIMAL(20, 8) NOT NULL,
          price_usd DECIMAL(20, 8) NOT NULL,
          total_value DECIMAL(20, 8) NOT NULL,
          from_symbol VARCHAR(10),
          to_symbol VARCHAR(10),
          from_amount DECIMAL(20, 8),
          to_amount DECIMAL(20, 8),
          from_price DECIMAL(20, 8),
          to_price DECIMAL(20, 8),
          status ENUM('pending', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
          transaction_hash VARCHAR(255),
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
        )
      `);
      
      // Insert initial crypto prices
      await connection.execute(`
        INSERT INTO crypto_prices (symbol, name, price_usd, change_24h, market_cap, volume_24h) VALUES
        ('BTC', 'Bitcoin', 45000.00, 2.5, 850000000000, 25000000000),
        ('ETH', 'Ethereum', 3000.00, -1.2, 360000000000, 15000000000),
        ('USDC', 'USD Coin', 1.00, 0.0, 30000000000, 5000000000),
        ('USDT', 'Tether', 1.00, 0.0, 80000000000, 30000000000),
        ('BNB', 'Binance Coin', 300.00, 1.5, 45000000000, 2000000000)
        ON DUPLICATE KEY UPDATE
        price_usd = VALUES(price_usd),
        change_24h = VALUES(change_24h),
        market_cap = VALUES(market_cap),
        volume_24h = VALUES(volume_24h),
        last_updated = CURRENT_TIMESTAMP
      `);
      
      console.log('✅ Crypto tables created successfully');
    }
    
    // Get user's crypto holdings
    const [holdings] = await connection.execute(
      'SELECT * FROM crypto_holdings WHERE user_id = ? ORDER BY created_at DESC',
      [req.user.id]
    );
    
    // If user has no crypto holdings, give them some initial holdings
    if (holdings.length === 0) {
      console.log('📊 User has no crypto holdings, creating initial holdings...');
      
      // Create initial crypto holdings for the user
      await connection.execute(
        'INSERT INTO crypto_holdings (user_id, symbol, name, amount, icon, color) VALUES (?, ?, ?, ?, ?, ?)',
        [req.user.id, 'BTC', 'Bitcoin', 0.01, '₿', '#F7931A']
      );
      
      await connection.execute(
        'INSERT INTO crypto_holdings (user_id, symbol, name, amount, icon, color) VALUES (?, ?, ?, ?, ?, ?)',
        [req.user.id, 'ETH', 'Ethereum', 0.1, 'Ξ', '#627EEA']
      );
      
      await connection.execute(
        'INSERT INTO crypto_holdings (user_id, symbol, name, amount, icon, color) VALUES (?, ?, ?, ?, ?, ?)',
        [req.user.id, 'USDC', 'USD Coin', 100, '$', '#2775CA']
      );
      
      // Get the newly created holdings
      const [newHoldings] = await connection.execute(
        'SELECT * FROM crypto_holdings WHERE user_id = ? ORDER BY created_at DESC',
        [req.user.id]
      );
      
      holdings.push(...newHoldings);
    }
    
    // Get crypto prices
    const [prices] = await connection.execute(
      'SELECT symbol, price_usd, change_24h FROM crypto_prices ORDER BY market_cap DESC'
    );
    
    connection.release();
    
    // Calculate portfolio value
    let totalValue = 0;
    const portfolio = holdings.map(holding => {
      const price = prices.find(p => p.symbol === holding.symbol);
      const currentValue = holding.amount * (price?.price_usd || 0);
      totalValue += currentValue;
      
      return {
        id: holding.id,
        symbol: holding.symbol,
        name: holding.name,
        amount: parseFloat(holding.amount),
        priceUsd: price?.price_usd || 0,
        value: currentValue,
        change24h: price?.change_24h || 0,
        icon: holding.icon,
        color: holding.color
      };
    });
    
    res.json({
      success: true,
      portfolio: portfolio,
      totalValue: totalValue,
      totalChange: 0 // Calculate based on previous day
    });
  } catch (error) {
    console.error('Crypto portfolio error:', error);
    res.status(500).json({ 
      success: false,
      error: 'Failed to fetch crypto portfolio',
      message: error.message 
    });
  }
});

// Get crypto market prices (live)
app.get('/api/crypto/market-prices', async (req, res) => {
  try {
    const connection = await pool.getConnection();
    
    // Get current prices from database
    const [prices] = await connection.execute(
      'SELECT symbol, name, price_usd, change_24h, market_cap, volume_24h, last_updated FROM crypto_prices ORDER BY market_cap DESC'
    );
    
    connection.release();
    
    res.json({
      success: true,
      prices: prices.map(price => ({
        symbol: price.symbol,
        name: price.name,
        priceUsd: parseFloat(price.price_usd),
        change24h: parseFloat(price.change_24h),
        marketCap: parseFloat(price.market_cap),
        volume24h: parseFloat(price.volume_24h),
        lastUpdated: price.last_updated
      })),
      isLiveData: true,
      message: 'Live crypto prices from external APIs'
    });
  } catch (error) {
    console.error('Market prices error:', error);
    res.status(500).json({ 
      success: false,
      error: 'Failed to fetch market prices',
      message: error.message 
    });
  }
});

// Manual price update endpoint
app.post('/api/crypto/update-prices', authenticateToken, async (req, res) => {
  try {
    console.log('🔄 Manual price update requested...');
    
    // Trigger immediate price update (disabled for mock mode)
    // await cryptoPriceService.updatePrices();
    
    res.json({
      success: true,
      message: 'Prices updated successfully',
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    console.error('Manual price update error:', error);
    res.status(500).json({ 
      success: false,
      error: 'Failed to update prices',
      message: error.message 
    });
  }
});

// Get price update status
app.get('/api/crypto/price-status', async (req, res) => {
  try {
    const connection = await pool.getConnection();
    
    // Get latest price update info
    const [latest] = await connection.execute(
      'SELECT MAX(last_updated) as last_update, COUNT(*) as price_count FROM crypto_prices'
    );
    
    connection.release();
    
    res.json({
      success: true,
      lastUpdate: latest[0].last_update,
      priceCount: latest[0].price_count,
      isLive: false, // Mock mode
      message: 'Live price service status'
    });
  } catch (error) {
    console.error('Price status error:', error);
    res.status(500).json({ 
      success: false,
      error: 'Failed to get price status',
      message: error.message 
    });
  }
});

// Buy crypto
app.post('/api/crypto/buy', authenticateToken, async (req, res) => {
  try {
    const { symbol, amount, paymentMethod, paymentAmount } = req.body;
    const connection = await pool.getConnection();
    
    await connection.beginTransaction();
    
    // Get current crypto price
    const [prices] = await connection.execute(
      'SELECT price_usd FROM crypto_prices WHERE symbol = ?',
      [symbol]
    );
    
    if (prices.length === 0) {
      await connection.rollback();
      connection.release();
      return res.status(400).json({ error: 'Crypto not found' });
    }
    
    const priceUsd = prices[0].price_usd;
    const totalCost = amount * priceUsd;
    
    // Check if user has sufficient balance
    const [userWallet] = await connection.execute(
      'SELECT balance FROM wallets WHERE user_id = ? AND currency = ? AND is_active = TRUE',
      [req.user.id, 'USD']
    );
    
    if (userWallet.length === 0 || userWallet[0].balance < totalCost) {
      await connection.rollback();
      connection.release();
      return res.status(400).json({ error: 'Insufficient balance' });
    }
    
    // Deduct from user's USD wallet
    await connection.execute(
      'UPDATE wallets SET balance = balance - ? WHERE user_id = ? AND currency = ?',
      [totalCost, req.user.id, 'USD']
    );
    
    // Add to crypto holdings or update existing
    const [existingHolding] = await connection.execute(
      'SELECT * FROM crypto_holdings WHERE user_id = ? AND symbol = ?',
      [req.user.id, symbol]
    );
    
    if (existingHolding.length > 0) {
      await connection.execute(
        'UPDATE crypto_holdings SET amount = amount + ? WHERE user_id = ? AND symbol = ?',
        [amount, req.user.id, symbol]
      );
    } else {
      await connection.execute(
        'INSERT INTO crypto_holdings (user_id, symbol, name, amount, icon, color) VALUES (?, ?, ?, ?, ?, ?)',
        [req.user.id, symbol, symbol, amount, '₿', '#F7931A']
      );
    }
    
    // Record transaction
    await connection.execute(
      'INSERT INTO crypto_transactions (user_id, type, symbol, amount, price_usd, total_cost, status) VALUES (?, ?, ?, ?, ?, ?, ?)',
      [req.user.id, 'buy', symbol, amount, priceUsd, totalCost, 'completed']
    );
    
    await connection.commit();
    connection.release();
    
    res.json({
      success: true,
      message: 'Crypto purchased successfully',
      transaction: {
        symbol,
        amount,
        priceUsd,
        totalCost
      }
    });
  } catch (error) {
    console.error('Buy crypto error:', error);
    res.status(500).json({ error: 'Failed to buy crypto' });
  }
});

// Sell crypto
app.post('/api/crypto/sell', authenticateToken, async (req, res) => {
  try {
    const { symbol, amount } = req.body;
    const connection = await pool.getConnection();
    
    await connection.beginTransaction();
    
    // Check if user has sufficient crypto balance
    const [holding] = await connection.execute(
      'SELECT * FROM crypto_holdings WHERE user_id = ? AND symbol = ?',
      [req.user.id, symbol]
    );
    
    if (holding.length === 0 || holding[0].amount < amount) {
      await connection.rollback();
      connection.release();
      return res.status(400).json({ error: 'Insufficient crypto balance' });
    }
    
    // Get current crypto price
    const [prices] = await connection.execute(
      'SELECT price_usd FROM crypto_prices WHERE symbol = ?',
      [symbol]
    );
    
    const priceUsd = prices[0].price_usd;
    const totalValue = amount * priceUsd;
    
    // Update crypto holdings
    await connection.execute(
      'UPDATE crypto_holdings SET amount = amount - ? WHERE user_id = ? AND symbol = ?',
      [amount, req.user.id, symbol]
    );
    
    // Add to user's USD wallet
    await connection.execute(
      'UPDATE wallets SET balance = balance + ? WHERE user_id = ? AND currency = ?',
      [totalValue, req.user.id, 'USD']
    );
    
    // Record transaction
    await connection.execute(
      'INSERT INTO crypto_transactions (user_id, type, symbol, amount, price_usd, total_value, status) VALUES (?, ?, ?, ?, ?, ?, ?)',
      [req.user.id, 'sell', symbol, amount, priceUsd, totalValue, 'completed']
    );
    
    await connection.commit();
    connection.release();
    
    res.json({
      success: true,
      message: 'Crypto sold successfully',
      transaction: {
        symbol,
        amount,
        priceUsd,
        totalValue
      }
    });
  } catch (error) {
    console.error('Sell crypto error:', error);
    res.status(500).json({ error: 'Failed to sell crypto' });
  }
});

// Send crypto
app.post('/api/crypto/send', authenticateToken, async (req, res) => {
  try {
    const { symbol, amount, recipientAddress, network } = req.body;
    const connection = await pool.getConnection();
    
    await connection.beginTransaction();
    
    // Check if user has sufficient crypto balance
    const [holding] = await connection.execute(
      'SELECT * FROM crypto_holdings WHERE user_id = ? AND symbol = ?',
      [req.user.id, symbol]
    );
    
    if (holding.length === 0 || holding[0].amount < amount) {
      await connection.rollback();
      connection.release();
      return res.status(400).json({ error: 'Insufficient crypto balance' });
    }
    
    // Get current crypto price for transaction value
    const [prices] = await connection.execute(
      'SELECT price_usd FROM crypto_prices WHERE symbol = ?',
      [symbol]
    );
    
    const priceUsd = prices[0].price_usd;
    const totalValue = amount * priceUsd;
    
    // Update crypto holdings
    await connection.execute(
      'UPDATE crypto_holdings SET amount = amount - ? WHERE user_id = ? AND symbol = ?',
      [amount, req.user.id, symbol]
    );
    
    // Record transaction
    await connection.execute(
      'INSERT INTO crypto_transactions (user_id, type, symbol, amount, price_usd, total_value, recipient_address, network, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)',
      [req.user.id, 'send', symbol, amount, priceUsd, totalValue, recipientAddress, network, 'completed']
    );
    
    await connection.commit();
    connection.release();
    
    res.json({
      success: true,
      message: 'Crypto sent successfully',
      transaction: {
        symbol,
        amount,
        recipientAddress,
        network,
        totalValue
      }
    });
  } catch (error) {
    console.error('Send crypto error:', error);
    res.status(500).json({ error: 'Failed to send crypto' });
  }
});

// Swap crypto
app.post('/api/crypto/swap', authenticateToken, async (req, res) => {
  try {
    const { fromSymbol, toSymbol, fromAmount, toAmount, exchangeRate, swapFee } = req.body;
    const connection = await pool.getConnection();
    
    await connection.beginTransaction();
    
    // Check if user has sufficient crypto balance
    const [holding] = await connection.execute(
      'SELECT * FROM crypto_holdings WHERE user_id = ? AND symbol = ?',
      [req.user.id, fromSymbol]
    );
    
    if (holding.length === 0 || holding[0].amount < fromAmount) {
      await connection.rollback();
      connection.release();
      return res.status(400).json({ success: false, message: 'Insufficient crypto balance' });
    }
    
    // Get current prices
    const [prices] = await connection.execute(
      'SELECT symbol, price_usd FROM crypto_prices WHERE symbol IN (?, ?)',
      [fromSymbol, toSymbol]
    );
    
    const fromPrice = prices.find(p => p.symbol === fromSymbol)?.price_usd || 0;
    const toPrice = prices.find(p => p.symbol === toSymbol)?.price_usd || 0;
    
    // Update crypto holdings
    await connection.execute(
      'UPDATE crypto_holdings SET amount = amount - ? WHERE user_id = ? AND symbol = ?',
      [fromAmount, req.user.id, fromSymbol]
    );
    
    // Add to crypto holdings or update existing
    const [existingHolding] = await connection.execute(
      'SELECT * FROM crypto_holdings WHERE user_id = ? AND symbol = ?',
      [req.user.id, toSymbol]
    );
    
    if (existingHolding.length > 0) {
      await connection.execute(
        'UPDATE crypto_holdings SET amount = amount + ? WHERE user_id = ? AND symbol = ?',
        [toAmount, req.user.id, toSymbol]
      );
    } else {
      await connection.execute(
        'INSERT INTO crypto_holdings (user_id, symbol, name, amount, icon, color) VALUES (?, ?, ?, ?, ?, ?)',
        [req.user.id, toSymbol, toSymbol, toAmount, '₿', '#F7931A']
      );
    }
    
    // Record swap in crypto_swaps table
    const [swapResult] = await connection.execute(
      'INSERT INTO crypto_swaps (user_id, from_symbol, to_symbol, from_amount, to_amount, from_price, to_price, exchange_rate, swap_fee, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
      [req.user.id, fromSymbol, toSymbol, fromAmount, toAmount, fromPrice, toPrice, exchangeRate, swapFee, 'completed']
    );
    
    // Record transaction in crypto_transactions table
    await connection.execute(
      'INSERT INTO crypto_transactions (user_id, type, symbol, amount, price_usd, total_value, from_symbol, to_symbol, from_amount, to_amount, from_price, to_price, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
      [req.user.id, 'swap', fromSymbol, fromAmount, fromPrice, fromAmount * fromPrice, fromSymbol, toSymbol, fromAmount, toAmount, fromPrice, toPrice, 'completed']
    );
    
    await connection.commit();
    connection.release();
    
    res.json({
      success: true,
      message: 'Crypto swapped successfully',
      swapId: swapResult.insertId,
      transaction: {
        fromSymbol,
        toSymbol,
        fromAmount,
        toAmount,
        exchangeRate,
        swapFee
      }
    });
  } catch (error) {
    console.error('Swap crypto error:', error);
    res.status(500).json({ success: false, message: 'Failed to swap crypto' });
  }
});

// Get crypto transaction history
app.get('/api/crypto/history', authenticateToken, async (req, res) => {
  try {
    const { type, symbol, limit = 50, offset = 0 } = req.query;
    const connection = await pool.getConnection();
    
    let query = 'SELECT * FROM crypto_transactions WHERE user_id = ?';
    const params = [req.user.id];
    
    if (type) {
      query += ' AND type = ?';
      params.push(type);
    }
    
    if (symbol) {
      query += ' AND symbol = ?';
      params.push(symbol);
    }
    
    query += ' ORDER BY created_at DESC LIMIT ? OFFSET ?';
    params.push(parseInt(limit), parseInt(offset));
    
    const [transactions] = await connection.execute(query, params);
    connection.release();
    
    res.json({
      success: true,
      transactions: transactions.map(tx => ({
        id: tx.id,
        type: tx.type,
        symbol: tx.symbol,
        amount: parseFloat(tx.amount),
        priceUsd: parseFloat(tx.price_usd),
        totalValue: parseFloat(tx.total_value),
        recipientAddress: tx.recipient_address,
        network: tx.network,
        status: tx.status,
        createdAt: tx.created_at
      }))
    });
  } catch (error) {
    console.error('Crypto history error:', error);
    res.status(500).json({ error: 'Failed to fetch crypto history' });
  }
});

// Fund account with crypto (convert crypto to fiat)
app.post('/api/crypto/fund-account', authenticateToken, async (req, res) => {
  try {
    const { cryptoAmount, cryptoCurrency, fiatAmount, fiatCurrency, exchangeRate, conversionFee } = req.body;
    const connection = await pool.getConnection();
    
    // Validate input
    if (!cryptoAmount || !cryptoCurrency || !fiatAmount || !fiatCurrency) {
      return res.status(400).json({ success: false, message: 'Missing required fields' });
    }
    
    // Check if user has sufficient crypto balance
    const [holdings] = await connection.execute(
      'SELECT amount FROM crypto_holdings WHERE user_id = ? AND symbol = ?',
      [req.user.id, cryptoCurrency]
    );
    
    if (holdings.length === 0 || parseFloat(holdings[0].amount) < parseFloat(cryptoAmount)) {
      return res.status(400).json({ success: false, message: 'Insufficient crypto balance' });
    }
    
    await connection.beginTransaction();
    
    try {
      // Deduct crypto from holdings
      await connection.execute(
        'UPDATE crypto_holdings SET amount = amount - ? WHERE user_id = ? AND symbol = ?',
        [cryptoAmount, req.user.id, cryptoCurrency]
      );
      
      // Add fiat to user's account balance
      await connection.execute(
        'UPDATE users SET balance = balance + ? WHERE id = ?',
        [fiatAmount, req.user.id]
      );
      
      // Record the conversion transaction
      await connection.execute(
        'INSERT INTO crypto_transactions (user_id, type, symbol, amount, price_usd, total_value, status, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())',
        [req.user.id, 'fund_account', cryptoCurrency, cryptoAmount, exchangeRate, fiatAmount, 'completed']
      );
      
      await connection.commit();
      connection.release();
      
      res.json({
        success: true,
        message: 'Account funded successfully',
        data: {
          cryptoAmount,
          cryptoCurrency,
          fiatAmount,
          fiatCurrency,
          exchangeRate,
          conversionFee
        }
      });
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  } catch (error) {
    console.error('Fund account error:', error);
    res.status(500).json({ success: false, message: 'Failed to fund account' });
  }
});

// Get user's crypto wallet addresses
app.get('/api/crypto/addresses', authenticateToken, async (req, res) => {
  try {
    const connection = await pool.getConnection();
    
    // Get user's crypto addresses from Magic Link
    const [addresses] = await connection.execute(
      'SELECT symbol, address, chain, created_at FROM crypto_addresses WHERE user_id = ?',
      [req.user.id]
    );
    
    connection.release();
    
    res.json({
      success: true,
      addresses: addresses
    });
  } catch (error) {
    console.error('Crypto addresses error:', error);
    res.status(500).json({ success: false, message: 'Failed to fetch crypto addresses' });
  }
});

// Validate crypto address
app.post('/api/crypto/validate-address', authenticateToken, async (req, res) => {
  try {
    const { address, currency } = req.body;
    
    if (!address || !currency) {
      return res.status(400).json({ success: false, message: 'Address and currency are required' });
    }
    
    // Basic address validation (Magic Link will handle the actual validation)
    let isValid = false;
    let message = '';
    
    if (currency === 'BTC') {
      isValid = address.startsWith('1') || address.startsWith('3') || address.startsWith('bc1');
      message = isValid ? 'Valid Bitcoin address' : 'Invalid Bitcoin address';
    } else if (currency === 'ETH') {
      isValid = address.startsWith('0x') && address.length === 42;
      message = isValid ? 'Valid Ethereum address' : 'Invalid Ethereum address';
    } else {
      isValid = address.length > 10;
      message = isValid ? 'Valid address' : 'Invalid address';
    }
    
    res.json({
      success: true,
      isValid: isValid,
      message: message
    });
  } catch (error) {
    console.error('Validate address error:', error);
    res.status(500).json({ success: false, message: 'Failed to validate address' });
  }
});

// Get transaction status
app.get('/api/crypto/transaction-status/:transactionId', authenticateToken, async (req, res) => {
  try {
    const { transactionId } = req.params;
    const connection = await pool.getConnection();
    
    const [transactions] = await connection.execute(
      'SELECT * FROM crypto_transactions WHERE id = ? AND user_id = ?',
      [transactionId, req.user.id]
    );
    
    connection.release();
    
    if (transactions.length === 0) {
      return res.status(404).json({ success: false, message: 'Transaction not found' });
    }
    
    const transaction = transactions[0];
    
    res.json({
      success: true,
      transaction: {
        id: transaction.id,
        type: transaction.type,
        symbol: transaction.symbol,
        amount: transaction.amount,
        status: transaction.status,
        hash: transaction.transaction_hash,
        createdAt: transaction.created_at
      }
    });
  } catch (error) {
    console.error('Transaction status error:', error);
    res.status(500).json({ success: false, message: 'Failed to fetch transaction status' });
  }
});

// Get crypto conversion rates (live)
app.get('/api/crypto/conversion-rates/:fromCurrency/:toCurrency', authenticateToken, async (req, res) => {
  try {
    const { fromCurrency, toCurrency } = req.params;
    // const cryptoPriceService = require('./cryptoPriceService');
    
    // Get live exchange rate (mock for now)
    const exchangeRate = 1.0; // Mock exchange rate
    
    res.json({
      success: true,
      fromCurrency: fromCurrency,
      toCurrency: toCurrency,
      rate: exchangeRate.exchangeRate,
      fromPrice: exchangeRate.fromPrice,
      toPrice: exchangeRate.toPrice,
      lastUpdated: exchangeRate.lastUpdated
    });
  } catch (error) {
    console.error('Conversion rates error:', error);
    res.status(500).json({ success: false, message: 'Failed to fetch conversion rates' });
  }
});

// Get swap quote (estimate swap amount)
app.post('/api/crypto/swap-quote', authenticateToken, async (req, res) => {
  try {
    const { fromSymbol, toSymbol, fromAmount } = req.body;
    const connection = await pool.getConnection();
    
    // Get current prices for both cryptocurrencies
    const [prices] = await connection.execute(
      'SELECT symbol, price_usd FROM crypto_prices WHERE symbol IN (?, ?)',
      [fromSymbol, toSymbol]
    );
    
    connection.release();
    
    if (prices.length < 2) {
      return res.status(404).json({ success: false, message: 'Price data not found' });
    }
    
    const fromPrice = prices.find(p => p.symbol === fromSymbol)?.price_usd || 0;
    const toPrice = prices.find(p => p.symbol === toSymbol)?.price_usd || 0;
    
    const fromValue = parseFloat(fromAmount) * fromPrice;
    const toAmount = fromValue / toPrice;
    const swapFee = toAmount * 0.0025; // 0.25% swap fee
    const netAmount = toAmount - swapFee;
    
    res.json({
      success: true,
      fromSymbol,
      toSymbol,
      fromAmount: parseFloat(fromAmount),
      toAmount: toAmount,
      netAmount: netAmount,
      swapFee: swapFee,
      exchangeRate: fromPrice / toPrice,
      fromPrice,
      toPrice
    });
  } catch (error) {
    console.error('Swap quote error:', error);
    res.status(500).json({ success: false, message: 'Failed to get swap quote' });
  }
});

// Get swap history
app.get('/api/crypto/swap-history', authenticateToken, async (req, res) => {
  try {
    const { limit = 50, offset = 0 } = req.query;
    const connection = await pool.getConnection();
    
    const [swaps] = await connection.execute(
      'SELECT * FROM crypto_swaps WHERE user_id = ? ORDER BY created_at DESC LIMIT ? OFFSET ?',
      [req.user.id, parseInt(limit), parseInt(offset)]
    );
    
    connection.release();
    
    res.json({
      success: true,
      swaps: swaps.map(swap => ({
        id: swap.id,
        fromSymbol: swap.from_symbol,
        toSymbol: swap.to_symbol,
        fromAmount: parseFloat(swap.from_amount),
        toAmount: parseFloat(swap.to_amount),
        exchangeRate: parseFloat(swap.exchange_rate),
        swapFee: parseFloat(swap.swap_fee),
        status: swap.status,
        transactionHash: swap.transaction_hash,
        createdAt: swap.created_at,
        completedAt: swap.completed_at
      }))
    });
  } catch (error) {
    console.error('Swap history error:', error);
    res.status(500).json({ success: false, message: 'Failed to fetch swap history' });
  }
});

// Get swap status
app.get('/api/crypto/swap-status/:swapId', authenticateToken, async (req, res) => {
  try {
    const { swapId } = req.params;
    const connection = await pool.getConnection();
    
    const [swaps] = await connection.execute(
      'SELECT * FROM crypto_swaps WHERE id = ? AND user_id = ?',
      [swapId, req.user.id]
    );
    
    connection.release();
    
    if (swaps.length === 0) {
      return res.status(404).json({ success: false, message: 'Swap not found' });
    }
    
    const swap = swaps[0];
    
    res.json({
      success: true,
      swap: {
        id: swap.id,
        fromSymbol: swap.from_symbol,
        toSymbol: swap.to_symbol,
        fromAmount: parseFloat(swap.from_amount),
        toAmount: parseFloat(swap.to_amount),
        exchangeRate: parseFloat(swap.exchange_rate),
        swapFee: parseFloat(swap.swap_fee),
        status: swap.status,
        transactionHash: swap.transaction_hash,
        magicTxId: swap.magic_tx_id,
        createdAt: swap.created_at,
        completedAt: swap.completed_at
      }
    });
  } catch (error) {
    console.error('Swap status error:', error);
    res.status(500).json({ success: false, message: 'Failed to fetch swap status' });
  }
});

// ==================== START LIVE PRICE SERVICE ====================

// Import and start crypto price service
const cryptoPriceService = require('./cryptoPriceService');

// Start automatic price updates
cryptoPriceService.startPriceUpdates();
console.log('🚀 Live crypto price service started');

// ==================== NEW TRANSFER ENDPOINTS ====================

// Get live exchange rate from Fixer.io
app.get('/api/exchange-rates/:fromCurrency/:toCurrency', authenticateToken, async (req, res) => {
  try {
    const { fromCurrency, toCurrency } = req.params;
    const { getExchangeRate } = require('./services/fixer');
    
    const rateData = await getExchangeRate(fromCurrency, toCurrency);
    
    res.json({
      success: true,
      rate: rateData.rate,
      date: rateData.date,
      timestamp: rateData.timestamp,
      fromCurrency: fromCurrency.toUpperCase(),
      toCurrency: toCurrency.toUpperCase(),
      lastUpdated: new Date().toISOString()
    });
  } catch (error) {
    console.error('Error fetching exchange rate:', error);
    res.status(500).json({
      success: false,
      error: error.message || 'Failed to fetch exchange rate',
      code: error.code
    });
  }
});

// Note: Currency conversion is done locally on the frontend using the exchange rate
// This endpoint is kept for backward compatibility but conversion should be done client-side

// Currency Exchange (conversion done locally on frontend, we just verify rate)
app.post('/api/transactions/currency-exchange', authenticateToken, requireKYCApproval, async (req, res) => {
    const connection = await pool.getConnection();
  try {
    const { fromCurrency, toCurrency, fromAmount, toAmount, exchangeRate, fromWalletId, toWalletId } = req.body;
    
    if (!fromCurrency || !toCurrency || !fromAmount || !toAmount || !exchangeRate || !fromWalletId) {
      connection.release();
      return res.status(400).json({ 
        success: false, 
        error: 'Missing required fields: fromCurrency, toCurrency, fromAmount, toAmount, exchangeRate, fromWalletId' 
      });
    }

    if (parseFloat(fromAmount) <= 0) {
      connection.release();
      return res.status(400).json({ 
        success: false, 
        error: 'Amount must be greater than 0' 
      });
    }
    
    await connection.beginTransaction();
    
    try {
      // Get source wallet
      const [fromWallets] = await connection.execute(
        'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND currency = ? AND is_active = TRUE',
        [fromWalletId, req.user.id, fromCurrency.toUpperCase()]
      );

      if (fromWallets.length === 0) {
        await connection.rollback();
        connection.release();
        return res.status(400).json({ 
          success: false, 
          error: 'Source wallet not found or invalid' 
        });
      }

      const fromWallet = fromWallets[0];
      const amount = parseFloat(fromAmount);

      if (parseFloat(fromWallet.balance) < amount) {
        await connection.rollback();
      connection.release();
        return res.status(400).json({ 
          success: false, 
          error: 'Insufficient balance' 
        });
      }

      // Verify exchange rate from Fixer.io (conversion already done on frontend)
      const { getExchangeRate } = require('./services/fixer');
      const rateData = await getExchangeRate(fromCurrency, toCurrency);
      const currentRate = rateData.rate;
      
      // Use the toAmount provided by frontend (calculated locally)
      const convertedAmount = parseFloat(toAmount);
      
      // Optional: Verify the rate matches (within 1% tolerance for rate fluctuations)
      const rateDifference = Math.abs((exchangeRate - currentRate) / currentRate);
      if (rateDifference > 0.01) {
        console.warn(`⚠️ Exchange rate mismatch: Frontend=${exchangeRate}, Current=${currentRate}`);
        // Still proceed, but log the difference
      }

      // Get or create destination wallet
      let toWallet;
      if (toWalletId) {
        const [toWallets] = await connection.execute(
          'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND currency = ? AND is_active = TRUE',
          [toWalletId, req.user.id, toCurrency.toUpperCase()]
        );
        if (toWallets.length === 0) {
      await connection.rollback();
      connection.release();
          return res.status(400).json({ 
            success: false, 
            error: 'Destination wallet not found or invalid' 
          });
        }
        toWallet = toWallets[0];
      } else {
        // Check if user already has a wallet for this currency
        const [existingWallets] = await connection.execute(
          'SELECT * FROM wallets WHERE user_id = ? AND currency = ? AND is_active = TRUE',
          [req.user.id, toCurrency.toUpperCase()]
        );
        if (existingWallets.length > 0) {
          // Use existing wallet
          toWallet = existingWallets[0];
        } else {
          // Only create new wallet if user doesn't have one for this currency
          const walletName = `${toCurrency.toUpperCase()} Wallet`;
          const [newWallet] = await connection.execute(
            'INSERT INTO wallets (user_id, wallet_name, currency, balance, wallet_type, is_active, is_default) VALUES (?, ?, ?, 0, "fiat", TRUE, FALSE)',
            [req.user.id, walletName, toCurrency.toUpperCase()]
          );
          toWallet = { id: newWallet.insertId, balance: 0 };
        }
      }

      // Debit from source wallet
      await connection.execute(
        'UPDATE wallets SET balance = balance - ? WHERE id = ?',
        [amount, fromWalletId]
      );

      // Credit to destination wallet (using amount calculated on frontend)
      await connection.execute(
        'UPDATE wallets SET balance = balance + ? WHERE id = ?',
        [convertedAmount, toWallet.id]
      );

      // Create transaction record
      const referenceId = `EXC-${Date.now()}`;
      const [transactionResult] = await connection.execute(
        'INSERT INTO transactions (user_id, transaction_type, amount, currency, description, status, reference_id, exchange_rate, original_amount, original_currency, converted_amount, converted_currency, sender_wallet_id, receiver_wallet_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
        [
          req.user.id, 
          'currency_exchange', 
          amount, 
          fromCurrency.toUpperCase(), 
          `Currency Exchange: ${fromCurrency} to ${toCurrency}`, 
          'completed', 
          referenceId, 
          currentRate, 
          amount, 
          fromCurrency.toUpperCase(), 
          convertedAmount, 
          toCurrency.toUpperCase(),
          fromWalletId,
          toWallet.id
        ]
      );
      
      await connection.commit();
      connection.release();
      
      res.json({
        success: true,
        message: 'Currency exchange completed successfully',
        transactionId: transactionResult.insertId,
        exchange: {
          fromCurrency: fromCurrency.toUpperCase(),
          toCurrency: toCurrency.toUpperCase(),
          fromAmount: amount,
          toAmount: convertedAmount,
          exchangeRate: currentRate,
          date: rateData.date
        }
      });
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  } catch (error) {
    console.error('Currency exchange error:', error);
    if (connection) connection.release();
    res.status(500).json({ 
      success: false, 
      message: 'Currency exchange failed',
      error: error.message 
    });
  }
});

// Account Transfer (Wallet to Wallet)
app.post('/api/transactions/account-transfer', authenticateToken, requireKYCApproval, async (req, res) => {
  try {
    const { fromWalletId, toWalletId, amount, currency, note, customerEmail } = req.body;
    const connection = await pool.getConnection();
    
    await connection.beginTransaction();
    
    try {
      // Get source wallet - check if it's active (not frozen)
      const [fromWallets] = await connection.execute(
        'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND is_active = TRUE',
        [fromWalletId, req.user.id]
      );
      
      if (fromWallets.length === 0) {
        await connection.rollback();
        connection.release();
        return res.status(400).json({ 
          success: false, 
          message: 'Source wallet not found or frozen. Please unfreeze the wallet to perform transactions.' 
        });
      }
      
      const fromWallet = fromWallets[0];
      
      // Check if user has sufficient balance
      if (parseFloat(fromWallet.balance) < parseFloat(amount)) {
        await connection.rollback();
        connection.release();
        return res.status(400).json({ success: false, message: 'Insufficient balance' });
      }
      
      // Get destination wallet - check if it's active (not frozen)
      const [toWallets] = await connection.execute(
        'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND is_active = TRUE',
        [toWalletId, req.user.id]
      );
      
      if (toWallets.length === 0) {
        await connection.rollback();
        connection.release();
        return res.status(400).json({ 
          success: false, 
          message: 'Destination wallet not found or frozen. Please unfreeze the wallet to receive transfers.' 
        });
      }
      
      const toWallet = toWallets[0];
      
      // Update wallet balances
      await connection.execute(
        'UPDATE wallets SET balance = balance - ? WHERE id = ?',
        [amount, fromWalletId]
      );
      
      await connection.execute(
        'UPDATE wallets SET balance = balance + ? WHERE id = ?',
        [amount, toWalletId]
      );
      
      // Create the transfer transaction
      const [transactionResult] = await connection.execute(
        'INSERT INTO transactions (user_id, transaction_type, amount, currency, description, status, reference_id, sender_wallet_id, receiver_wallet_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)',
        [req.user.id, 'account_transfer', amount, currency, note || `Transfer from ${fromWallet.wallet_name} to ${toWallet.wallet_name}`, 'completed', `AT-${Date.now()}`, fromWalletId, toWalletId]
      );
      
      await connection.commit();
      connection.release();
      
      res.json({
        success: true,
        message: 'Account transfer completed successfully',
        transactionId: transactionResult.insertId,
        transfer: {
          fromWallet: fromWallet.wallet_name,
          toWallet: toWallet.wallet_name,
          amount: amount,
          currency: currency
        }
      });
    } catch (error) {
      await connection.rollback();
      connection.release();
      throw error;
    }
  } catch (error) {
    console.error('Account transfer error:', error);
    res.status(500).json({ success: false, message: 'Account transfer failed' });
  }
});

// ==================== MAGIC LINK ENDPOINTS ====================

// Import Magic Link backend service
const magicLinkBackend = require('./magicLinkBackend');

// Verify Magic Link token
app.post('/api/magic/verify-token', authenticateToken, async (req, res) => {
  try {
    const { didToken } = req.body;
    
    if (!didToken) {
      return res.status(400).json({ error: 'DID token is required' });
    }
    
    const result = await magicLinkBackend.verifyToken(didToken);
    res.json(result);
  } catch (error) {
    console.error('Magic token verification error:', error);
    res.status(500).json({ error: 'Token verification failed' });
  }
});

// Get Magic wallet info
app.get('/api/magic/wallet', authenticateToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const result = await magicLinkBackend.getUserWallet(userId);
    res.json(result);
  } catch (error) {
    console.error('Magic wallet info error:', error);
    res.status(500).json({ error: 'Failed to get wallet info' });
  }
});

// Get Magic wallet balance
app.get('/api/magic/balance', authenticateToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const result = await magicLinkBackend.getWalletBalance(userId);
    res.json(result);
  } catch (error) {
    console.error('Magic wallet balance error:', error);
    res.status(500).json({ error: 'Failed to get wallet balance' });
  }
});

// Send Magic transaction
app.post('/api/magic/send-transaction', authenticateToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const { transaction } = req.body;
    
    if (!transaction) {
      return res.status(400).json({ error: 'Transaction data is required' });
    }
    
    const result = await magicLinkBackend.sendTransaction(userId, transaction);
    res.json(result);
  } catch (error) {
    console.error('Magic transaction error:', error);
    res.status(500).json({ error: 'Transaction failed' });
  }
});

// Sign message with Magic wallet
app.post('/api/magic/sign-message', authenticateToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const { message } = req.body;
    
    if (!message) {
      return res.status(400).json({ error: 'Message is required' });
    }
    
    const result = await magicLinkBackend.signMessage(userId, message);
    res.json(result);
  } catch (error) {
    console.error('Magic message signing error:', error);
    res.status(500).json({ error: 'Message signing failed' });
  }
});

// Export Magic wallet keys (admin only)
app.post('/api/magic/export-keys', authenticateToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const result = await magicLinkBackend.exportWalletKeys(userId);
    res.json(result);
  } catch (error) {
    console.error('Magic key export error:', error);
    res.status(500).json({ error: 'Key export failed' });
  }
});

// Logout from Magic Link
app.post('/api/magic/logout', authenticateToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const result = await magicLinkBackend.logoutUser(userId);
    res.json(result);
  } catch (error) {
    console.error('Magic logout error:', error);
    res.status(500).json({ error: 'Logout failed' });
  }
});

// MetaMap KYC Routes
app.post('/api/kyc/start-verification', authenticateToken, MetaMapController.startVerification);
app.post('/api/kyc/upload-input', authenticateToken, metamapUpload.single('file'), MetaMapController.uploadInput);
app.get('/api/kyc/status/:identityId', authenticateToken, MetaMapController.getVerificationStatus);
app.post('/api/kyc/cancel/:identityId', authenticateToken, MetaMapController.cancelVerification);
app.post('/api/kyc/webhook', MetaMapController.handleWebhook);

// Send Money Routes
app.get('/api/transfer/recipients', authenticateToken, async (req, res) => {
  const connection = await pool.getConnection();
  try {
    const userId = req.user.id;
    
    const [recipients] = await connection.execute(`
      SELECT r.*, 
             u.first_name, u.last_name, u.email as user_email, u.phone as user_phone,
             u.account_number as user_account_number
      FROM recipients r 
      LEFT JOIN users u ON r.recipient_user_id = u.id 
      WHERE r.user_id = ? AND r.is_active = 1
      ORDER BY r.created_at DESC
    `, [userId]);
    
    res.json({ success: true, recipients });
  } catch (error) {
    console.error('Error fetching recipients:', error);
    res.status(500).json({ success: false, error: 'Failed to fetch recipients', details: error.message });
  } finally {
    connection.release();
  }
});

// Card to Wallet Withdrawal (using Flutterwave)
app.post('/api/withdraw/card-to-wallet', authenticateToken, async (req, res) => {
  const connection = await pool.getConnection();
  try {
    const { amount, currency = 'NGN', walletId, cardNumber, cvv, expiryMonth, expiryYear, cardHolderName, email, phoneNumber } = req.body;

    if (!amount || !walletId || !cardNumber || !cvv || !expiryMonth || !expiryYear) {
      return res.status(400).json({ success: false, error: 'Missing required fields' });
    }

    if (currency.toUpperCase() !== 'NGN') {
      return res.status(400).json({ success: false, error: 'Card withdrawal is only available for NGN. USD and GBP coming soon.' });
    }

    await connection.beginTransaction();

    // Verify wallet exists and belongs to user
    const [wallets] = await connection.execute(
      'SELECT * FROM wallets WHERE id = ? AND user_id = ? AND currency = ? AND is_active = TRUE',
      [walletId, req.user.id, currency.toUpperCase()]
    );

    if (wallets.length === 0) {
      await connection.rollback();
      return res.status(400).json({ success: false, error: 'Wallet not found or invalid currency' });
    }

    const wallet = wallets[0];
    const txRef = `withdraw_${Date.now()}_${req.user.id}`;

    // Charge card using Flutterwave
    const { chargeCard } = require('./services/flutterwave');
    let chargeResponse;
    try {
      chargeResponse = await chargeCard({
        cardNumber,
        cvv,
        expiryMonth,
        expiryYear,
        amount,
        currency,
        email: email || req.user.email,
        phoneNumber: phoneNumber || req.user.phone,
        fullName: cardHolderName || `${req.user.first_name} ${req.user.last_name}`,
        txRef
      });
    } catch (error) {
      await connection.rollback();
      return res.status(400).json({ 
        success: false, 
        error: error.message || 'Card charge failed',
        details: error.details 
      });
    }

    // Create transaction record
    const referenceId = chargeResponse.flw_ref || txRef;
    const [txResult] = await connection.execute(`
      INSERT INTO transactions (
        user_id, amount, transaction_type, status, description,
        receiver_wallet_id, currency, reference_id
      ) VALUES (?, ?, 'deposit', 'pending', ?, ?, ?, ?)
    `, [req.user.id, amount, `Card to wallet deposit - ${currency}`, walletId, currency.toUpperCase(), referenceId]);

    const transactionId = txResult.insertId;

    // Verify the charge
    try {
      const { verifyCardCharge } = require('./services/flutterwave');
      const verification = await verifyCardCharge(chargeResponse.flw_ref);
      
      if (verification.status === 'successful' && verification.charged_amount >= amount) {
        // Credit wallet
        await connection.execute(
          'UPDATE wallets SET balance = balance + ? WHERE id = ?',
          [amount, walletId]
        );

        await connection.execute(
          'UPDATE transactions SET status = ? WHERE id = ?',
          ['completed', transactionId]
        );

        await connection.commit();
        
        return res.json({
          success: true,
          transactionId,
          message: 'Funds deposited successfully',
          wallet: {
            id: wallet.id,
            balance: parseFloat(wallet.balance) + parseFloat(amount)
          }
        });
      } else {
        // Charge verification failed
        await connection.execute(
          'UPDATE transactions SET status = ?, description = CONCAT(description, ?) WHERE id = ?',
          ['failed', ' - Charge verification failed', transactionId]
        );
        await connection.commit();
        
        return res.status(400).json({
          success: false,
          error: 'Payment verification failed',
          transactionId
        });
      }
    } catch (verifyError) {
      // Keep transaction as pending for manual review
      await connection.commit();
      return res.json({
        success: true,
        transactionId,
        message: 'Transaction initiated. Status will be updated shortly.',
        pending: true
      });
    }
  } catch (error) {
    await connection.rollback();
    console.error('Card withdrawal error:', error);
    res.status(500).json({ success: false, error: error.message || 'Withdrawal failed' });
  } finally {
    connection.release();
  }
});

app.post('/api/transfer/send', authenticateToken, requireKYCApproval, async (req, res) => {
  const connection = await pool.getConnection();
  
  try {
    await connection.beginTransaction();
    
    const { recipientId, amount, note, fromAccount } = req.body;
    const userId = req.user.id;
    
    // Validate recipient
    const [recipients] = await connection.execute(`
      SELECT * FROM recipients WHERE id = ? AND user_id = ?
    `, [recipientId, userId]);
    
    if (recipients.length === 0) {
      throw new Error('Recipient not found');
    }
    
    const recipient = recipients[0];
    
    // Check sender balance - also check if wallet is active (not frozen)
    const [senderWallets] = await connection.execute(`
      SELECT * FROM wallets WHERE user_id = ? AND account_type = ? AND is_active = TRUE
    `, [userId, fromAccount]);
    
    if (senderWallets.length === 0) {
      throw new Error('Wallet not found or frozen. Please unfreeze the wallet to perform transactions.');
    }
    
    if (senderWallets[0].balance < amount) {
      throw new Error('Insufficient funds');
    }
    
    const senderWallet = senderWallets[0];
    
    // Create transaction
    const [transactionResult] = await connection.execute(`
      INSERT INTO transactions (
        user_id, recipient_id, amount, transaction_type, status, 
        description, from_account, created_at
      ) VALUES (?, ?, ?, 'send_money', 'pending', ?, ?, NOW())
    `, [userId, recipientId, amount, note, fromAccount]);
    
    const transactionId = transactionResult.insertId;
    
    // Update sender balance
    await connection.execute(`
      UPDATE wallets SET balance = balance - ? WHERE user_id = ? AND account_type = ?
    `, [amount, userId, fromAccount]);
    
    // Add to recipient balance (if they have a wallet)
    const [recipientWallets] = await connection.execute(`
      SELECT * FROM wallets WHERE user_id = ?
    `, [recipient.recipient_user_id]);
    
    if (recipientWallets.length > 0) {
      await connection.execute(`
        UPDATE wallets SET balance = balance + ? WHERE user_id = ?
      `, [amount, recipient.recipient_user_id]);
    }
    
    // Update transaction status
    await connection.execute(`
      UPDATE transactions SET status = 'completed' WHERE id = ?
    `, [transactionId]);
    
    await connection.commit();
    
    res.json({ 
      success: true, 
      transactionId,
      message: 'Transfer completed successfully' 
    });
    
  } catch (error) {
    await connection.rollback();
    console.error('Error processing transfer:', error);
    res.status(500).json({ success: false, error: error.message });
  } finally {
    connection.release();
  }
});

app.get('/api/transfer/status/:transactionId', authenticateToken, async (req, res) => {
  try {
    const { transactionId } = req.params;
    const userId = req.user.id;
    
    const [transactions] = await pool.execute(`
      SELECT t.*, r.name as recipient_name, r.email as recipient_email
      FROM transactions t
      LEFT JOIN recipients r ON t.recipient_id = r.id
      WHERE t.id = ? AND t.user_id = ?
    `, [transactionId, userId]);
    
    if (transactions.length === 0) {
      return res.status(404).json({ success: false, error: 'Transaction not found' });
    }
    
    res.json({ success: true, transaction: transactions[0] });
  } catch (error) {
    console.error('Error fetching transaction status:', error);
    res.status(500).json({ success: false, error: 'Failed to fetch transaction status' });
  }
});

// Flutterwave bank transfer (payout)
app.post('/api/transfer/flutterwave', authenticateToken, async (req, res) => {
  const connection = await pool.getConnection();
  try {
    const { amount, currency = 'NGN', bankCode, accountNumber, narration, fromAccount, recipientId } = req.body;
    if (!amount || !bankCode || !accountNumber || !fromAccount) {
      return res.status(400).json({ success: false, error: 'amount, bankCode, accountNumber, fromAccount are required' });
    }

    if (currency.toUpperCase() !== 'NGN') {
      return res.status(400).json({ success: false, error: 'Only NGN transfers are supported via Flutterwave' });
    }

    await connection.beginTransaction();

    // Validate balance - ensure it's a NGN wallet
    const walletId = parseInt(fromAccount);
    const [senderWallets] = await connection.execute(`
      SELECT * FROM wallets WHERE user_id = ? AND id = ? AND currency = 'NGN' AND is_active = TRUE
    `, [req.user.id, walletId]);

    if (senderWallets.length === 0) {
      throw new Error('Wallet not found or not a NGN wallet');
    }

    if (parseFloat(senderWallets[0].balance) < parseFloat(amount)) {
      throw new Error('Insufficient funds');
    }

    const senderWallet = senderWallets[0];

    // Create pending transaction
    const referenceId = `flw_${Date.now()}`;
    const recipientBankName = req.body.bankName || bankCode;
    const [txResult] = await connection.execute(`
      INSERT INTO transactions (
        user_id, amount, transaction_type, status, description, 
        sender_wallet_id, currency, reference_id, recipient_bank, recipient_account_number
      ) VALUES (?, ?, 'transfer', 'pending', ?, ?, ?, ?, ?, ?)
    `, [req.user.id, amount, narration || 'Bank transfer via Flutterwave', walletId, currency.toUpperCase(), referenceId, recipientBankName, accountNumber]);

    const transactionId = txResult.insertId;

    // Debit immediately to prevent double spend
    await connection.execute(`
      UPDATE wallets SET balance = balance - ? WHERE id = ? AND user_id = ?
    `, [amount, walletId, req.user.id]);

    // Call Flutterwave
    const { createTransfer } = require('./services/flutterwave');
    let flwResponse;
    try {
      flwResponse = await createTransfer({ amount, currency, bankCode, accountNumber, narration, reference: `txn_${transactionId}` });
    } catch (e) {
      // Refund on failure
      await connection.execute(`
        UPDATE wallets SET balance = balance + ? WHERE id = ? AND user_id = ?
      `, [amount, walletId, req.user.id]);

      await connection.execute(`
        UPDATE transactions SET status = 'failed', description = CONCAT(description, ' - Flutterwave Error: ', ?) WHERE id = ?
      `, [e.message, transactionId]);

      await connection.commit();
      return res.status(400).json({ success: false, error: e.message, details: e.details });
    }

    const providerStatus = flwResponse?.status || 'PENDING';
    const finalStatus = providerStatus === 'SUCCESSFUL' ? 'completed' : 'pending';
    const flwReference = flwResponse?.reference || referenceId;

    await connection.execute(`
      UPDATE transactions SET status = ?, reference_id = ?, description = CONCAT(?, ' - FLW: ', ?) WHERE id = ?
    `, [finalStatus, flwReference, narration || 'Bank transfer via Flutterwave', String(flwResponse?.id || ''), transactionId]);

    await connection.commit();

    res.json({
      success: true,
      transactionId,
      providerTransfer: {
        id: flwResponse?.id,
        status: providerStatus,
        reference: flwResponse?.reference
      },
      message: providerStatus === 'SUCCESSFUL' ? 'Transfer completed successfully' : 'Transfer initiated, pending confirmation'
    });
  } catch (error) {
    await connection.rollback();
    console.error('Flutterwave transfer error:', error);
    res.status(500).json({ success: false, error: error.message });
  } finally {
    connection.release();
  }
});

// List banks via Flutterwave
app.get('/api/transfer/banks', authenticateToken, async (req, res) => {
  try {
    const { country = 'NG' } = req.query;
    const { getBanks } = require('./services/flutterwave');
    const banks = await getBanks(country);
    res.json({ success: true, banks });
  } catch (error) {
    console.error('Banks fetch error:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Resolve account name via Flutterwave
app.post('/api/transfer/resolve-account', authenticateToken, async (req, res) => {
  try {
    const { bankCode, accountNumber } = req.body;
    if (!bankCode || !accountNumber) {
      return res.status(400).json({ success: false, error: 'bankCode and accountNumber are required' });
    }
    const { resolveAccount } = require('./services/flutterwave');
    const info = await resolveAccount({ bankCode, accountNumber });
    res.json({ success: true, account: info });
  } catch (error) {
    console.error('Account resolve error:', error);
    res.status(400).json({ success: false, error: error.message, details: error.details });
  }
});

// Get Flutterwave transfer status
app.get('/api/transfer/flutterwave/:providerId/status', authenticateToken, async (req, res) => {
  try {
    const { providerId } = req.params;
    const { getTransfer } = require('./services/flutterwave');
    const data = await getTransfer(providerId);
    res.json({ success: true, providerTransfer: data });
  } catch (error) {
    console.error('Provider status error:', error);
    res.status(400).json({ success: false, error: error.message, details: error.details });
  }
});

// Validate transfer amount
app.post('/api/transfer/validate', authenticateToken, async (req, res) => {
  try {
    const { amount, fromAccount } = req.body;
    
    // Get wallet balance
    const [wallets] = await pool.execute(
      `SELECT balance, currency FROM wallets WHERE id = ? AND user_id = ? AND is_active = 1`,
      [fromAccount, req.user.id]
    );

    if (wallets.length === 0) {
      return res.status(404).json({ 
        success: false,
        error: 'Wallet not found',
        isValid: false,
        availableBalance: 0
      });
    }

    const wallet = wallets[0];
    const availableBalance = parseFloat(wallet.balance);
    const transferAmount = parseFloat(amount);

    // Validate amount format
    if (isNaN(transferAmount) || !isFinite(transferAmount)) {
      return res.status(400).json({ 
        success: false,
        error: 'Invalid amount format. Please enter a valid number.',
        isValid: false,
        availableBalance
      });
    }

    // Check if sufficient balance
    const isValid = transferAmount > 0 && transferAmount <= availableBalance;
    
    if (!isValid && transferAmount > 0) {
      return res.json({
        success: true,
        isValid: false,
        availableBalance,
        currency: wallet.currency,
        requestedAmount: transferAmount,
        error: `Insufficient balance. Available: ${wallet.currency} ${availableBalance.toFixed(2)}`
      });
    }

    res.json({
      success: true,
      isValid,
      availableBalance,
      currency: wallet.currency,
      requestedAmount: transferAmount
    });
  } catch (error) {
    console.error('Validate transfer error:', error);
    res.status(500).json({ 
      success: false,
      error: 'Validation failed',
      isValid: false,
      availableBalance: 0
    });
  }
});

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({ error: 'Endpoint not found' });
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`GlobanaMobile API server running on port ${PORT}`);
  console.log(`Health check: http://localhost:${PORT}/api/health`);
  console.log(`Server accessible from: http://10.151.176.117:${PORT}`);
});

// Exchange rate service (mock implementation - in production, use a real API like Fixer.io)

// Test endpoint for debugging
app.get('/api/test-charge-endpoint', (req, res) => {
  console.log('🔍 Test endpoint hit');
  res.json({ 
    success: true, 
    message: 'Charge endpoint is accessible',
    timestamp: new Date().toISOString()
  });
});

