const FIXER_BASE_URL = 'https://data.fixer.io/api';

function getApiKey() {
  // Try environment variable first, then fallback to provided key
  const apiKey = process.env.FIXER_API_KEY?.trim() || '3bc0f144dec0d92b91f3a25f65355c5b';
  if (!apiKey) {
    throw new Error('Missing FIXER_API_KEY. Please set FIXER_API_KEY in your .env file or update server/services/fixer.js');
  }
  return apiKey;
}

async function getLatestRates(baseCurrency = 'EUR', symbols = null) {
  const apiKey = getApiKey();
  // Note: Free plan only supports EUR as base. For other bases, we'll convert via EUR
  let url = `${FIXER_BASE_URL}/latest?access_key=${apiKey}`;
  
  // Only set base if it's EUR (free plan limitation)
  // For paid plans, you can use: &base=${baseCurrency}
  if (baseCurrency === 'EUR') {
    url += `&base=EUR`;
  }
  
  if (symbols && Array.isArray(symbols) && symbols.length > 0) {
    url += `&symbols=${symbols.join(',')}`;
  } else if (baseCurrency !== 'EUR') {
    // If base is not EUR, we need to include it in symbols to convert
    url += `&symbols=${baseCurrency},${symbols ? symbols.join(',') : ''}`;
  }

  const res = await fetch(url);
  const data = await res.json();

  if (!res.ok || !data.success) {
    const errMsg = data?.error?.info || 'Failed to fetch exchange rates';
    const err = new Error(errMsg);
    err.code = data?.error?.code;
    err.details = data;
    throw err;
  }

  return data;
}

async function convertCurrency(from, to, amount) {
  const apiKey = getApiKey();
  const url = `${FIXER_BASE_URL}/convert?access_key=${apiKey}&from=${from}&to=${to}&amount=${amount}`;

  const res = await fetch(url);
  const data = await res.json();

  if (!res.ok || !data.success) {
    const errMsg = data?.error?.info || 'Currency conversion failed';
    const err = new Error(errMsg);
    err.code = data?.error?.code;
    err.details = data;
    throw err;
  }

  return data;
}

async function getExchangeRate(fromCurrency, toCurrency) {
  try {
    if (fromCurrency === toCurrency) {
      return { rate: 1, date: new Date().toISOString().split('T')[0] };
    }

    const from = fromCurrency.toUpperCase();
    const to = toCurrency.toUpperCase();

    // Fixer.io uses EUR as base by default, so we need to handle conversion
    // If from is EUR, get rate directly
    // If from is not EUR, we need to convert: 1 FROM = (1/EUR_FROM) * EUR_TO
    let rate;
    let data;
    let date;
    let timestamp;
    
    if (from === 'EUR') {
      data = await getLatestRates('EUR', [to]);
      rate = data.rates[to];
      date = data.date;
      timestamp = data.timestamp;
    } else if (to === 'EUR') {
      data = await getLatestRates('EUR', [from]);
      rate = 1 / data.rates[from];
      date = data.date;
      timestamp = data.timestamp;
    } else {
      // Both are not EUR - convert via EUR
      data = await getLatestRates('EUR', [from, to]);
      const fromRate = data.rates[from];
      const toRate = data.rates[to];
      rate = toRate / fromRate;
      date = data.date;
      timestamp = data.timestamp;
    }

    if (!rate || isNaN(rate)) {
      throw new Error(`Exchange rate not found for ${from} to ${to}`);
    }

    return {
      rate: rate,
      date: date || new Date().toISOString().split('T')[0],
      timestamp: timestamp || Math.floor(Date.now() / 1000),
      base: from
    };
  } catch (error) {
    console.error('Fixer API error:', error);
    throw error;
  }
}

module.exports = {
  getLatestRates,
  convertCurrency,
  getExchangeRate
};

