const FLW_BASE_URL = 'https://api.flutterwave.com/v3';

function getAuthHeader() {
  const secretKey = process.env.FLW_SECRET_KEY?.trim();
  if (!secretKey) {
    throw new Error('Missing FLW_SECRET_KEY environment variable. Please set it in your .env file.');
  }
  if (!secretKey.startsWith('FLWSECK_TEST_') && !secretKey.startsWith('FLWSECK_')) {
    throw new Error('Invalid FLW_SECRET_KEY format. Should start with FLWSECK_TEST_ (test) or FLWSECK_ (live)');
  }
  return { Authorization: `Bearer ${secretKey}` };
}

async function createTransfer({ amount, currency = 'NGN', bankCode, accountNumber, narration, reference }) {
  const url = `${FLW_BASE_URL}/transfers`;
  
  const numericBankCode = String(bankCode).trim();
  if (!/^\d+$/.test(numericBankCode)) {
    throw new Error('Bank code must be numeric');
  }
  
  const body = {
    account_bank: numericBankCode,
    account_number: String(accountNumber).trim(),
    amount: Number(amount),
    currency: currency.toUpperCase(),
    narration: narration || 'Wallet transfer',
    reference: reference || `flw_${Date.now()}`,
    debit_currency: currency.toUpperCase()
  };

  const res = await fetch(url, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      ...getAuthHeader()
    },
    body: JSON.stringify(body)
  });

  const data = await res.json();
  if (!res.ok || data.status !== 'success') {
    const errMsg = data?.message || 'Flutterwave transfer failed';
    const err = new Error(errMsg);
    err.details = data;
    throw err;
  }
  return data.data; // contains id, status, reference, etc.
}

async function getBanks(country = 'NG') {
  const url = `${FLW_BASE_URL}/banks/${country}`;
  const res = await fetch(url, { headers: getAuthHeader() });
  const data = await res.json();
  if (!res.ok || data.status !== 'success') {
    const errorMsg = data?.message || 'Failed to fetch banks';
    const error = new Error(`Flutterwave API Error: ${errorMsg}`);
    error.details = {
      status: res.status,
      statusText: res.statusText,
      flwResponse: data
    };
    throw error;
  }
  return data.data;
}

async function resolveAccount({ accountNumber, bankCode }) {
  const url = `${FLW_BASE_URL}/accounts/resolve`;
  
  const numericBankCode = String(bankCode).trim();
  if (!/^\d+$/.test(numericBankCode)) {
    throw new Error('Bank code must be numeric');
  }
  
  const res = await fetch(url, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      ...getAuthHeader()
    },
    body: JSON.stringify({ 
      account_number: String(accountNumber).trim(), 
      account_bank: numericBankCode 
    })
  });
  const data = await res.json();
  if (!res.ok || data.status !== 'success') {
    const err = new Error(data?.message || 'Failed to resolve account');
    err.details = data;
    throw err;
  }
  return data.data; // { account_number, account_name } etc.
}

async function getTransfer(transferId) {
  const url = `${FLW_BASE_URL}/transfers/${transferId}`;
  const res = await fetch(url, { headers: getAuthHeader() });
  const data = await res.json();
  if (!res.ok || data.status !== 'success') {
    const err = new Error(data?.message || 'Failed to fetch transfer');
    err.details = data;
    throw err;
  }
  return data.data; // includes status
}

async function chargeCard({ cardNumber, cvv, expiryMonth, expiryYear, amount, currency = 'NGN', email, phoneNumber, fullName, txRef }) {
  const url = `${FLW_BASE_URL}/charges?type=card`;
  
  const body = {
    card_number: String(cardNumber).replace(/\s/g, ''),
    cvv: String(cvv),
    expiry_month: String(expiryMonth).padStart(2, '0'),
    expiry_year: String(expiryYear),
    currency: currency.toUpperCase(),
    amount: Number(amount),
    email: email,
    phone_number: phoneNumber,
    fullname: fullName,
    tx_ref: txRef || `card_charge_${Date.now()}`,
    enckey: process.env.FLW_ENCRYPTION_KEY || '',
    authorization: {
      mode: 'pin',
      pin: '' // Will be collected via Flutterwave inline
    }
  };

  const res = await fetch(url, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      ...getAuthHeader()
    },
    body: JSON.stringify(body)
  });

  const data = await res.json();
  if (!res.ok || data.status !== 'success') {
    const errMsg = data?.message || 'Card charge failed';
    const err = new Error(errMsg);
    err.details = data;
    throw err;
  }
  return data.data; // contains flw_ref, status, etc.
}

async function verifyCardCharge(transactionId) {
  // Flutterwave verification endpoint accepts transaction ID
  // Can also verify by tx_ref using: /transactions/verify_by_reference?tx_ref=xxx
  let url;
  let isNumericId = false;
  
  // Check if it's a numeric transaction ID or a reference
  const cleanId = String(transactionId).trim();
  if (/^\d+$/.test(cleanId)) {
    // Numeric transaction ID
    isNumericId = true;
    url = `${FLW_BASE_URL}/transactions/${cleanId}/verify`;
  } else {
    // Reference (tx_ref) - use verify_by_reference endpoint
    url = `${FLW_BASE_URL}/transactions/verify_by_reference?tx_ref=${encodeURIComponent(cleanId)}`;
  }
  
  console.log('🔍 Flutterwave verification:', { 
    transactionId: cleanId, 
    isNumericId, 
    url: url.replace(/Bearer\s+\w+/g, 'Bearer ***') // Mask any auth tokens in URL if present
  });
  
  const res = await fetch(url, { headers: getAuthHeader() });
  const data = await res.json();
  
  if (!res.ok || data.status !== 'success') {
    // If verification by ID failed and it was numeric, try by reference as fallback
    if (isNumericId && data.message?.includes('No transaction')) {
      console.log('⚠️ Verification by ID failed, this might be a reference instead');
    }
    
    const err = new Error(data?.message || 'Failed to verify charge');
    err.details = data;
    throw err;
  }
  
  if (!data.data) {
    throw new Error('Invalid verification response: missing transaction data');
  }
  
  return data.data; // includes transaction details, status, amount, etc.
}

// Create Flutterwave subaccount (like Stripe Connect)
async function createSubaccount({ accountBank, accountNumber, businessName, businessEmail, splitType = 'percentage', splitValue = 90 }) {
  const url = `${FLW_BASE_URL}/subaccounts`;
  
  const numericBankCode = String(accountBank).trim();
  if (!/^\d+$/.test(numericBankCode)) {
    throw new Error('Bank code must be numeric');
  }
  
  const body = {
    account_bank: numericBankCode,
    account_number: String(accountNumber).trim(),
    business_name: businessName,
    business_email: businessEmail,
    split_type: splitType, // 'percentage' or 'flat'
    split_value: splitValue // percentage or flat amount
  };

  const res = await fetch(url, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      ...getAuthHeader()
    },
    body: JSON.stringify(body)
  });

  const data = await res.json();
  if (!res.ok || data.status !== 'success') {
    const errMsg = data?.message || 'Failed to create subaccount';
    const err = new Error(errMsg);
    err.details = data;
    throw err;
  }
  return data.data; // { id, account_number, account_bank, split_value, split_type }
}

// Get subaccount details
async function getSubaccount(subaccountId) {
  const url = `${FLW_BASE_URL}/subaccounts/${subaccountId}`;
  const res = await fetch(url, { headers: getAuthHeader() });
  const data = await res.json();
  if (!res.ok || data.status !== 'success') {
    const err = new Error(data?.message || 'Failed to fetch subaccount');
    err.details = data;
    throw err;
  }
  return data.data;
}

// Update subaccount
async function updateSubaccount(subaccountId, { splitType, splitValue }) {
  const url = `${FLW_BASE_URL}/subaccounts/${subaccountId}`;
  
  const body = {};
  if (splitType) body.split_type = splitType;
  if (splitValue !== undefined) body.split_value = splitValue;

  const res = await fetch(url, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      ...getAuthHeader()
    },
    body: JSON.stringify(body)
  });

  const data = await res.json();
  if (!res.ok || data.status !== 'success') {
    const err = new Error(data?.message || 'Failed to update subaccount');
    err.details = data;
    throw err;
  }
  return data.data;
}

// Delete subaccount
async function deleteSubaccount(subaccountId) {
  const url = `${FLW_BASE_URL}/subaccounts/${subaccountId}`;
  const res = await fetch(url, {
    method: 'DELETE',
    headers: getAuthHeader()
  });
  const data = await res.json();
  if (!res.ok || data.status !== 'success') {
    const err = new Error(data?.message || 'Failed to delete subaccount');
    err.details = data;
    throw err;
  }
  return data.data;
}

module.exports = {
  createTransfer,
  getBanks,
  resolveAccount,
  getTransfer,
  chargeCard,
  verifyCardCharge,
  createSubaccount,
  getSubaccount,
  updateSubaccount,
  deleteSubaccount
};


