const axios = require('axios');

class MetaMapService {
  constructor() {
    this.baseURL = 'https://api.getmati.com';
    this.clientId = process.env.METAMAP_CLIENT_ID;
    this.clientSecret = process.env.METAMAP_CLIENT_SECRET;
    this.jwtToken = null;
    this.tokenExpiry = null;
    this.activeUploads = new Map();
  }

  // Get headers for API requests
  getHeaders() {
    return {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${this.jwtToken}`,
    };
  }

  // Authenticate with MetaMap API
  async authenticate() {
    try {
      const response = await axios.post(`${this.baseURL}/oauth`, {
        client_id: this.clientId,
        client_secret: this.clientSecret,
      });

      this.jwtToken = response.data.access_token;
      this.tokenExpiry = Date.now() + (3600 * 1000); // 1 hour from now
      
      return {
        success: true,
        data: response.data,
        error: null
      };
    } catch (error) {
      return {
        success: false,
        data: null,
        error: error.response?.data?.error || error.message
      };
    }
  }

  // Check if token is valid and refresh if needed
  async ensureValidToken() {
    if (!this.jwtToken || Date.now() >= this.tokenExpiry) {
      const authResult = await this.authenticate();
      if (!authResult.success) {
        throw new Error(authResult.error);
      }
    }
  }

  // Create verification instance
  async createVerification(metamapId, metadata = {}) {
    try {
      await this.ensureValidToken();

      const response = await axios.post(`${this.baseURL}/v2/verifications`, {
        metamap: metamapId,
        metadata: metadata,
      }, {
        headers: this.getHeaders()
      });

      return {
        success: true,
        data: response.data,
        error: null
      };
    } catch (error) {
      return {
        success: false,
        data: null,
        error: error.response?.data?.error || error.message
      };
    }
  }

  // Send user input (documents, images, videos)
  async sendUserInput(identityId, inputData) {
    try {
      await this.ensureValidToken();

      const response = await axios.post(
        `${this.baseURL}/v2/identities/${identityId}/send-input`,
        inputData,
        {
          headers: this.getHeaders()
        }
      );

      return {
        success: true,
        data: response.data,
        error: null
      };
    } catch (error) {
      return {
        success: false,
        data: null,
        error: error.response?.data?.error || error.message
      };
    }
  }

  // Get verification status
  async getVerificationStatus(identityId) {
    try {
      await this.ensureValidToken();

      const response = await axios.get(`${this.baseURL}/v2/identities/${identityId}`, {
        headers: this.getHeaders()
      });

      return {
        success: true,
        data: response.data,
        error: null
      };
    } catch (error) {
      return {
        success: false,
        data: null,
        error: error.response?.data?.error || error.message
      };
    }
  }

  // Cancel active upload
  cancelUpload(uploadId) {
    if (this.activeUploads.has(uploadId)) {
      const upload = this.activeUploads.get(uploadId);
      upload.status = 'cancelled';
      this.activeUploads.set(uploadId, upload);
      
      // Clean up after 5 seconds
      setTimeout(() => {
        this.activeUploads.delete(uploadId);
      }, 5000);
      
      return {
        success: true,
        message: 'Upload cancelled successfully'
      };
    }
    
    return {
      success: false,
      message: 'Upload not found or already completed'
    };
  }

  // Enhanced error handling with specific error messages
  getSpecificErrorMessage(error, context = 'verification') {
    const errorMessages = {
      // Authentication errors
      'invalid_client': 'Invalid client credentials. Please check your MetaMap configuration.',
      'unauthorized': 'Authentication failed. Please try again.',
      'token_expired': 'Session expired. Please restart the verification process.',
      
      // Document verification errors
      'document_blurry': 'Document image is too blurry. Please take a clearer photo.',
      'document_dark': 'Document image is too dark. Please ensure good lighting.',
      'document_cut_off': 'Document is cut off. Please capture the entire document.',
      'document_invalid': 'Invalid document type. Please use a government-issued ID.',
      'document_expired': 'Document has expired. Please use a valid, unexpired document.',
      'document_damaged': 'Document appears damaged. Please use an undamaged document.',
      
      // Biometric verification errors
      'face_not_detected': 'No face detected in the image. Please ensure your face is visible.',
      'face_multiple': 'Multiple faces detected. Please ensure only your face is visible.',
      'face_quality': 'Face image quality is too low. Please take a clearer selfie.',
      'face_match_failed': 'Face does not match the document photo. Please try again.',
      
      // Network and server errors
      'network_error': 'Network connection failed. Please check your internet connection.',
      'server_error': 'Server error occurred. Please try again later.',
      'timeout': 'Request timed out. Please try again.',
      
      // General errors
      'invalid_input': 'Invalid input provided. Please check your information.',
      'quota_exceeded': 'Verification quota exceeded. Please try again later.',
      'maintenance': 'Service is under maintenance. Please try again later.'
    };

    // Try to match specific error codes
    for (const [code, message] of Object.entries(errorMessages)) {
      if (error.toLowerCase().includes(code) || error.includes(code)) {
        return message;
      }
    }

    // Return context-specific generic messages
    switch (context) {
      case 'document':
        return 'Document verification failed. Please ensure your document is clear, valid, and government-issued.';
      case 'biometric':
        return 'Biometric verification failed. Please ensure your face is clearly visible and matches your document.';
      case 'address':
        return 'Address verification failed. Please ensure your document is a valid proof of address.';
      default:
        return `Verification failed: ${error}. Please try again.`;
    }
  }

  // Handle webhook events with enhanced error handling
  handleWebhookEvent(webhookData) {
    try {
      const { event, data } = webhookData;
      
      switch (event) {
        case 'verification.completed':
          return {
            success: true,
            data: {
              status: 'completed',
              result: data.result,
              identityId: data.identityId,
              score: data.score || 100
            }
          };
        case 'verification.failed':
          const specificError = this.getSpecificErrorMessage(data.reason || data.error, data.type);
          return {
            success: false,
            data: {
              status: 'failed',
              reason: data.reason,
              specificError: specificError,
              identityId: data.identityId,
              errorCode: data.errorCode
            }
          };
        case 'verification.processing':
          return {
            success: true,
            data: {
              status: 'processing',
              identityId: data.identityId,
              progress: data.progress || 0
            }
          };
        case 'verification.pending':
          return {
            success: true,
            data: {
              status: 'pending',
              identityId: data.identityId,
              message: 'Verification is pending review'
            }
          };
        default:
          return {
            success: false,
            data: null,
            error: 'Unknown webhook event'
          };
      }
    } catch (error) {
      return {
        success: false,
        data: null,
        error: error.message
      };
    }
  }
}

module.exports = new MetaMapService();
