const axios = require('axios');

class VTPassService {
  constructor() {
    this.baseURL = process.env.VTPASS_BASE_URL || 'https://sandbox.vtpass.com/api';
    this.username = process.env.VTPASS_USERNAME || 'sandbox@vtpass.com';
    this.password = process.env.VTPASS_PASSWORD || 'sandbox';
    
    // Check if credentials are available
    this.hasCredentials = !!(this.username && this.password && 
      this.username !== 'sandbox@vtpass.com' && this.password !== 'sandbox');
    
    if (this.hasCredentials) {
      this.client = axios.create({
        baseURL: this.baseURL,
        timeout: 30000,
        auth: {
          username: this.username,
          password: this.password
        }
      });
    } else {
      console.log('⚠️ VTPass: No credentials provided, using mock data for testing');
      this.client = null;
    }

    // Add request interceptor for logging
    if (this.client) {
      this.client.interceptors.request.use(
        (config) => {
          console.log('🌐 VTPass Request:', {
            method: config.method?.toUpperCase(),
            url: config.url,
            data: config.data
          });
          return config;
        },
        (error) => {
          console.error('❌ VTPass Request Error:', error);
          return Promise.reject(error);
        }
      );

      // Add response interceptor for logging
      this.client.interceptors.response.use(
        (response) => {
          console.log('✅ VTPass Response:', {
            status: response.status,
            url: response.config.url,
            data: response.data
          });
          return response;
        },
        (error) => {
          console.error('❌ VTPass Response Error:', {
            status: error.response?.status,
            url: error.config?.url,
            message: error.message,
            data: error.response?.data
          });
          return Promise.reject(error);
        }
      );
    }
  }

  // Get all available services from VTPass
  async getServices() {
    try {
      console.log('🚀 VTPass: Getting all services...');
      
      if (!this.hasCredentials) {
        // Return mock data for testing
        return {
          success: true,
          data: this.getMockServices()
        };
      }
      
      const response = await this.client.get('/services');
      
      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      console.error('💥 VTPass getServices error:', error.response?.data || error.message);
      return {
        success: false,
        error: error.response?.data?.message || 'Failed to fetch services'
      };
    }
  }

  // Get service variations (different plans/packages)
  async getServiceVariations(serviceId) {
    try {
      console.log('🚀 VTPass: Getting service variations for:', serviceId);
      
      if (!this.hasCredentials) {
        // Return mock variations for testing
        return {
          success: true,
          data: this.getMockServiceVariations(serviceId)
        };
      }
      
      const response = await this.client.get(`/service-variations?serviceID=${serviceId}`);
      
      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      console.error('💥 VTPass getServiceVariations error:', error.response?.data || error.message);
      return {
        success: false,
        error: error.response?.data?.message || 'Failed to fetch service variations'
      };
    }
  }

  // Verify customer number
  async verifyCustomer(serviceId, customerNumber) {
    try {
      console.log('🚀 VTPass: Verifying customer:', { serviceId, customerNumber });
      
      if (!this.hasCredentials) {
        // Return mock verification for testing
        return {
          success: true,
          data: this.getMockCustomerVerification(serviceId, customerNumber)
        };
      }
      
      const response = await this.client.post('/merchant-verify', {
        serviceID: serviceId,
        billersCode: customerNumber
      });

      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      console.error('💥 VTPass verify error:', error.response?.data || error.message);
      return {
        success: false,
        error: error.response?.data?.message || 'Verification failed'
      };
    }
  }

  // Get bill amount
  async getBillAmount(serviceId, customerNumber) {
    try {
      console.log('🚀 VTPass: Getting bill amount:', { serviceId, customerNumber });
      
      if (!this.hasCredentials) {
        // Return mock bill amount for testing
        return {
          success: true,
          data: this.getMockBillAmount(serviceId, customerNumber)
        };
      }
      
      const response = await this.client.post('/query', {
        serviceID: serviceId,
        billersCode: customerNumber
      });

      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      console.error('💥 VTPass query error:', error.response?.data || error.message);
      return {
        success: false,
        error: error.response?.data?.message || 'Failed to get bill amount'
      };
    }
  }

  // Process bill payment
  async processPayment(paymentData) {
    try {
      const {
        serviceId,
        customerNumber,
        amount,
        phone,
        reference
      } = paymentData;

      console.log('🚀 VTPass: Processing payment:', paymentData);
      
      if (!this.hasCredentials) {
        // Return mock payment result for testing
        return {
          success: true,
          data: this.getMockPaymentResult(paymentData)
        };
      }
      
      const response = await this.client.post('/pay', {
        serviceID: serviceId,
        billersCode: customerNumber,
        amount: amount,
        phone: phone,
        request_id: reference
      });

      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      console.error('💥 VTPass payment error:', error.response?.data || error.message);
      return {
        success: false,
        error: error.response?.data?.message || 'Payment processing failed'
      };
    }
  }

  // Get transaction status
  async getTransactionStatus(reference) {
    try {
      console.log('🚀 VTPass: Getting transaction status for:', reference);
      const response = await this.client.post('/requery', {
        request_id: reference
      });

      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      console.error('💥 VTPass requery error:', error.response?.data || error.message);
      return {
        success: false,
        error: error.response?.data?.message || 'Failed to get transaction status'
      };
    }
  }

  // Map VTPass services to our categories
  mapServicesToCategories(services) {
    const categories = {
      'Electricity': [],
      'Airtime & Data': [],
      'Cable TV': [],
      'Internet': [],
      'Water': []
    };

    if (!services || !Array.isArray(services)) {
      return categories;
    }

    services.forEach(service => {
      const serviceName = service.name?.toLowerCase() || '';
      
      if (serviceName.includes('electric') || serviceName.includes('power')) {
        categories['Electricity'].push(service);
      } else if (serviceName.includes('airtime') || serviceName.includes('data') || 
                 serviceName.includes('mtn') || serviceName.includes('airtel') || 
                 serviceName.includes('glo') || serviceName.includes('9mobile')) {
        categories['Airtime & Data'].push(service);
      } else if (serviceName.includes('dstv') || serviceName.includes('gotv') || 
                 serviceName.includes('startimes') || serviceName.includes('cable')) {
        categories['Cable TV'].push(service);
      } else if (serviceName.includes('internet') || serviceName.includes('broadband')) {
        categories['Internet'].push(service);
      } else if (serviceName.includes('water')) {
        categories['Water'].push(service);
      }
    });

    return categories;
  }

  // Mock services for testing when credentials are not available
  getMockServices() {
    return [
      {
        id: 'ikeja-electric',
        name: 'Ikeja Electric',
        category: 'Electricity',
        description: 'Ikeja Electricity Distribution Company',
        logo: 'https://example.com/ikeja-electric.png'
      },
      {
        id: 'eko-electric',
        name: 'Eko Electricity',
        category: 'Electricity',
        description: 'Eko Electricity Distribution Company',
        logo: 'https://example.com/eko-electric.png'
      },
      {
        id: 'mtn-airtime',
        name: 'MTN Airtime',
        category: 'Airtime & Data',
        description: 'MTN Nigeria Airtime Recharge',
        logo: 'https://example.com/mtn.png'
      },
      {
        id: 'airtel-airtime',
        name: 'Airtel Airtime',
        category: 'Airtime & Data',
        description: 'Airtel Nigeria Airtime Recharge',
        logo: 'https://example.com/airtel.png'
      },
      {
        id: 'glo-airtime',
        name: 'Glo Airtime',
        category: 'Airtime & Data',
        description: 'Glo Nigeria Airtime Recharge',
        logo: 'https://example.com/glo.png'
      },
      {
        id: 'dstv',
        name: 'DSTV',
        category: 'Cable TV',
        description: 'DSTV Cable TV Subscription',
        logo: 'https://example.com/dstv.png'
      },
      {
        id: 'gotv',
        name: 'GOTV',
        category: 'Cable TV',
        description: 'GOTV Cable TV Subscription',
        logo: 'https://example.com/gotv.png'
      },
      {
        id: 'startimes',
        name: 'StarTimes',
        category: 'Cable TV',
        description: 'StarTimes Cable TV Subscription',
        logo: 'https://example.com/startimes.png'
      }
    ];
  }

  // Mock service variations for testing
  getMockServiceVariations(serviceId) {
    const variations = {
      'ikeja-electric': [
        { id: 'prepaid', name: 'Prepaid Meter', amount: 1000 },
        { id: 'postpaid', name: 'Postpaid Meter', amount: 2000 }
      ],
      'eko-electric': [
        { id: 'prepaid', name: 'Prepaid Meter', amount: 1200 },
        { id: 'postpaid', name: 'Postpaid Meter', amount: 2500 }
      ],
      'mtn-airtime': [
        { id: '100', name: '₦100 Airtime', amount: 100 },
        { id: '200', name: '₦200 Airtime', amount: 200 },
        { id: '500', name: '₦500 Airtime', amount: 500 },
        { id: '1000', name: '₦1000 Airtime', amount: 1000 }
      ],
      'airtel-airtime': [
        { id: '100', name: '₦100 Airtime', amount: 100 },
        { id: '200', name: '₦200 Airtime', amount: 200 },
        { id: '500', name: '₦500 Airtime', amount: 500 },
        { id: '1000', name: '₦1000 Airtime', amount: 1000 }
      ],
      'glo-airtime': [
        { id: '100', name: '₦100 Airtime', amount: 100 },
        { id: '200', name: '₦200 Airtime', amount: 200 },
        { id: '500', name: '₦500 Airtime', amount: 500 },
        { id: '1000', name: '₦1000 Airtime', amount: 1000 }
      ],
      'dstv': [
        { id: 'premium', name: 'Premium Package', amount: 24500 },
        { id: 'compact-plus', name: 'Compact Plus', amount: 16500 },
        { id: 'compact', name: 'Compact Package', amount: 10900 }
      ],
      'gotv': [
        { id: 'max', name: 'GOtv Max', amount: 4650 },
        { id: 'jolli', name: 'GOtv Jolli', amount: 2650 },
        { id: 'jinja', name: 'GOtv Jinja', amount: 1650 }
      ],
      'startimes': [
        { id: 'nova', name: 'Nova Package', amount: 1200 },
        { id: 'basic', name: 'Basic Package', amount: 800 },
        { id: 'smart', name: 'Smart Package', amount: 1600 }
      ]
    };

    return variations[serviceId] || [];
  }

  // Mock customer verification for testing
  getMockCustomerVerification(serviceId, customerNumber) {
    const mockData = {
      customerName: 'John Doe',
      customerNumber: customerNumber,
      serviceId: serviceId,
      isValid: true, // Changed from status: 'verified' to isValid: true
      status: 'verified',
      message: 'Customer verified successfully'
    };

    // Add service-specific data
    if (serviceId.includes('electric')) {
      mockData.customerName = 'Adebayo Johnson';
      mockData.meterNumber = customerNumber;
      mockData.meterType = 'Prepaid';
      mockData.address = '123 Victoria Island, Lagos';
      mockData.phone = '08012345678';
    } else if (serviceId.includes('airtime')) {
      mockData.customerName = 'Sarah Williams';
      mockData.phoneNumber = customerNumber;
      mockData.network = serviceId.split('-')[0].toUpperCase();
      mockData.address = '456 Lekki Phase 1, Lagos';
      mockData.phone = customerNumber;
    } else if (serviceId.includes('dstv') || serviceId.includes('gotv') || serviceId.includes('startimes')) {
      mockData.customerName = 'Michael Okechukwu';
      mockData.smartCardNumber = customerNumber;
      mockData.customerType = 'Active';
      mockData.address = '789 Ikeja GRA, Lagos';
      mockData.phone = '09087654321';
    }

    return mockData;
  }

  // Mock bill amount for testing
  getMockBillAmount(serviceId, customerNumber) {
    const mockData = {
      customerNumber: customerNumber,
      serviceId: serviceId,
      amount: 0,
      currency: 'NGN',
      dueDate: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0], // 30 days from now
      status: 'pending'
    };

    // Add service-specific amounts
    if (serviceId.includes('electric')) {
      mockData.amount = Math.floor(Math.random() * 5000) + 1000; // ₦1000 - ₦6000
      mockData.meterNumber = customerNumber;
      mockData.meterType = 'Prepaid';
    } else if (serviceId.includes('airtime')) {
      mockData.amount = 0; // Airtime doesn't have bills, it's prepaid
      mockData.phoneNumber = customerNumber;
    } else if (serviceId.includes('dstv') || serviceId.includes('gotv') || serviceId.includes('startimes')) {
      mockData.amount = Math.floor(Math.random() * 5000) + 2000; // ₦2000 - ₦7000
      mockData.smartCardNumber = customerNumber;
      mockData.packageName = serviceId.includes('dstv') ? 'Premium' : 'Max';
    }

    return mockData;
  }

  // Mock payment result for testing
  getMockPaymentResult(paymentData) {
    const {
      serviceId,
      customerNumber,
      amount,
      phone,
      reference
    } = paymentData;

    return {
      requestId: reference,
      serviceId: serviceId,
      customerNumber: customerNumber,
      amount: amount,
      phone: phone,
      status: 'successful',
      message: 'Payment processed successfully',
      transactionId: `TXN_${Date.now()}`,
      transaction_id: `TXN_${Date.now()}`, // Add both formats for compatibility
      timestamp: new Date().toISOString(),
      code: '000',
      response_description: 'Transaction successful'
    };
  }
}

module.exports = VTPassService; 